﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.IO;
using System.Linq;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Serialization;
using EffectMaker.UIControls.EffectBrowser.Config;
using EffectMaker.UIControls.EffectBrowser.Utilities;

using Constants = EffectMaker.UIControls.EffectBrowser.Data.Constants;

namespace EffectMaker.UIControls.EffectBrowser
{
    /// <summary>
    /// The effect browser panel.
    /// </summary>
    public partial class EffectBrowserPanel
    {
        /// <summary>
        /// The setting.
        /// </summary>
        private Setting setting = new Setting();

        /// <summary>
        /// configを読み込んだかどうかのフラグ
        /// </summary>
        private bool configLoaded = false;

        /// <summary>
        /// The save config.
        /// </summary>
        public void SaveConfig()
        {
            // configを読み込んでいなかったら、何もせずに終了する
            if (this.configLoaded == false)
            {
                return;
            }

            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                this.SaveSetting();
                this.favoritesManager.SaveConfig();
            }
        }

        /// <summary>
        /// The load config.
        /// </summary>
        private void LoadConfig()
        {
            // コントロール更新ブロッカー内から呼ばれているのでそれに乗っかって外します
            // 利用箇所を拡大する際は注意してください
            this.favoritesManager.LoadConfig();
            this.LoadSetting();

            // configを読み込み終えた印をつける
            this.configLoaded = true;
        }

        /// <summary>
        /// The load setting.
        /// </summary>
        private void LoadSetting()
        {
            try
            {
                if (File.Exists(Constants.SettingFilepath))
                {
                    this.setting = SerializationHelper.LoadXmlDocSerializable<Setting>(
                        Constants.SettingFilepath);
                }
            }

            // ReSharper disable once EmptyGeneralCatchClause
            catch
            {
            }

            System.Diagnostics.Debug.Assert(this.setting != null, "this.setting != null");

            this.pnlTreeView.Width = this.setting.FileTreeView.Width;
            this.ftvComputer.DesktopExpanded = this.setting.FileTreeView.DesktopExpanded;
            this.ftvComputer.DocumentsExpanded = this.setting.FileTreeView.DocumentsExpanded;
            this.ftvComputer.ComputerExpanded = this.setting.FileTreeView.ComputerExpanded;

            this.texturesToolStripMenuItem.Checked = this.setting.FileListView.TexturesViewMode;
            this.primitivesToolStripMenuItem.Checked = this.setting.FileListView.PrimitivesViewMode;
            this.combinersToolStripMenuItem.Checked = this.setting.FileListView.CombinersViewMode;
            this.otherToolStripMenuItem.Checked = this.setting.FileListView.OtherViewMode;

            this.setting.FileListView.Columns.ForEach(x =>
            {
                var index = 0;
                this.FileListViews.ForEach(y =>
                {
                    var dstColumn = y.ColumnInfos[x.ColumnKinds];

                    dstColumn.Index = index++;
                    dstColumn.DisplayIndex = x.DisplayIndex;
                    dstColumn.ColumnHeader.Width = x.Width;
                    dstColumn.Visible = x.Visible;
                    dstColumn.SortOrder = x.SortOrder;
                });
            });

            this.directoryPathHistory = this.setting.History.Paths;
        }

        /// <summary>
        /// The save setting.
        /// </summary>
        private void SaveSetting()
        {
            this.setting.FileTreeView.Width = this.pnlTreeView.Width;
            this.setting.FileTreeView.DesktopExpanded = this.ftvComputer.DesktopExpanded;
            this.setting.FileTreeView.DocumentsExpanded = this.ftvComputer.DocumentsExpanded;
            this.setting.FileTreeView.ComputerExpanded = this.ftvComputer.ComputerExpanded;
            this.setting.FileTreeView.ComputerDirectory = this.ftvComputer.Directory;
            this.setting.FileTreeView.FavoritesDirectory = this.ftvFavorites.Directory;
            this.setting.FileTreeView.LastTab = this.tbcTrees.SelectedIndex;

            this.FileListViews.ForEach(x =>
            {
                this.setting.FileListView.Columns = x.ColumnInfos.Select(y =>
                    new FileListViewColumn
                    {
                        Name = y.Key.ToString(),
                        DisplayIndex = y.Value.DisplayIndex,
                        Width = y.Value.ColumnHeader.Width,
                        Visible = y.Value.Visible,
                        SortOrder = y.Value.SortOrder
                    }).ToList();
            });

            this.setting.FileListView.TexturesViewMode = this.texturesToolStripMenuItem.Checked;
            this.setting.FileListView.PrimitivesViewMode = this.primitivesToolStripMenuItem.Checked;
            this.setting.FileListView.CombinersViewMode = this.combinersToolStripMenuItem.Checked;
            this.setting.FileListView.OtherViewMode = this.otherToolStripMenuItem.Checked;

            this.setting.History.Paths = this.directoryPathHistory;
            try
            {
                SerializationHelper.SaveXmlDocSerializable(this.setting, Constants.SettingFilepath);
            }
            catch
            {
                // Failed accessing the file, just skip saving the options.
                // (another EffectMaker instance is accessing it)
            }
        }
    }
}
