﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView
{
    using System;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb favorites file tree view.
    /// </summary>
    public class EBFavoritesFileTreeView : EBFileTreeView
    {
        #region Fields

        /// <summary>
        /// The double clicked.
        /// </summary>
        private bool doubleClicked;

        /// <summary>
        /// The drop pos.
        /// </summary>
        private DropPosKind dropPos = DropPosKind.None;

        /// <summary>
        /// The drop source.
        /// </summary>
        private EBFileTreeNode dropSource;

        /// <summary>
        /// The drop target.
        /// </summary>
        private EBFileTreeNode dropTarget;

        #endregion

        #region Enums

        /// <summary>
        /// The drop pos kind.
        /// </summary>
        private enum DropPosKind
        {
            /// <summary>
            /// The before.
            /// </summary>
            Before,

            /// <summary>
            /// The child.
            /// </summary>
            Child,

            /// <summary>
            /// The after.
            /// </summary>
            After,

            /// <summary>
            /// The none.
            /// </summary>
            None
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The expand current directory node.
        /// </summary>
        public override void ExpandCurrentDirectoryNode()
        {
            if (PathUtility.DirectoryExists(this.Directory) == false)
            {
                return;
            }

            var targetNode = this.FindDirectoryNode();
            if (targetNode == null)
            {
                return;
            }

            using (new UpdateBlock(this))
            {
                var targetPath = targetNode.Filepath;
                if (!this.Directory.StartsWith(targetPath + @"\"))
                {
                    return;
                }
                targetNode.Expand();

                var dirs = this.Directory.Substring(targetNode.Filepath.Length)
                    .Split(new[] { @"\" }, StringSplitOptions.RemoveEmptyEntries);

                targetNode = this.ExpandCurrentDirectoryNode(targetPath, targetNode, dirs);
            }

            this.SelectedNode = targetNode;
            this.SelectedNode.EnsureVisible();
        }

        /// <summary>
        /// The initialize.
        /// </summary>
        public override void Initialize()
        {
            this.AllowDrop = true;
        }

        /// <summary>
        /// The rename.
        /// </summary>
        /// <param name="node">
        /// The node.
        /// </param>
        public void Rename(EBFileTreeNode node)
        {
            System.Diagnostics.Debug.Assert(node != null, "node != null");

            this.LabelEdit = true;
            node.BeginEdit();
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on after label edit.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnAfterLabelEdit(NodeLabelEditEventArgs e)
        {
            base.OnAfterLabelEdit(e);

            this.LabelEdit = false;
        }

        // http://groups.yahoo.co.jp/group/dotnet-csharp/messages/713?viscount=-15&expand=1

        /// <summary>
        /// The on before collapse.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnBeforeCollapse(TreeViewCancelEventArgs e)
        {
            if (this.doubleClicked)
            {
                e.Cancel = true;
                this.doubleClicked = false;
            }

            base.OnBeforeCollapse(e);
        }

        /// <summary>
        /// The on before expand.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnBeforeExpand(TreeViewCancelEventArgs e)
        {
            if (this.doubleClicked)
            {
                e.Cancel = true;
                this.doubleClicked = false;
            }

            base.OnBeforeExpand(e);
        }

        /// <summary>
        /// The on before label edit.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnBeforeLabelEdit(NodeLabelEditEventArgs e)
        {
            base.OnBeforeLabelEdit(e);

            var node = e.Node as EBFileTreeNode;

            if (node != null && node.CanRename == false)
            {
                e.CancelEdit = true;
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.DragDrop"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">
        /// イベント データを格納している <see cref="T:System.Windows.Forms.DragEventArgs"/>。
        /// </param>
        protected override void OnDragDrop(DragEventArgs e)
        {
            base.OnDragDrop(e);

            // ドロップされたデータがTreeNodeか調べる
            if (e.Data.GetDataPresent(typeof(EBFileTreeNode)))
            {
                var source = e.Data.GetData(typeof(EBFileTreeNode)) as EBFileTreeNode;
                var target = this.GetNodeAt(this.PointToClient(new Point(e.X, e.Y))) as EBFileTreeNode;

                // マウス下のNodeがドロップ先として適切か調べる
                if ((target != null) && (target != source) && (IsChildNode(source, target) == false))
                {
                    this.dropTarget = target;
                    this.dropSource = source;
                }
                else
                {
                    e.Effect = DragDropEffects.None;
                }
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.DragOver"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">
        /// イベント データを格納している <see cref="T:System.Windows.Forms.DragEventArgs"/>。
        /// </param>
        protected override void OnDragOver(DragEventArgs e)
        {
            base.OnDragOver(e);

            // ドラッグされているデータがTreeNodeか調べる
            if (e.Data.GetDataPresent(typeof(EBFileTreeNode)))
            {
                if (((e.KeyState & 8) == 8) && ((e.AllowedEffect & DragDropEffects.Copy) == DragDropEffects.Copy))
                {
                    // Ctrlキーが押されていればCopy
                    // "8"はCtrlキーを表す
                    e.Effect = DragDropEffects.Copy;
                }
                else if ((e.AllowedEffect & DragDropEffects.Move) == DragDropEffects.Move)
                {
                    // 何も押されていなければMove
                    e.Effect = DragDropEffects.Move;
                }
                else
                {
                    e.Effect = DragDropEffects.None;
                }
            }
            else
            {
                // TreeNodeでなければ受け入れない
                e.Effect = DragDropEffects.None;
            }

            // マウス下のNodeを選択する
            if (e.Effect != DragDropEffects.None)
            {
                var clientPos = this.PointToClient(new Point(e.X, e.Y));

                // マウスのあるNodeを取得する
                var target = this.GetNodeAt(clientPos) as EBFileTreeNode;

                if (target != null)
                {
                    // ドラッグされているNodeを取得する
                    var source = e.Data.GetData(typeof(EBFileTreeNode)) as EBFileTreeNode;

                    // マウス下のNodeがドロップ先として適切か調べる
                    if ((target != source) && (IsChildNode(source, target) == false))
                    {
                        this.dropTarget = target;
                        this.dropSource = source;
                    }
                    else
                    {
                        e.Effect = DragDropEffects.None;
                    }
                }
                else
                {
                    e.Effect = DragDropEffects.None;
                    this.dropTarget = null;
                    this.dropSource = null;
                }

                // ドロップ位置を作る
                if (target != null)
                {
                    DropPosKind nextDropPos;
                    {
                        var cellH = target.Bounds.Height;
                        var vertPos = clientPos.Y - target.Bounds.Y;

                        if (target.NodeKind == EBFileTreeNode.NodeKindType.FavFolder)
                        {
                            nextDropPos = (vertPos < (cellH / 4))
                                              ? DropPosKind.Before
                                              : (vertPos < (cellH * 3 / 4)) ? DropPosKind.Child : DropPosKind.After;
                        }
                        else
                        {
                            nextDropPos = (vertPos < (cellH * 2 / 4)) ? DropPosKind.Before : DropPosKind.After;
                        }
                    }

                    if (nextDropPos != this.dropPos)
                    {
                        this.dropPos = nextDropPos;
                        this.Invalidate();
                    }
                }
            }
        }

        /// <summary>
        /// The on draw node.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnDrawNode(DrawTreeNodeEventArgs e)
        {
            base.OnDrawNode(e);

            var node = e.Node as EBFileTreeNode;
            if (node == null)
            {
                return;
            }

            if ((e.Bounds.Width == 0) || (e.Bounds.Height == 0))
            {
                return;
            }

            var rect = e.Bounds;

            if (this.dropTarget == node)
            {
                var x = rect.Left - Constants.IconSize - Constants.IconMargin;

                switch (this.dropPos)
                {
                    case DropPosKind.Before:
                        e.Graphics.DrawLine(SystemPens.HotTrack, x, rect.Top, e.Graphics.ClipBounds.Right, rect.Top);
                        break;

                    case DropPosKind.Child:
                        e.Graphics.DrawRectangle(
                            SystemPens.HotTrack,
                            rect.Left,
                            rect.Top,
                            rect.Width - 1,
                            rect.Height - 1);
                        break;

                    case DropPosKind.After:
                        e.Graphics.DrawLine(
                            SystemPens.HotTrack,
                            x,
                            rect.Bottom - 1,
                            e.Graphics.ClipBounds.Right,
                            rect.Bottom - 1);
                        break;
                }
            }
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.TreeView.ItemDrag"/> イベントを発生させます。
        /// </summary>
        /// <param name="e">
        /// イベント データを格納している <see cref="T:System.Windows.Forms.ItemDragEventArgs"/>。
        /// </param>
        protected override void OnItemDrag(ItemDragEventArgs e)
        {
            base.OnItemDrag(e);

            System.Diagnostics.Debug.Assert(e.Item is EBFileTreeNode, "e.Item is EBFileTreeNode");
            var node = e.Item as EBFileTreeNode;

            this.SelectedNode = node;
            this.Focus();

            // お気に入りフォルダとブックマークのみD&Dできる
            if ((node.NodeKind != EBFileTreeNode.NodeKindType.FavFolder)
                && (node.NodeKind != EBFileTreeNode.NodeKindType.Bookmark))
            {
                return;
            }

            // ドラッグ処理準備
            this.dropPos = DropPosKind.None;
            this.dropTarget = null;
            this.dropSource = null;

            // ノードのドラッグを開始する
            var dde = this.DoDragDrop(e.Item, DragDropEffects.All);

            using (new UpdateBlock(this))
            {
                // 移動した時は、ドラッグしたノードを削除する
                if ((dde & DragDropEffects.Move) == DragDropEffects.Move)
                {
                    this.Nodes.Remove(node);
                }

                if (dde != DragDropEffects.None)
                {
                    this.DropNode(this.dropTarget, this.dropSource, this.dropPos);
                }
            }

            // ドラッグ処理準備
            this.dropPos = DropPosKind.None;
            this.dropTarget = null;
            this.dropSource = null;
        }

        /// <summary>
        /// The on mouse down.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            this.doubleClicked = e.Button == MouseButtons.Left && e.Clicks == 2;

            base.OnMouseDown(e);
        }

        /// <summary>
        /// The on node mouse double click.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnNodeMouseDoubleClick(TreeNodeMouseClickEventArgs e)
        {
            var isRenamed = false;

            if ((this.HitTest(e.Location).Location & TreeViewHitTestLocations.PlusMinus)
                != TreeViewHitTestLocations.PlusMinus)
            {
                if ((this.SelectedNode != null) && this.SelectedNode.CanRename)
                {
                    this.Rename(this.SelectedNode);
                    isRenamed = true;
                }
            }

            if (isRenamed == false)
            {
                if (e.Node.IsExpanded)
                {
                    e.Node.Collapse();
                }
                else
                {
                    e.Node.Expand();
                }
            }

            base.OnNodeMouseDoubleClick(e);
        }

        /// <summary>
        /// あるTreeNodeが別のTreeNodeの子ノードか調べる
        /// </summary>
        /// <param name="parentNode">
        /// 親ノードか調べるTreeNode
        /// </param>
        /// <param name="childNode">
        /// 子ノードか調べるTreeNode
        /// </param>
        /// <returns>
        /// 子ノードの時はTrue
        /// </returns>
        private static bool IsChildNode(TreeNode parentNode, TreeNode childNode)
        {
            if (childNode.Parent == parentNode)
            {
                return true;
            }

            if (childNode.Parent != null)
            {
                return IsChildNode(parentNode, childNode.Parent);
            }

            return false;
        }

        /// <summary>
        /// The drop node.
        /// </summary>
        /// <param name="target">
        /// The target.
        /// </param>
        /// <param name="source">
        /// The source.
        /// </param>
        /// <param name="pos">
        /// The pos.
        /// </param>
        private void DropNode(EBFileTreeNode target, EBFileTreeNode source, DropPosKind pos)
        {
            System.Diagnostics.Debug.Assert(target != null, "target != null");
            System.Diagnostics.Debug.Assert(source != null, "source != null");

            var clonedNode = source.Clone() as EBFileTreeNode;
            System.Diagnostics.Debug.Assert(clonedNode != null, "clonedNode != null");

            // 追加されたNodeを選択
            this.SelectedNode = clonedNode;

            var targetNodes = (target.Parent != null) ? target.Parent.Nodes : this.Nodes;

            switch (pos)
            {
                case DropPosKind.Before:
                    targetNodes.Insert(target.Index + 0, clonedNode);
                    break;

                case DropPosKind.After:
                    targetNodes.Insert(target.Index + 1, clonedNode);
                    break;

                case DropPosKind.Child:
                    target.Nodes.Add(clonedNode);
                    target.ExpandWithoutNodeClear();
                    break;
            }
        }

        /// <summary>
        /// ツリーノードからFilePathからDirectoryなノードを見つけてくる見つからない時は null
        /// </summary>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        private EBFileTreeNode FindDirectoryNode()
        {
            foreach (var node in
                this.Nodes.OfType<EBFileTreeNode>()
                    .Where(
                        x =>
                        (x.NodeKind == EBFileTreeNode.NodeKindType.Bookmark)
                        || (x.NodeKind == EBFileTreeNode.NodeKindType.FavFolder)))
            {
                if (this.IsFavDirectoryNode(node))
                {
                    return node;
                }

                var foundNode = this.FindDirectoryNode(node);
                if (foundNode != null)
                {
                    return foundNode;
                }
            }

            return null;
        }

        /// <summary>
        /// The find directory node.
        /// </summary>
        /// <param name="parent">
        /// The parent.
        /// </param>
        /// <returns>
        /// The <see cref="EBFileTreeNode"/>.
        /// </returns>
        private EBFileTreeNode FindDirectoryNode(EBFileTreeNode parent)
        {
            foreach (var node in
                parent.Nodes.OfType<EBFileTreeNode>()
                    .Where(
                        x =>
                        (x.NodeKind == EBFileTreeNode.NodeKindType.Bookmark)
                        || (x.NodeKind == EBFileTreeNode.NodeKindType.FavFolder)))
            {
                if (this.IsFavDirectoryNode(node))
                {
                    return node;
                }

                var foundNode = this.FindDirectoryNode(node);
                if (foundNode != null)
                {
                    return foundNode;
                }
            }

            return null;
        }

        /// <summary>
        /// The is fav directory node.
        /// </summary>
        /// <param name="node">
        /// The node.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool IsFavDirectoryNode(EBFileTreeNode node)
        {
            if (node.NodeKind == EBFileTreeNode.NodeKindType.Bookmark)
            {
                if (this.Directory.StartsWith(node.Filepath))
                {
                    return true;
                }
            }

            return false;
        }

        #endregion
    }
}
