﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.IO;

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileTreeView
{
    using System;
    using System.ComponentModel;
    using System.Linq;

    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb computer file tree view.
    /// </summary>
    public class EBComputerFileTreeView : EBFileTreeView
    {
        #region Fields

        /// <summary>
        /// The desktop node.
        /// </summary>
        private EBFileTreeNode desktopNode;

        /// <summary>
        /// The my computer node.
        /// </summary>
        private EBFileTreeNode myComputerNode;

        /// <summary>
        /// The my documents node.
        /// </summary>
        private EBFileTreeNode myDocumentsNode;

        /// <summary>
        /// The network node.
        /// </summary>
        private EBFileTreeNode networkNode;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets a value indicating whether computer expanded.
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool ComputerExpanded
        {
            get
            {
                return this.myComputerNode.IsExpanded;
            }

            set
            {
                this.myComputerNode.IsExpanded = value;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether desktop expanded.
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool DesktopExpanded
        {
            get
            {
                return this.desktopNode.IsExpanded;
            }

            set
            {
                this.desktopNode.IsExpanded = value;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether documents expanded.
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool DocumentsExpanded
        {
            get
            {
                return this.myDocumentsNode.IsExpanded;
            }

            set
            {
                this.myDocumentsNode.IsExpanded = value;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The expand current directory node.
        /// </summary>
        public override void ExpandCurrentDirectoryNode()
        {
            if (PathUtility.IsNetworkPath(this.Directory))
            {
                // ネットワーク上に存在しないなら、何もしない
                if (PathUtility.IsAccessibleDirectory(this.Directory) == false)
                {
                    return;
                }

                var splitted = this.Directory.Substring(2).Split('\\');
                if (splitted.Length < 2 || string.IsNullOrEmpty(splitted[0]))
                {
                    return;
                }

                var rootPath = @"\\" + splitted[0] + @"\" + splitted[1] + @"\";
                var node = this.networkNode.ChildNodes.FirstOrDefault(n => n.Filepath.Equals(rootPath));
                if (node == null)
                {
                    using (new UpdateBlock(this))
                    {
                        node = this.AddNode(
                            this.networkNode,
                            rootPath.Substring(0, rootPath.Length - 1),
                            rootPath,
                            IconKeyDirectory,
                            EBFileTreeNode.NodeKindType.Directory);
                        node.IsRoot = true;
                    }
                }

                using (new UpdateBlock(this))
                {
                    this.networkNode.Expand();
                    node.Expand();

                    if (splitted.Length > 1)
                    {
                        var dirs = this.Directory.Substring(Math.Min(rootPath.Length, this.Directory.Length))
                            .Split(new[] { @"\" }, StringSplitOptions.RemoveEmptyEntries);
                        node = this.ExpandCurrentDirectoryNode(rootPath, node, dirs);
                    }

                    this.SelectedNode = node;
                    this.SelectedNode.EnsureVisible();
                }

                return;
            }

            if (PathUtility.DirectoryExists(this.Directory) == false)
            {
                return;
            }

            using (new UpdateBlock(this))
            {
                var drive = this.Directory.Split(':')[0] + @":\";
                var dirs = this.Directory.Substring(3).Split(new[] { @"\" }, StringSplitOptions.RemoveEmptyEntries);

                var targetNode = this.myComputerNode.ChildNodes.First(x => x.Filepath == drive);
                var targetPath = drive;

                // ドライブ階層まで開く
                this.myComputerNode.Expand();
                targetNode.Expand();

                var isRoot = (dirs.Count() == 1) && (dirs[0] == string.Empty);
                if (isRoot == false)
                {
                    targetNode = this.ExpandCurrentDirectoryNode(targetPath, targetNode, dirs);
                }

                this.SelectedNode = targetNode;
                this.SelectedNode.EnsureVisible();
            }
        }

        /// <summary>
        /// The initialize.
        /// </summary>
        public override void Initialize()
        {
            using (new UpdateBlock(this))
            {
                // デスクトップ
                this.desktopNode = this.AddNode(
                    null,
                    Properties.Resources.FileTreeView_Desktop,
                    Environment.GetFolderPath(Environment.SpecialFolder.Desktop),
                    EBFileTreeView.IconKeyDesktop,
                    EBFileTreeNode.NodeKindType.Directory);

                // マイドキュメント
                this.myDocumentsNode = this.AddNode(
                    null,
                    Properties.Resources.FileTreeView_MyDocuments,
                    Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments),
                    EBFileTreeView.IconKeyMyDocuments,
                    EBFileTreeNode.NodeKindType.Directory);

                // コンピュータ
                this.myComputerNode = this.AddNode(
                    null,
                    Properties.Resources.FileTreeView_MyComputer,
                    Environment.GetFolderPath(Environment.SpecialFolder.MyComputer),
                    EBFileTreeView.IconKeyMyComputer,
                    EBFileTreeNode.NodeKindType.Directory);

                // ネットワーク
                this.networkNode = this.AddNode(
                    null,
                    Properties.Resources.FileTreeView_Network,
                    Environment.GetFolderPath(Environment.SpecialFolder.NetworkShortcuts),
                    EBFileTreeView.IconKeyNetwork,
                    EBFileTreeNode.NodeKindType.Directory);

                foreach (var drivePath in Environment.GetLogicalDrives().Where(PathUtility.IsAccessibleDirectory))
                {
                    var driveNode = this.AddNode(
                        this.myComputerNode,
                        drivePath,
                        drivePath,
                        GetLogicalDriveIconKey(drivePath),
                        EBFileTreeNode.NodeKindType.Directory);
                    driveNode.IsRoot = true;
                }
            }
        }

        #endregion
    }
}
