﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Diagnostics;
using System.IO;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UIControls.EffectBrowser.Controls.FileListView.Base
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Drawing;
    using System.Drawing.Imaging;
    using System.Linq;
    using System.Threading;
    using System.Windows.Forms;

    using EffectMaker.Foundation.Extensions;
    using EffectMaker.UIControls.EffectBrowser.Controls.Basic;
    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb file list view.
    /// </summary>
    public partial class EBFileListView : EBListView
    {
        /// <summary>
        /// The column widths.
        /// </summary>
        private static readonly Dictionary<ColumnKinds, int> ColumnWidths =
            new Dictionary<ColumnKinds, int>
            {
                { ColumnKinds.Name, 200 },
                { ColumnKinds.UpdateTimestamp, 120 },
                { ColumnKinds.CreateTimestamp, 80 },
                { ColumnKinds.FileKind, 120 },
                { ColumnKinds.ByteSize, 70 },
                { ColumnKinds.LabelColor, 90 },
                { ColumnKinds.Comment, 200 },
                { ColumnKinds.FileFullPath, 340 }
            };

        #region Fields

        /// <summary>
        /// The column 0 left.
        /// </summary>
        private int column0Left;

        /// <summary>
        /// The file not found image.
        /// </summary>
        private Bitmap fileNotFoundImage;

        /// <summary>
        /// The last mouse down file info.
        /// </summary>
        private FileInfo lastMouseDownFileInfo;

        /// <summary>
        /// 文字キー選択バッファ
        /// </summary>
        private string inputKeyword = string.Empty;

        /// <summary>
        /// 最後の文字キー入力からの経過時間測定
        /// </summary>
        private Stopwatch inputTimer = new Stopwatch();

        /// <summary>
        /// ツールチップ
        /// </summary>
        private EBToolTip toolTip = new EBToolTip();

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EBFileListView()
        {
            this.OwnerDraw = true;
            this.MultiSelect = true;
            this.VirtualMode = true;
            this.AllowColumnReorder = true;

            this.inputTimer.Restart();

            this.Initialize_Update();
        }

        #endregion

        #region Public Events

        /// <summary>
        /// The after column reordered.
        /// </summary>
        public event EventHandler AfterColumnReordered;

        /// <summary>
        /// The file open.
        /// </summary>
        public event EventHandler FileOpen;

        #endregion

        #region Enums

        /// <summary>
        /// The column kinds.
        /// </summary>
        public enum ColumnKinds
        {
            /// <summary>
            /// The name.
            /// </summary>
            Name,

            /// <summary>
            /// The update timestamp.
            /// </summary>
            UpdateTimestamp,

            /// <summary>
            /// The create timestamp.
            /// </summary>
            CreateTimestamp,

            /// <summary>
            /// The file kind.
            /// </summary>
            FileKind,

            /// <summary>
            /// The byte size.
            /// </summary>
            ByteSize,

            /// <summary>
            /// The label color.
            /// </summary>
            LabelColor,

            /// <summary>
            /// The comment.
            /// </summary>
            Comment,

            /// <summary>
            /// The file full path.
            /// </summary>
            FileFullPath
        }

        /// <summary>
        /// The view type kind.
        /// </summary>
        public enum ViewTypeKind
        {
            /// <summary>
            /// The file list.
            /// </summary>
            FileList,

            /// <summary>
            /// The child file list.
            /// </summary>
            ChildFileList
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// フィルタリング前のファイルリストを取得します。
        /// </summary>
        public int UnfilteredFilesCount
        {
            get { return this.unfilteredFiles.Count(); }
        }

        /// <summary>
        /// Gets a value indicating whether can edit.
        /// </summary>
        public bool CanEdit
        {
            get
            {
                return this.UpdateState == UpdateStateKind.Wait;
            }
        }

        // 注意:abstractなコントロールはフォームデザイナがエラーが起きるため virtual にしてます。

        /// <summary>
        /// Gets the column infos.
        /// </summary>
        public Dictionary<ColumnKinds, EBFileListViewColumn> ColumnInfos { get; private set; }

        /// <summary>
        /// Gets the opened all files.
        /// </summary>
        [Browsable(false)]
        public IEnumerable<FileInfo> OpenedAllFiles
        {
            get
            {
                return this.openedFiles;
            }
        }

        /// <summary>
        /// Gets the search filter info.
        /// </summary>
        public string SearchFilterInfo
        {
            get
            {
                return string.Format("{0} / {1}", this.searchedFiles.Count(), this.unfilteredFiles.Count());
            }
        }

        /// <summary>
        /// Gets the searched all files count.
        /// </summary>
        public int SearchedAllFilesCount
        {
            get
            {
                return this.searchedFiles.Count();
            }
        }

        /// <summary>
        /// Gets or sets the selected file.
        /// </summary>
        [Browsable(false)]
        public FileInfo SelectedFile
        {
            get
            {
                FileInfo item = null;
                {
                    var index = (this.SelectedIndices.Count > 0) ? this.SelectedIndices[0] : -1;
                    if (index != -1)
                    {
                        item = this.openedFiles[index];
                    }
                }

                return item;
            }

            set
            {
                var index = 0;
                this.openedFiles.ForEach(x => x.Index = index++);

                if (value != null)
                {
                    this.SelectedIndices.Clear();
                    this.SelectedIndices.Add(value.Index);
                    this.Items[value.Index].Focused = true;
                }
            }
        }

        /// <summary>
        /// Gets the selected files.
        /// </summary>
        [Browsable(false)]
        public IEnumerable<FileInfo> SelectedFiles
        {
            get
            {
                return this.SelectedIndices.OfType<int>().Select(x => this.openedFiles[x]);
            }
        }

        /// <summary>
        /// Gets the selected files count.
        /// </summary>
        public int SelectedFilesCount
        {
            get
            {
                return this.SelectedIndices.Count;
            }
        }

        /// <summary>
        /// Gets or sets the view type.
        /// </summary>
        public ViewTypeKind ViewType { get; set; }

        #endregion

        #region Properties

        /// <summary>
        /// リストビューで管理するファイルリストを取得します。
        /// 画面にはこのファイルリストをソートしてフィルタリングした結果が表示されます。
        /// </summary>
        /// <exception cref="NotImplementedException">NotImplementedException</exception>
        protected virtual IEnumerable<FileInfo> Files
        {
            get
            {
                throw new NotImplementedException();
            }
        }

        /// <summary>
        /// Gets the file not found image.
        /// </summary>
        private Bitmap FileNotFoundImage
        {
            get
            {
                if (this.fileNotFoundImage == null)
                {
                    var width = Math.Max(1, this.Width);
                    var height = Math.Max(1, this.Height);

                    var bitmap = new Bitmap(width, height, PixelFormat.Format16bppRgb565);
                    using (var g = Graphics.FromImage(bitmap))
                    {
                        using (var b = new SolidBrush(Color.Gray))
                        {
                            g.Clear(Color.White);
                            var textSize = g.MeasureString(Properties.Resources.FileNotFound, this.Font);
                            g.DrawString(
                                Properties.Resources.FileNotFound,
                                this.Font,
                                b,
                                (width - textSize.Width) / 2.0f,
                                (height - textSize.Height) / 2.0f);
                        }
                    }

                    this.fileNotFoundImage = bitmap;
                }

                return this.fileNotFoundImage;
            }
        }

        #endregion

        #region Public Methods and Operators

        /// <summary>
        /// The copy column info.
        /// </summary>
        /// <param name="srcFlv">
        /// The src flv.
        /// </param>
        public void CopyColumnInfo(EBFileListView srcFlv)
        {
            System.Diagnostics.Debug.Assert(srcFlv != null, "srcFlv != null");

            if ((this.ColumnInfos == null) || (srcFlv.ColumnInfos == null))
            {
                return;
            }

            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                this.ColumnInfos.Values.Zip(srcFlv.ColumnInfos.Values, (d, s) => new { Dst = d, Src = s })
                    .ForEach(x => x.Dst.Copy(x.Src));
            }

            this.UpdateColumnHeader();
        }

        /// <summary>
        /// The destory.
        /// </summary>
        public void Destory()
        {
            this.CancelAsyncJob();
            this.throttleExecuter.Dispose();
        }

        /// <summary>
        /// The get column visible.
        /// </summary>
        /// <param name="kind">
        /// The kind.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public bool GetColumnVisible(ColumnKinds kind)
        {
            System.Diagnostics.Debug.Assert(this.ColumnInfos.ContainsKey(kind), "this.ColumnInfos.ContainsKey(kind)");

            return this.ColumnInfos[kind].Visible;
        }

        /// <summary>
        /// The normalize column header.
        /// </summary>
        public void NormalizeColumnHeader()
        {
            // return;
            if (this.ColumnInfos == null)
            {
                return;
            }

            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                this.Columns.Clear();
                this.Columns.AddRange(
                    this.ColumnInfos.Values.Where(x => x.Visible)
                        .OrderBy(x => x.DisplayIndex)
                        .Select(x => x.ColumnHeader)
                        .ToArray());
                this.ColumnInfos.Values.ForEach(x => x.Index = x.DisplayIndex);
                this.SetColumnHeaderSortIcon();

                if (this.Columns.Count > 0)
                {
                    // 注意：これがないと横スクロールバーが消える
                    this.Columns[0].Width += 1;
                    this.Columns[0].Width -= 1;
                }
            }
        }

        /// <summary>
        /// The reset column visible.
        /// </summary>
        public void ResetColumnVisible()
        {
            var index = 0;
            this.ColumnInfos.Values.Where(x => x.Visible).ForEach(
                x =>
                    {
                        x.Visible = true;
                        x.DisplayIndex = index++;
                    });

            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                this.Columns.Clear();
                this.Columns.AddRange(this.ColumnInfos.Values.Select(x => x.ColumnHeader).ToArray());

                this.ColumnInfos.Values.ForEach(
                    x =>
                        {
                            x.ColumnHeader.DisplayIndex = x.DisplayIndex;

                            if (this.Columns.Count > 0)
                            {
                                // 注意：これがないと横スクロールバーが消える
                                this.Columns[0].Width += 1;
                                this.Columns[0].Width -= 1;
                            }

                            this.Columns.Cast<ColumnHeader>().ForEach(y => y.Width = ColumnWidths[(ColumnKinds)y.Tag]);

                            x.Visible = true;
                            x.SortOrder = SortOrder.Ascending;
                        });

                this.currentSortColumn = this.ColumnInfos[0];
            }

            this.ColumnInfos.Values.ForEach(x => { x.ColumnHeader.DisplayIndex = x.DisplayIndex; });

            this.NormalizeColumnHeader();

            this.RunUpdate(UpdateStateKind.Resort);
        }

        /// <summary>
        /// The select first file.
        /// </summary>
        public void SelectFirstFile()
        {
            if (this.ItemsCount > 0)
            {
                this.SelectedIndices.Clear();
                this.SelectedIndices.Add(0);
                this.Items[0].Focused = true;
            }
        }

        /// <summary>
        /// The set column visible.
        /// </summary>
        /// <param name="kind">
        /// The kind.
        /// </param>
        /// <param name="visible">
        /// The visible.
        /// </param>
        public void SetColumnVisible(ColumnKinds kind, bool visible)
        {
            System.Diagnostics.Debug.Assert(this.ColumnInfos.ContainsKey(kind), "this.ColumnInfos.ContainsKey(kind)");

            this.ColumnInfos[kind].Visible = visible;

            this.UpdateColumnHeader();
        }

        /// <summary>
        /// The update column header.
        /// </summary>
        public void UpdateColumnHeader()
        {
            using (new LockWindowUpdateBlock(this))
            using (new LayoutSuspendBlock(this))
            using (new ControlEventSuppressBlock())
            {
                this.Columns.Clear();
                this.Columns.AddRange(
                    this.ColumnInfos.Values.Where(x => x.Visible).Select(x => x.ColumnHeader).ToArray());

                // DisplayIndex を復元
                this.ColumnInfos.Values.Where(x => x.Visible)
                    .ForEach(
                        x =>
                        x.ColumnHeader.DisplayIndex =
                        x.DisplayIndex
                        - this.ColumnInfos.Values.Where(y => y.Visible == false)
                              .Count(y => y.DisplayIndex < x.DisplayIndex));

                // ソートアイコンを設定
                this.SetColumnHeaderSortIcon();
            }

            this.NormalizeColumnHeader();
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on click.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);

            // 名前カラム表示中
            if (this.ColumnInfos[ColumnKinds.Name].Visible == false)
            {
                return;
            }

            // 子の開閉
            if (this.SelectedFile != null)
            {
                var x = this.PointToClient(Cursor.Position).X;

                if (this.InChildrenExpander(x))
                {
                    if ((this.SelectedFile.FileKind == FileKindType.EsetFile) && this.SelectedFile.HasChildren)
                    {
                        this.SetIsOpend(this.SelectedFile, !this.SelectedFile.IsOpend);
                    }
                }
            }

            this.StartInlineEditing();
        }

        /// <summary>
        /// The on column reordered.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnColumnReordered(ColumnReorderedEventArgs e)
        {
            this.UpdateColumn0Left();

            RunOnIdleProcess.Execute(
                () =>
                    {
                        this.ColumnInfos.Values.ForEach(x => x.DisplayIndex = x.ColumnHeader.DisplayIndex);

                        if (this.AfterColumnReordered != null)
                        {
                            this.AfterColumnReordered(this, EventArgs.Empty);
                        }
                    });

            base.OnColumnReordered(e);
        }

        /// <summary>
        /// The on column width changed.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnColumnWidthChanged(ColumnWidthChangedEventArgs e)
        {
            if (this.IsSettingColumnHeaderSortIcon)
            {
                return;
            }

            this.UpdateColumn0Left();

            base.OnColumnWidthChanged(e);
        }

        /// <summary>
        /// The on double click.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnDoubleClick(EventArgs e)
        {
            var handled = false;

            if (this.SelectedFile != null)
            {
                if ((this.SelectedFile.FileKind == FileKindType.EsetFile) && this.SelectedFile.HasChildren)
                {
                    // エクスパンダー上ではダブルクリックを受け付けない
                    var x = this.PointToClient(Cursor.Position).X;
                    if (this.InChildrenExpander(x))
                    {
                        handled = true;
                    }
                }
            }

            if (handled == false)
            {
                if (this.SelectedFile != null)
                {
                    if (this.FileOpen != null)
                    {
                        this.FileOpen(this, EventArgs.Empty);
                        handled = true;
                    }
                }
            }

            if (handled == false)
            {
                base.OnDoubleClick(e);
            }
        }

        /// <summary>
        /// The on draw item.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnDrawItem(DrawListViewItemEventArgs e)
        {
            if (this.View == View.Details)
            {
                this.UpdateColumn0Left();
                base.OnDrawItem(e);
            }
            else if (this.View == View.LargeIcon)
            {
                this.DrawLargeItem(e);
            }
        }

        /// <summary>
        /// The on key down.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Left:
                    if (this.SelectedFile != null)
                    {
                        if (this.View == View.Details)
                        {
                            if ((this.SelectedFile.FileKind == FileKindType.EsetFile) && this.SelectedFile.HasChildren)
                            {
                                this.SetIsOpend(this.SelectedFile, false);
                                e.Handled = true;
                            }
                            else
                            {
                                var parent = this.SelectedFile.Parent;
                                if (parent != null)
                                {
                                    this.SetIsOpend(parent, false);
                                    this.SelectedFile = parent;
                                    e.Handled = true;
                                }
                            }
                        }
                    }

                    break;

                case Keys.Right:
                    if (this.SelectedFile != null)
                    {
                        if (this.View == View.Details)
                        {
                            if ((this.SelectedFile.FileKind == FileKindType.EsetFile) && this.SelectedFile.HasChildren)
                            {
                                this.SetIsOpend(this.SelectedFile, true);
                                e.Handled = true;
                            }
                        }
                    }

                    break;

                case Keys.Return:
                    if (this.SelectedFile != null)
                    {
                        // テクスチャのオープンは受け付けない
                        if (this.SelectedFile.FileKind != FileKindType.TextureFile)
                        {
                            if (this.FileOpen != null)
                            {
                                this.FileOpen(this, EventArgs.Empty);
                                e.Handled = true;
                            }
                        }
                    }

                    break;
            }

            base.OnKeyDown(e);
        }

        /// <summary>
        /// キー入力によるファイル選択
        /// </summary>
        /// <param name="e">キーイベント</param>
        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            FileInfo target = null;

            string inputChar = e.KeyChar.ToString();

            if (this.inputTimer.ElapsedMilliseconds < 500)
            {
                this.inputKeyword += inputChar;
            }
            else
            {
                this.inputKeyword = inputChar;
            }

            this.inputTimer.Restart();

            var wordSearch = this.OpenedAllFiles.Where(x =>
                x.DisplayFileName.StartsWith(
                this.inputKeyword,
                StringComparison.OrdinalIgnoreCase)).ToList();

            var charSearch = this.OpenedAllFiles.Where(x => string.Compare(
                x.DisplayFileName[0].ToString(),
                inputChar,
                StringComparison.OrdinalIgnoreCase) == 0).ToList();

            var targetList = wordSearch.Count > 0 ? wordSearch : charSearch;

            if (targetList.Count > 0)
            {
                int index = targetList.IndexOf(this.SelectedFile);
                if (index >= 0 && index < targetList.Count - 1)
                {
                    target = targetList[index + 1];
                }
                else
                {
                    target = targetList[0];
                }
            }

            if (target != null)
            {
                this.SelectedFile = target;
                this.EnsureVisible(target.Index);
                e.Handled = true;
            }

            base.OnKeyPress(e);
        }

        /// <summary>
        /// The on mouse down.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);
            this.lastMouseDownFileInfo = this.SelectedFile;
        }

        /// <summary>
        /// The on parent changed.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        /// <exception cref="NotSupportedException">
        /// NotSupportedException
        /// </exception>
        protected override void OnParentChanged(EventArgs e)
        {
            base.OnParentChanged(e);

            if (this.DesignMode)
            {
                return;
            }

            this.UpdateBackgroundImage();

            switch (this.ViewType)
            {
                case ViewTypeKind.FileList:
                    this.Columns[0].Tag = ColumnKinds.Name;
                    this.Columns[1].Tag = ColumnKinds.UpdateTimestamp;
                    this.Columns[2].Tag = ColumnKinds.CreateTimestamp;
                    this.Columns[3].Tag = ColumnKinds.FileKind;
                    this.Columns[4].Tag = ColumnKinds.ByteSize;
                    this.Columns[5].Tag = ColumnKinds.LabelColor;
                    this.Columns[6].Tag = ColumnKinds.Comment;
                    break;

                case ViewTypeKind.ChildFileList:
                    this.Columns[0].Tag = ColumnKinds.Name;
                    this.Columns[1].Tag = ColumnKinds.FileFullPath;
                    this.Columns[2].Tag = ColumnKinds.ByteSize;
                    this.Columns[3].Tag = ColumnKinds.UpdateTimestamp;
                    break;

                default:
                    throw new NotSupportedException();
            }

            var index = 0;
            this.ColumnInfos = this.Columns.OfType<ColumnHeader>().Select(
                x =>
                    {
                        var column = new EBFileListViewColumn { ColumnHeader = x, Index = index, DisplayIndex = index };

                        ++index;
                        return column;
                    }).ToDictionary(x => (ColumnKinds)x.ColumnHeader.Tag);

            // 初期ソート状態を設定する
            this.currentSortColumn = this.ColumnInfos.Values.First(x => x.Index == 0);
            this.currentSortColumn.SortOrder = SortOrder.Ascending;
            this.SetColumnHeaderSortIcon();
        }

        /// <summary>
        /// The on resize.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            this.UpdateBackgroundImage();
        }

        /// <summary>
        /// コメントとカラーラベルの編集がブロックされている場合のツールチップを処理します。
        /// </summary>
        /// <returns>ブロック中の場合はtrue,そうでない場合はfalse.</returns>
        private bool HandleToolTipForBlockingEdit()
        {
            var pos = this.PointToClient(Cursor.Position);
            var result = this.HitTest(pos);
            if (result.Item == null || result.SubItem == null)
            {
                return false;
            }

            var info = (FileInfo)result.Item.Tag;
            var kind = (ColumnKinds)result.SubItem.Tag;
            if ((kind == ColumnKinds.LabelColor || kind == ColumnKinds.Comment) && info.IsBlockingEditByOpened)
            {
                this.toolTip.Show(Properties.Resources.ToolTipBlockingEdit, this, pos.X, pos.Y + 25, 2000);

                return true;
            }
            else
            {
                this.toolTip.Hide(this);

                return false;
            }
        }

        /// <summary>
        /// 小開閉の[+]に入っているか？
        /// </summary>
        /// <param name="x">
        /// The x.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool InChildrenExpander(int x)
        {
            var nameIndex = this.ColumnInfos[ColumnKinds.Name].ColumnHeader.DisplayIndex;
            var nameLeft =
                this.Columns.OfType<ColumnHeader>()
                    .OrderBy(c => c.DisplayIndex)
                    .Where(ch => ch.DisplayIndex < nameIndex)
                    .Sum(ch => ch.Width);

            var left = nameLeft;
            var right = nameLeft + Math.Min(this.ColumnInfos[ColumnKinds.Name].ColumnHeader.Width, Constants.IconSize);

            return (x >= left) && (x <= right);
        }

        /// <summary>
        /// The set is opend.
        /// </summary>
        /// <param name="fileInfo">
        /// The file info.
        /// </param>
        /// <param name="isOpend">
        /// The is opend.
        /// </param>
        private void SetIsOpend(FileInfo fileInfo, bool isOpend)
        {
            System.Diagnostics.Debug.Assert(fileInfo != null, "fileInfo != null");

            fileInfo.IsOpend = isOpend;
            this.RunUpdate(UpdateStateKind.CloseOpen);
        }

        /// <summary>
        /// The update background image.
        /// </summary>
        private void UpdateBackgroundImage()
        {
            if (this.VirtualListSize > 0)
            {
                this.BackgroundImage = null;
            }
            else
            {
                var width = Math.Max(1, this.Width);
                var height = Math.Max(1, this.Height);

                if ((this.FileNotFoundImage.Width != width) || (this.FileNotFoundImage.Height != height))
                {
                    this.fileNotFoundImage.Dispose();
                    this.fileNotFoundImage = null;
                }

                this.BackgroundImage = this.FileNotFoundImage;
            }
        }

        /// <summary>
        /// The update column 0 left.
        /// </summary>
        private void UpdateColumn0Left()
        {
            this.column0Left =
                this.Columns.OfType<ColumnHeader>()
                    .OrderBy(x => x.DisplayIndex)
                    .TakeWhile(ch => ch.Index != 0)
                    .Sum(ch => ch.Width);
            this.column0Left -= this.HScrollPosition;
        }

        #endregion
    }
}
