﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList
{
    using System.ComponentModel;
    using System.Drawing;
    using System.IO;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb dir breadcrumb list name.
    /// </summary>
    public class EBDirBreadcrumbListName : EBDirBreadcrumbListItem
    {
        #region Constants

        /// <summary>
        /// The horizontal margin.
        /// </summary>
        private const int HorizontalMargin = 4;

        /// <summary>
        /// The text draw flag.
        /// </summary>
        private const TextFormatFlags TextDrawFlag = TextFormatFlags.HorizontalCenter | TextFormatFlags.VerticalCenter;

        #endregion

        #region Fields

        /// <summary>
        /// The directory.
        /// </summary>
        private string directory = string.Empty;

        /// <summary>
        /// The name.
        /// </summary>
        private string name;

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets the directory.
        /// </summary>
        [Browsable(false)]
        public string Directory
        {
            get
            {
                return this.directory;
            }

            set
            {
                if (this.directory == value)
                {
                    return;
                }

                this.directory = value;

                this.MakeName();

                this.Width =
                    TextRenderer.MeasureText(this.name, this.Font, new Size(10000, this.Height), TextDrawFlag).Width + (HorizontalMargin * 2); // *2 は　左右

                this.Invalidate();
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The on paint.
        /// </summary>
        /// <param name="e">
        /// イベント引数
        /// </param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            TextRenderer.DrawText(e.Graphics, this.name, this.Font, e.ClipRectangle, Color.Black, TextDrawFlag);
        }

        /// <summary>
        /// The make name.
        /// </summary>
        private void MakeName()
        {
            System.Diagnostics.Debug.Assert(this.Directory != null, "Directory != null");

            string rootPath = Path.GetPathRoot(this.Directory);
            // ルートパスかどうか
            if (this.Directory == rootPath)
            {
                // ネットワークパスかどうか
                this.name = this.Directory.StartsWith(@"\\")
                                 ? rootPath.Substring(0, rootPath.Length - 1)
                                 : this.Directory.Substring(0, 2);
            }
            else
            {
                // ネットワークパスかどうか
                this.name = PathUtility.IsMatchToNetworkActiveDirectory(this.Directory)
                                 ? this.Directory.Substring(0, this.Directory.Length - 1)
                                 : Path.GetFileName(this.Directory.Substring(0, this.Directory.Length - 1));
            }
        }

        #endregion
    }
}
