﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList
{
    using System.Drawing;
    using System.Windows.Forms;

    using EffectMaker.UIControls.EffectBrowser.Utilities;

    /// <summary>
    /// The eb dir breadcrumb list item set.
    /// </summary>
    public class EBDirBreadcrumbListItemSet : Control
    {
        #region Fields

        /// <summary>
        /// The arrow.
        /// </summary>
        private readonly EBDirBreadcrumbListDirectoryArrow arrow = new EBDirBreadcrumbListDirectoryArrow
                                                                        {
                                                                            Dock =
                                                                                DockStyle
                                                                                .Left
                                                                        };

        /// <summary>
        /// The name.
        /// </summary>
        private readonly EBDirBreadcrumbListName name = new EBDirBreadcrumbListName { Dock = DockStyle.Left };

        #endregion

        #region Constructors and Destructors

        /// <summary>
        /// Initializes a new instance of the <see cref="EBDirBreadcrumbListItemSet"/> class.
        /// </summary>
        /// <param name="directory">
        /// The directory.
        /// </param>
        /// <param name="font">
        /// The font.
        /// </param>
        public EBDirBreadcrumbListItemSet(string directory, Font font)
        {
            this.AutoSize = true;
            this.DoubleBuffered = true;

            this.arrow.Font = font;
            this.name.Font = font;

            this.Controls.Add(this.arrow);
            this.Controls.Add(this.name);

            this.name.MouseEnter += (s, e) => this.NameMouseEnter();
            this.name.MouseLeave += (s, e) => this.NameMouseLeave();
            this.name.MouseDown += (s, e) => this.NameMouseDown();
            this.name.MouseUp += (s, e) => this.NameMouseUp();
            this.arrow.MouseEnter += (s, e) => this.ArrowMouseEnter();
            this.arrow.MouseLeave += (s, e) => this.ArrowMouseLeave();
            this.arrow.PopupChanged += (s, e) => this.ArrowPopupChanged();

            this.name.Directory = directory;
            this.arrow.Directory = directory;

            this.name.Click += (s, e) => this.SetCurrentDirectory(this.name.Directory);

            this.arrow.MenuClicked += (s, e) =>
                {
                    System.Diagnostics.Debug.Assert(s is ToolStripMenuItem, "s is ToolStripMenuItem");

                    this.SetCurrentDirectory((s as ToolStripMenuItem).Tag as string);
                };

            this.UpdateSize();
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// Gets or sets a value indicating whether auto size.
        /// </summary>
        public override sealed bool AutoSize
        {
            get
            {
                return base.AutoSize;
            }

            set
            {
                base.AutoSize = value;
            }
        }

        /// <summary>
        /// Gets or sets the next depth directory.
        /// </summary>
        public string NextDepthDirectory { get; set; }

        #endregion

        #region Properties

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether double buffered.
        /// </summary>
        protected override sealed bool DoubleBuffered
        {
            get
            {
                return base.DoubleBuffered;
            }

            set
            {
                base.DoubleBuffered = value;
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The arrow mouse enter.
        /// </summary>
        private void ArrowMouseEnter()
        {
            this.name.State = EBDirBreadcrumbListItem.StateKind.MouseOverGray;
            this.arrow.State = EBDirBreadcrumbListItem.StateKind.MouseOver;
        }

        /// <summary>
        /// The arrow mouse leave.
        /// </summary>
        private void ArrowMouseLeave()
        {
            this.name.State = EBDirBreadcrumbListItem.StateKind.Wait;
            this.arrow.State = EBDirBreadcrumbListItem.StateKind.Wait;
        }

        /// <summary>
        /// The arrow popup changed.
        /// </summary>
        private void ArrowPopupChanged()
        {
            if (this.arrow.IsPopupped)
            {
                this.name.State = EBDirBreadcrumbListItem.StateKind.MouseDown;
                this.arrow.State = EBDirBreadcrumbListItem.StateKind.MouseDown;
            }
            else
            {
                this.name.State = EBDirBreadcrumbListItem.StateKind.Wait;
                this.arrow.State = EBDirBreadcrumbListItem.StateKind.Wait;
            }
        }

        /// <summary>
        /// The name mouse down.
        /// </summary>
        private void NameMouseDown()
        {
            this.name.State = EBDirBreadcrumbListItem.StateKind.MouseDown;
            this.arrow.State = EBDirBreadcrumbListItem.StateKind.MouseDown;
        }

        /// <summary>
        /// The name mouse enter.
        /// </summary>
        private void NameMouseEnter()
        {
            this.name.State = EBDirBreadcrumbListItem.StateKind.MouseOver;
            this.arrow.State = EBDirBreadcrumbListItem.StateKind.MouseOver;
        }

        /// <summary>
        /// The name mouse leave.
        /// </summary>
        private void NameMouseLeave()
        {
            this.name.State = EBDirBreadcrumbListItem.StateKind.Wait;
            this.arrow.State = EBDirBreadcrumbListItem.StateKind.Wait;
        }

        /// <summary>
        /// The name mouse up.
        /// </summary>
        private void NameMouseUp()
        {
            this.name.State = EBDirBreadcrumbListItem.StateKind.Wait;
            this.arrow.State = EBDirBreadcrumbListItem.StateKind.Wait;
        }

        /// <summary>
        /// The set current directory.
        /// </summary>
        /// <param name="directory">
        /// The directory.
        /// </param>
        private void SetCurrentDirectory(string directory)
        {
            System.Diagnostics.Debug.Assert(string.IsNullOrEmpty(directory) == false, "string.IsNullOrEmpty(directory) == false");
            System.Diagnostics.Debug.Assert(this.Parent is EBDirBreadcrumbList, "this.Parent is EBDirBreadcrumbList");

            (this.Parent as EBDirBreadcrumbList).Directory = directory;
        }

        /// <summary>
        /// The update size.
        /// </summary>
        private void UpdateSize()
        {
            this.Width = this.arrow.Right;
        }

        #endregion
    }
}
