﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Serialization;

using EffectMaker.Foundation.Serialization;

namespace EffectMaker.UIControls.EffectBrowser.Config
{
    using System;
    using System.Collections.Generic;
    using System.Xml.Serialization;

    /// <summary>
    /// The favorites config.
    /// </summary>
    [Serializable]
    public class FavoritesConfig : FavoriteFolder
    {
    }

    /// <summary>
    /// The favorite node base.
    /// </summary>
    [Serializable]
    public class FavoriteNodeBase : IXmlDocSerializable
    {
        #region Public Properties

        /// <summary>
        /// Gets or sets the name.
        /// </summary>
        [XmlAttribute]
        public string Name { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool ReadXml(XmlDocSerializationContext context)
        {
            this.Name = this.ReadAttribute(context, "Name", this.Name);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "Name", this.Name);
            return true;
        }
    }

    /// <summary>
    /// The favorite bookmark.
    /// </summary>
    [Serializable]
    public class FavoriteBookmark : FavoriteNodeBase
    {
        #region Public Properties

        /// <summary>
        /// Gets or sets the path.
        /// </summary>
        [XmlAttribute]
        public string Path { get; set; }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            if (base.ReadXml(context) == false)
            {
                return false;
            }

            this.Path = this.ReadAttribute(context, "Path", this.Path);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            if (base.WriteXml(context) == false)
            {
                return false;
            }

            this.WriteAttribute(context, "Path", this.Path);
            return true;
        }
    }

    /// <summary>
    /// The favorite folder.
    /// </summary>
    [Serializable]
    public class FavoriteFolder : FavoriteNodeBase
    {
        #region Public Properties

        /// <summary>
        /// Gets or sets the children.
        /// </summary>
        [XmlElement(ElementName = "Bookmark", Type = typeof(FavoriteBookmark))]
        [XmlElement(ElementName = "Folder", Type = typeof(FavoriteFolder))]
        public List<FavoriteNodeBase> Children { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether is expanded.
        /// </summary>
        [XmlAttribute]
        public bool IsExpanded { get; set; }

        /// <summary>
        /// Gets a value indicating whether is expanded specified.
        /// </summary>
        [XmlIgnore]
        public bool IsExpandedSpecified
        {
            get
            {
                return (this is FavoritesConfig) == false;
            }
        }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            if (base.ReadXml(context) == false)
            {
                return false;
            }

            var nameTypeMap = new Dictionary<string, Type>()
            {
                { "Bookmark", typeof(FavoriteBookmark) },
                { "Folder", typeof(FavoriteFolder) },
            };

            this.Children = this.ReadElementsByTagNames<FavoriteNodeBase>(
                context, nameTypeMap).ToList();

            this.IsExpanded = this.ReadAttribute(context, "IsExpanded", this.IsExpanded);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            if (base.WriteXml(context) == false)
            {
                return false;
            }

            foreach (FavoriteNodeBase child in this.Children)
            {
                if (child is FavoriteBookmark)
                {
                    this.WriteElement(context, "Bookmark", (FavoriteBookmark)child);
                }
                else if (child is FavoriteFolder)
                {
                    this.WriteElement(context, "Folder", (FavoriteFolder)child);
                }
            }

            this.WriteAttribute(context, "IsExpanded", this.IsExpanded);

            return true;
        }
    }
}
