﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UIDialogs.SearchDialog
{
    #region SearchValueProcessor

    /// <summary>
    /// 条件値の比較・置換処理を行います。
    /// </summary>
    public abstract class SearchValueProcessor : ISettable, ICloneable
    {
        /// <summary>
        /// ConditionValues プロパティのバッキングフィールドです。
        /// </summary>
        private Dictionary<string, string> conditionValues;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="typeName">型名</param>
        public SearchValueProcessor(string typeName)
        {
            this.TypeName = typeName;
        }

        #region ISettable, ICloneable methods

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public virtual bool Set(object src)
        {
            SearchValueProcessor srcData = src as SearchValueProcessor;

            if (srcData == null || this.TypeName != srcData.TypeName)
            {
                return false;
            }

            this.conditionValues = (srcData.conditionValues != null ? new Dictionary<string, string>(srcData.conditionValues) : null);
            this.OverrideConditionValues = (srcData.OverrideConditionValues != null ? new Dictionary<string, string>(srcData.OverrideConditionValues) : null);
            this.Operators = (srcData.Operators != null ? new Dictionary<string, int>(srcData.Operators) : null);
            this.ReplaceOperators = (srcData.ReplaceOperators != null ? new Dictionary<string, int>(srcData.ReplaceOperators) : null);

            return true;
        }

        /// <summary>
        /// インスタンスのコピー生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public abstract object Clone();

        #endregion

        /// <summary>
        /// 型名を取得します。
        /// </summary>
        public string TypeName { get; private set; }

        /// <summary>
        /// 選択可能な条件値を取得します。
        /// </summary>
        public virtual Dictionary<string, string> ConditionValues
        {
            get
            {
                if (this.OverrideConditionValues != null && this.OverrideConditionValues.Any())
                {
                    return this.OverrideConditionValues;
                }

                return this.conditionValues;
            }

            protected set
            {
                this.conditionValues = value;
            }
        }

        /// <summary>
        /// 上書き用の条件値を取得または設定します。
        /// </summary>
        public virtual Dictionary<string, string> OverrideConditionValues { get; set; }

        /// <summary>
        /// 選択可能な比較方法を取得します。
        /// </summary>
        public virtual Dictionary<string, int> Operators { get; protected set; }

        /// <summary>
        /// 選択可能な置換方法を取得します。
        /// </summary>
        public virtual Dictionary<string, int> ReplaceOperators { get; protected set; }

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public abstract bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex);

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public abstract bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex);
    }

    #endregion

    #region SearchValueProcessorBool

    /// <summary>
    /// bool 値の比較・置換処理を行います。
    /// </summary>
    public class SearchValueProcessorBool : SearchValueProcessor
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchValueProcessorBool()
            : base("bool")
        {
            this.ConditionValues = new Dictionary<string, string>
            {
                { "真", "True"  },
                { "偽", "False" },
            };

            this.Operators = new Dictionary<string, int>
            {
                { "と等しい", 0 },
                { "ではない", 1 },
            };

            this.ReplaceOperators = new Dictionary<string, int>
            {
                { "に置換", 0 }
            };
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchValueProcessorBool(SearchValueProcessorBool obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            SearchValueProcessorBool srcData = src as SearchValueProcessorBool;

            if (src == null)
            {
                return false;
            }

            bool resBaseSet = base.Set(src);

            if (resBaseSet == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new SearchValueProcessorBool(this);
        }

        #endregion

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public override bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex)
        {
            bool targetValue;

            bool resParse = bool.TryParse(xmlElement.InnerText, out targetValue);

            if (resParse == false)
            {
                return false;
            }

            bool result = false;

            // TODO...
            if (operatorIndex == 0)
            {
                result = (xmlElement.InnerText == conditionValue);
            }
            else if (operatorIndex == 1)
            {
                result = (xmlElement.InnerText != conditionValue);
            }

            return result;
        }

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public override bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex)
        {
            xmlElement.InnerText = replaceValue;

            return true;
        }
    }

    #endregion

    #region SearchValueProcessorInt

    /// <summary>
    /// int 値の比較・置換処理を行います。
    /// </summary>
    public class SearchValueProcessorInt: SearchValueProcessor
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchValueProcessorInt()
            : base("int")
        {
            this.Operators = new Dictionary<string, int>
            {
                { "と等しい", 0 },
                { "ではない", 1 },
            };

            this.ReplaceOperators = new Dictionary<string, int>
            {
                { "に置換", 0 }
            };
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchValueProcessorInt(SearchValueProcessorInt obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            SearchValueProcessorInt srcData = src as SearchValueProcessorInt;

            if (src == null)
            {
                return false;
            }

            bool resBaseSet = base.Set(src);

            if (resBaseSet == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new SearchValueProcessorInt(this);
        }

        #endregion

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public override bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex)
        {
            int targetValue;

            bool resParse = int.TryParse(xmlElement.InnerText, out targetValue);

            if (resParse == false)
            {
                return false;
            }

            bool result = false;

            // TODO...
            if (operatorIndex == 0)
            {
                result = (xmlElement.InnerText == conditionValue);
            }
            else if (operatorIndex == 1)
            {
                result = (xmlElement.InnerText != conditionValue);
            }

            return result;
        }

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public override bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex)
        {
            xmlElement.InnerText = replaceValue;

            return true;
        }
    }

    #endregion

    #region SearchValueProcessorFloat

    /// <summary>
    /// float 値の比較・置換処理を行います。
    /// </summary>
    public class SearchValueProcessorFloat : SearchValueProcessor
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchValueProcessorFloat()
            : base("float")
        {
            this.Operators = new Dictionary<string, int>
            {
                { "と等しい", 0 },
                { "ではない", 1 },
            };

            this.ReplaceOperators = new Dictionary<string, int>
            {
                { "に置換", 0 }
            };
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchValueProcessorFloat(SearchValueProcessorFloat obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            SearchValueProcessorFloat srcData = src as SearchValueProcessorFloat;

            if (src == null)
            {
                return false;
            }

            bool resBaseSet = base.Set(src);

            if (resBaseSet == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new SearchValueProcessorFloat(this);
        }

        #endregion

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public override bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex)
        {
            float targetValue;

            bool resParse = float.TryParse(xmlElement.InnerText, out targetValue);

            if (resParse == false)
            {
                return false;
            }

            bool result = false;

            // TODO...
            if (operatorIndex == 0)
            {
                result = (xmlElement.InnerText == conditionValue);
            }
            else if (operatorIndex == 1)
            {
                result = (xmlElement.InnerText != conditionValue);
            }

            return result;
        }

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public override bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex)
        {
            xmlElement.InnerText = replaceValue;

            return true;
        }
    }

    #endregion

    #region SearchValueProcessorVector3

    /// <summary>
    /// Vector3 値の比較・置換処理を行います。
    /// </summary>
    public class SearchValueProcessorVector3 : SearchValueProcessor
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchValueProcessorVector3()
            : base("Vector3f")
        {
            this.Operators = new Dictionary<string, int>
            {
                { "と等しい", 0 },
                { "ではない", 1 },
            };

            this.ReplaceOperators = new Dictionary<string, int>
            {
                { "に置換", 0 }
            };
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchValueProcessorVector3(SearchValueProcessorVector3 obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            SearchValueProcessorVector3 srcData = src as SearchValueProcessorVector3;

            if (src == null)
            {
                return false;
            }

            bool resBaseSet = base.Set(src);

            if (resBaseSet == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new SearchValueProcessorVector3(this);
        }

        #endregion

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public override bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex)
        {
            bool result = false;

            var expected = ToVector3(conditionValue);

            // TODO: 入力値が不正な場合はもっと早い段階でエラーダイアログを出す
            if (expected?.Count() != 3)
            {
                return false;
            }

            var actual = ToVector3(xmlElement.InnerText);

            if (actual?.Count() != 3)
            {
                return false;
            }

            result = (expected[0] == actual[0] && expected[1] == actual[1] && expected[2] == actual[2]);

            // 「と等しい」の場合
            if (operatorIndex == 0)
            {
                return result;
            }
            else
            {
                return !result;
            }
        }

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public override bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex)
        {
            xmlElement.InnerText = replaceValue;

            return true;
        }

        private float[] ToVector3(string str)
        {
            char[] delimiterChars = { ' ', ',', '\t' };

            string[] words = str.Split(delimiterChars, StringSplitOptions.RemoveEmptyEntries);

            List<float> result = new List<float>();

            foreach (var word in words)
            {
                float v = 0.0f;

                bool resParse = float.TryParse(word, out v);

                if (resParse == false)
                {
                    return null;
                }

                result.Add(v);
            }

            return result.ToArray();
        }
    }

    #endregion

    #region SearchValueProcessorString

    /// <summary>
    /// string 値の比較・置換処理を行います。
    /// </summary>
    public class SearchValueProcessorString : SearchValueProcessor
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchValueProcessorString ()
            : base("string")
        {
            this.Operators = new Dictionary<string, int>
            {
                { "と等しい", 0 }
            };

            this.ReplaceOperators = new Dictionary<string, int>
            {
                { "に置換", 0 }
            };
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchValueProcessorString(SearchValueProcessorString obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            SearchValueProcessorString srcData = src as SearchValueProcessorString;

            if (src == null)
            {
                return false;
            }

            bool resBaseSet = base.Set(src);

            if (resBaseSet == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new SearchValueProcessorString(this);
        }

        #endregion

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public override bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex)
        {
            bool result = false;

            if (operatorIndex == 0)
            {
                result = (xmlElement.InnerText == conditionValue);
            }
            else if (operatorIndex == 1)
            {
                result = (xmlElement.InnerText != conditionValue);
            }

            return result;
        }

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public override bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex)
        {
            xmlElement.InnerText = replaceValue;

            return true;
        }
    }

    #endregion

    #region SearchValueProcessorBitTable

    /// <summary>
    /// BitTable 値の比較・置換処理を行います。
    /// </summary>
    public class SearchValueProcessorBitTable : SearchValueProcessor
    {
        /// <summary>
        /// FlagName プロパティのバッキングフィールドです。
        /// </summary>
        private string flagName;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SearchValueProcessorBitTable ()
            : base("BitTable")
        {
            // this.Operators = new Dictionary<string, int>
            // {
            //     { "と等しい", 0 },
            //     { "ではない", 1 },
            //     { "が真"    , 2 },
            //     { "が偽"    , 3 },
            // };

            this.Operators = new Dictionary<string, int>
            {
                { "と等しい", 0 }
            };

            // this.ReplaceOperators = new Dictionary<string, int>
            // {
            //     { "に置換"  , 0 },
            //     { "をオン"  , 2 },
            //     { "をオフ"  , 3 },
            // };

            this.ReplaceOperators = new Dictionary<string, int>
            {
                { "に置換", 0 }
            };
        }

        #region CopyConstructor, ISettable, ICloneable methods

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="obj">初期値</param>
        public SearchValueProcessorBitTable(SearchValueProcessorBitTable obj)
            : this()
        {
            this.Set(obj);
        }

        /// <summary>
        /// 値をセットします。
        /// </summary>
        /// <param name="src">値</param>
        /// <returns>処理が成功したときは true、失敗したときは false を返します。</returns>
        public override bool Set(object src)
        {
            SearchValueProcessorBitTable srcData = src as SearchValueProcessorBitTable;

            if (src == null)
            {
                return false;
            }

            bool resBaseSet = base.Set(src);

            if (resBaseSet == false)
            {
                return false;
            }

            this.FlagName = srcData.FlagName;

            return true;
        }

        /// <summary>
        /// インスタンスのコピーを生成します。
        /// </summary>
        /// <returns>インスタンスのコピーを返します。</returns>
        public override object Clone()
        {
            return new SearchValueProcessorBitTable(this);
        }

        #endregion

        /// <summary>
        /// 上書き用の条件値を取得または設定します。
        /// </summary>
        public override Dictionary<string, string> OverrideConditionValues
        {
            get
            {
                return base.OverrideConditionValues;
            }

            set
            {
                base.OverrideConditionValues = value;

                if (base.OverrideConditionValues != null && base.OverrideConditionValues.Any())
                {
                    this.Operators = new Dictionary<string, int>
                    {
                        { "と等しい", 0 },
                        { "ではない", 1 },
                    };
                }
                else
                {
                    this.ConditionValues = null;
                }
            }
        }

        /// <summary>
        /// フラグ名を取得または設定します。
        /// </summary>
        public string FlagName
        {
            get
            {
                return this.flagName;
            }

            set
            {
                this.flagName = value;

                if (string.IsNullOrEmpty(this.flagName) == false)
                {
                    this.Operators = new Dictionary<string, int>
                    {
                        { "と等しい", 0 }
                    };

                    this.ConditionValues = new Dictionary<string, string>
                    {
                        { "真", "True"  },
                        { "偽", "False" },
                    };
                }
            }
        }

        /// <summary>
        /// 条件に一致するか比較します。
        /// </summary>
        /// <param name="xmlElement">比較対象の XML 要素</param>
        /// <param name="conditionValue">条件値</param>
        /// <param name="operatorIndex">比較方法のインデックス</param>
        /// <returns>条件に一致するとき true、それ以外は false を返します。</returns>
        public override bool IsMatch(XmlElement xmlElement, string conditionValue, int operatorIndex)
        {
            if (string.IsNullOrEmpty(this.FlagName) == false)
            {
                // 条件値の "偽" が選択されたときは比較方法を "ではない" に書き換える
                if (conditionValue == "False")
                {
                    operatorIndex = 1;
                }

                // 条件値をフラグ名に置き換える
                conditionValue = this.FlagName;
            }

            // ビットフラグの完全一致を比較
            if (operatorIndex == 0 || operatorIndex == 1)
            {
                XmlNodeList itemElements = xmlElement.GetElementsByTagName("item");

                bool result = (operatorIndex == 0 ? true : false);

                foreach (XmlNode itemNode in itemElements)
                {
                    XmlElement keyNode = (XmlElement)itemNode.SelectSingleNode("key/String");

                    if (keyNode != null)
                    {
                        string constrainValue = (keyNode.InnerText == conditionValue ? "True" : "False");

                        XmlElement valueNode = (XmlElement)itemNode.SelectSingleNode("value/Boolean");

                        if (valueNode != null && valueNode.InnerText != constrainValue)
                        {
                            result = (operatorIndex == 0 ? false : true);
                            break;
                        }
                    }
                }

                return result;
            }
            // 特定のビット値の一致を比較
            else if (operatorIndex == 2 || operatorIndex == 3)
            {
                string constrainValue = (operatorIndex == 2 ? "True" : "False");

                XmlNodeList itemElements = xmlElement.GetElementsByTagName("item");

                bool result = false;

                foreach (XmlNode itemNode in itemElements)
                {
                    XmlElement keyNode = (XmlElement)itemNode.SelectSingleNode("key/String");

                    if (keyNode != null && keyNode.InnerText == conditionValue)
                    {
                        XmlElement valueNode = (XmlElement)itemNode.SelectSingleNode("value/Boolean");

                        if (valueNode != null)
                        {
                            result = (valueNode.InnerText == constrainValue);
                            break;
                        }
                    }
                }

                return result;
            }

            return false;
        }

        /// <summary>
        /// 値を置換します。
        /// </summary>
        /// <param name="xmlElement">置換対象の XML 要素</param>
        /// <param name="replaceValue">置換する値</param>
        /// <param name="operatorIndex">置換方法のインデックス</param>
        /// <returns>置換を行ったとき true、それ以外は false を返します。</returns>
        public override bool Replace(XmlElement xmlElement, string replaceValue, int operatorIndex)
        {
            if (string.IsNullOrEmpty(this.FlagName) == false)
            {
                // 条件値の "真" が選択されたときは置換方法を "をオン" に、"偽" が選択されたときは "をオフ" に書き換える
                operatorIndex = (replaceValue == "True" ? 2 : 3);

                // 条件値をフラグ名に置き換える
                replaceValue = this.FlagName;
            }

            // ビットフラグを置換
            if (operatorIndex == 0)
            {
                XmlNodeList itemElements = xmlElement.GetElementsByTagName("item");

                bool result = false;

                foreach (XmlNode itemNode in itemElements)
                {
                    XmlElement keyNode = (XmlElement)itemNode.SelectSingleNode("key/String");

                    if (keyNode != null)
                    {
                        string constrainValue = (keyNode.InnerText == replaceValue ? "True" : "False");

                        XmlElement valueNode = (XmlElement)itemNode.SelectSingleNode("value/Boolean");

                        if (valueNode != null)
                        {
                            valueNode.InnerText = constrainValue;

                            if (keyNode.InnerText == replaceValue)
                            {
                                result = true;
                            }
                        }
                    }
                }

                return result;
            }
            // 特定のビット値のオンオフを置換
            else if (operatorIndex == 2 || operatorIndex == 3)
            {
                string constrainValue = (operatorIndex == 2 ? "True" : "False");

                XmlNodeList itemElements = xmlElement.GetElementsByTagName("item");

                bool result = false;

                foreach (XmlNode itemNode in itemElements)
                {
                    XmlElement keyNode = (XmlElement)itemNode.SelectSingleNode("key/String");

                    if (keyNode != null && keyNode.InnerText == replaceValue)
                    {
                        XmlElement valueNode = (XmlElement)itemNode.SelectSingleNode("value/Boolean");

                        if (valueNode != null)
                        {
                            valueNode.InnerText = constrainValue;
                            result = true;
                            break;
                        }
                    }
                }

                return result;
            }

            return false;
        }
    }

    #endregion
}
