﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Configuration;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.BusinessLogic.Search;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIDialogs.SearchDialog
{
    public class SearchArgs
    {
        public bool IsDoReplace { get; set; } = false;

        public string Folder { get; set; }

        public SearchLogger Logger { get; set; }

        public List<UISearchCondition> SearchPanels { get; set; }

        public List<UISearchCondition> ReplacePanels { get; set; }

        public SearchTree Tree { get; set; }
    }

    public static class SearchUtility
    {
        private const int uiConditionHeight = 32;

        public static bool Evaluate(this SearchCondition condition, XmlNode node, SearchLogger logger)
        {
            // プラグインを持つ場合
            if (condition.Plugin != null)
            {
                condition.Plugin.Operator = condition.Operator;
                return condition.Plugin.Search(node, logger);
            }

            // アトリビュートの場合
            if (condition.IsAttribute)
            {
                return EvaluateAttribute(condition, node, logger);
            }

            // 通常のプロパティの場合
            {
                return EvaluateStandard(condition, node, logger);
            }
        }

        /// <summary>
        /// 通常の検索を行います。
        /// </summary>
        private static bool EvaluateStandard(SearchCondition condition, XmlNode node, SearchLogger logger)
        {
            string xpathSub = ".//" + condition.XPath;

            XmlNode subNode = node.SelectSingleNode(xpathSub);

            if (subNode == null || subNode.NodeType != XmlNodeType.Element)
            {
                return false;
            }

            string text = condition.Value.ToString();

            logger.Log("    " + subNode.OuterXml);
            logger.Log("");

            bool isMatch = condition.Processor.IsMatch((XmlElement)subNode, condition.Value.ToString(), condition.Operator);

            return isMatch;
        }

        /// <summary>
        /// アトリビュートに対する検索を行います。
        /// </summary>
        private static bool EvaluateAttribute(SearchCondition condition, XmlNode node, SearchLogger logger)
        {
            string xpathSub = ".//" + condition.XPath;

            XmlNode subNode = node.SelectSingleNode(xpathSub);

            if (subNode == null)
            {
                return false;
            }

            logger.Log("    " + subNode.OuterXml);
            logger.Log("");

            string attr = ((XmlElement) subNode).GetAttribute(condition.Attribute);

            if (String.IsNullOrEmpty(attr))
            {
                return false;
            }

            //// TODO: マジックナンバー排除
            //if (condition.Operator == 0)
            //{
            //    return attr == condition.Value.ToString();
            //}
            //else if (condition.Operator == 1)
            //{
            //    return attr != condition.Value.ToString();
            //}

            return false;
        }

        public static void Replace(this SearchCondition condition, XmlNode node)
        {
        }

        public static UISearchCondition Find(string name, List<UISearchCondition> uiSearchConditions)
        {
            foreach (var item in uiSearchConditions)
            {
                if (item.UniqueKey == name)
                {
                    return item;
                }
            }

            return null;
        }

        public static UISearchCombination Find(string name, List<UISearchCombination> uiSearchCombinations)
        {
            foreach (var item in uiSearchCombinations)
            {
                if (item.ID == name)
                {
                    return item;
                }
            }

            return null;
        }

        public static void SetupTree(
            SearchTree tree,
            UISearchCombination combination,
            List<UISearchCondition> uiConditions,
            List<UISearchCombination> uiCombinations)
        {
            if (uiConditions.Count == 1)
            {
                tree.Condition = uiConditions[0].Condition;
                return;
            }

            tree.SetupChildren();

            if (combination.Operator == "AND")
            {
                tree.Operation = SearchCombinationOperation.And;
            }
            else if(combination.Operator == "OR")
            {
                tree.Operation = SearchCombinationOperation.Or;
            }

            var lhsCondition = SearchUtility.Find(combination.Lhs, uiConditions);
            if (lhsCondition != null)
            {
                tree.Lhs.Condition = lhsCondition.Condition;
            }
            else
            {
                var lhsCombination = SearchUtility.Find(combination.Lhs, uiCombinations);
                SetupTree(tree.Lhs, lhsCombination, uiConditions, uiCombinations);
            }

            var rhsCondition = SearchUtility.Find(combination.Rhs, uiConditions);
            if (rhsCondition != null)
            {
                tree.Rhs.Condition = rhsCondition.Condition;
            }
            else
            {
                var rhsCombination = SearchUtility.Find(combination.Rhs, uiCombinations);
                SetupTree(tree.Rhs, rhsCombination, uiConditions, uiCombinations);
            }
        }

        public static void DoSearch(SearchArgs args, SearchProgressDialog pd)
        {
            string folder = args.Folder;
            SearchLogger logger = args.Logger;
            List<UISearchCondition> panels = args.SearchPanels;

            List<string> result = new List<string>();

            if (String.IsNullOrEmpty(folder))
            {
                logger.Log("Search folder is empty.");
                return;
            }

            if (!Directory.Exists(folder))
            {
                logger.Log("Search folder is not exist.");
                return;
            }

            if (panels.Count <= 0)
            {
                logger.Log("Search condition is empty.");
                return;
            }

            logger.LogBar();
            logger.Log("Start searching.");
            logger.LogBar();

            logger.Log("Folder = " + folder);
            logger.Log("");

            var inputFiles = SearchUtility.GetFolderFmdbs(folder);

            int num = inputFiles.Length;


            for (int i = 0; i < inputFiles.Length; i++)
            {
                var file = inputFiles[i];

                bool searchResult = SearchUtility.SearchAndReplaceEsetFile(args, file);

                if (searchResult)
                {
                    logger.Log("find = " + file);
                    logger.Log("");
                    result.Add(file);
                }

                if (pd.Canceled)
                {
                    logger.Log("処理がキャンセルされました。");
                    break;
                }

                int progress = (int)(100f * i / num);
                pd.Value = progress;
                pd.Message = string.Format("進行度 : {0,3}%", progress);
                Application.DoEvents();
            }

            logger.LogResultBar();
            foreach (var item in result)
            {
                logger.LogResult(item);
            }
            logger.LogResultBar();

            logger.LogBar();
            logger.Log("Completed.");
            logger.LogBar();
        }

        private static bool SearchAndReplaceEsetFile(SearchArgs args, string file)
        {
            string folder = args.Folder;
            SearchLogger logger = args.Logger;
            List<UISearchCondition> panels = args.SearchPanels;

            XmlDocument xml = new XmlDocument();

            try
            {
                xml.Load(file);
            }
            catch (Exception)
            {

            }

            /*
             * 「//」  で始まる xpath : ドキュメント内のすべて
             * 「/」   で始まる xpath : 絶対パス
             * 「.//」 で始まる xpath : カレントパスの下のすべて
             */

            string xpathBase = "/EmitterSetData/EmitterList/EmitterData";

            XmlNodeList emitterNodes = xml.SelectNodes(xpathBase);

            if (emitterNodes == null)
            {
                return false;
            }

            bool isFind = false;
            bool isReplaced = false;

            foreach (XmlNode emitterNode in emitterNodes)
            {
                bool find = args.Tree.Evaluate(emitterNode);
                if (find)
                {
                    isFind = true;
                }

                if (!args.IsDoReplace)
                {
                    // 置換をしない場合
                    continue;
                }

                if (!isFind)
                {
                    continue;
                }

                foreach (var replacePanel in args.ReplacePanels)
                {
                    var replaceCondition = replacePanel.Condition;

                    // プラグインの場合：
                    if (replaceCondition.Plugin != null)
                    {
                        if (!replaceCondition.Plugin.CanReplace())
                        {
                            continue;
                        }

                        if (replaceCondition.Plugin.Replace(replaceCondition.Value, emitterNode, logger))
                        {
                            isReplaced = true;
                        }
                        continue;
                    }

                    // アトリビュートの場合：
                    if (replaceCondition.IsAttribute)
                    {
                        ReplaceAttribute(replaceCondition, emitterNode, logger, ref isReplaced);
                        continue;
                    }

                    // 通常のプロパティの場合：
                    {
                        ReplaceStandard(replaceCondition, emitterNode, logger, ref isReplaced);
                        continue;
                    }
                }
            }

            if (isReplaced)
            {
                logger.Log("置換します。");
                logger.Log("");
                try
                {
                    xml.Save(file);
                }
                catch (Exception)
                {
                    logger.Log("ファイルの書き込みに失敗しました。 file = " + file);
                }
            }

            return isFind;
        }

        /// <summary>
        /// アトリビュートの置換を行います。
        /// </summary>
        private static void ReplaceAttribute(
            SearchCondition replaceCondition,
            XmlNode emitterNode,
            SearchLogger logger,
            ref bool isReplaced)
        {
            string replacePath = ".//" + replaceCondition.XPath;

            XmlNode replaceNode = emitterNode.SelectSingleNode(replacePath);

            if (replaceNode == null)
            {
                return;
            }

            logger.Log("置換前 : " + replaceNode.OuterXml);
            //replaceNode.InnerText = replaceCondition.Value.ToString();

            ((XmlElement) replaceNode).SetAttribute(replaceCondition.Attribute, replaceCondition.Value.ToString());

            logger.Log("置換後 : " + replaceNode.OuterXml);
            isReplaced = true;
        }

        /// <summary>
        /// 通常のプロパティの置換を行います。
        /// </summary>
        private static void ReplaceStandard(
            SearchCondition replaceCondition,
            XmlNode emitterNode,
            SearchLogger logger,
            ref bool isReplaced)
        {
            string replacePath = ".//" + replaceCondition.XPath;

            XmlNode replaceNode = emitterNode.SelectSingleNode(replacePath);

            if (replaceNode == null || replaceNode.NodeType != XmlNodeType.Element)
            {
                return;
            }

            logger.Log("置換前 : " + replaceNode.OuterXml);

            isReplaced = replaceCondition.Processor.Replace((XmlElement)replaceNode, replaceCondition.Value.ToString(), replaceCondition.Operator);

            logger.Log("置換後 : " + replaceNode.OuterXml);
        }

        public static bool IsValid(List<UISearchCondition> uiConditions, List<UISearchCombination> uiCombinations)
        {
            return true;
        }

        private static string[] GetFolderFmdbs(string folderPath)
        {
            IEnumerable<string> files =
                Directory.EnumerateFiles(
                folderPath, "*.eset", SearchOption.AllDirectories);

            return files.ToArray();
        }

        public static string GetID(int count)
        {
            switch (count)
            {
                case 0:
                    return "A";
                case 1:
                    return "B";
                case 2:
                    return "C";
                case 3:
                    return "D";
                case 4:
                    return "E";
                case 5:
                    return "F";
                case 6:
                    return "G";
                case 7:
                    return "H";
                case 8:
                    return "I";
                case 9:
                    return "J";
                default:
                    return "-";
            }
        }

        public static void SetupCombobox(IList<string> list, UIComboBox target)
        {
            target.Items.Clear();

            foreach (var item in list)
            {
                target.Items.Add(item);
            }

            target.SelectedIndex = 0;
        }

        public static string GetSelectedString(UIComboBox cbx)
        {
            int index = cbx.SelectedIndex;
            return cbx.Items[index].ToString();
        }

        public static SearchCondition GetSearchCondition(string label, SearchConditions conditions)
        {
            foreach (var item in conditions.Items)
            {
                if (item.Label == label)
                {
                    return item;
                }
            }

            return null;
        }

        public static UISearchCondition CreateUICondition(int count, SearchCondition condition, SearchDialog dialog)
        {
            var uiCondition = new UISearchCondition();
            uiCondition.ParentDialog = dialog;
            uiCondition.Top = uiConditionHeight * count;
            uiCondition.Label = condition.Label;
            if (condition.Processor != null)
            {
                uiCondition.ConditionValues = condition.Processor.ConditionValues;
                uiCondition.Operators = condition.Processor.Operators;
            }
            if (condition.Plugin?.Operators != null)
            {
                uiCondition.Operators = condition.Plugin.Operators;
                uiCondition.ConditionValues = condition.Plugin.ConditionValues;
            }
            uiCondition.Value = condition.Value.ToString();
            uiCondition.SelectedComparison = condition.Operator;
            uiCondition.Condition = condition.Clone() as SearchCondition;
            uiCondition.UniqueKey = SearchUtility.GetID(count);
            return uiCondition;
        }

        public static UISearchCondition CreateReplaceCondition(int count, SearchCondition condition, SearchDialog dialog)
        {
            var uiCondition = new UISearchCondition();
            uiCondition.ParentDialog = dialog;
            uiCondition.Top = uiConditionHeight * count;
            uiCondition.Label = condition.Label;
            if (condition.Processor != null)
            {
                uiCondition.ConditionValues = condition.Processor.ConditionValues;
                uiCondition.Operators = condition.Processor.ReplaceOperators;
            }
            if (condition.Plugin?.Operators != null)
            {
                uiCondition.Operators = condition.Plugin.Operators;
                uiCondition.ConditionValues = condition.Plugin.ConditionValues;
            }
            uiCondition.Value = condition.Value.ToString();
            uiCondition.SelectedComparison = condition.Operator;
            uiCondition.Condition = condition.Clone() as SearchCondition;
            uiCondition.UniqueKey = SearchUtility.GetID(count);
            uiCondition.IsReplace = true;
            return uiCondition;
        }

        public static IEnumerable<SearchCondition> CreateSearchConditions(SearchTableByDataModelMaker table)
        {
            List<SearchCondition> result = new List<SearchCondition>();

            foreach (var item in table.SearchItems)
            {
                SearchCondition condition = new SearchCondition();
                condition.Setup(item);
                result.Add(condition);
            }

            return result;
        }
    }
}
