﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Windows.Forms;
using EffectMaker.UIControls.Threading;

namespace EffectMaker.UIDialogs.MessageDialogs
{
    /// <summary>
    /// スレッドセーフにMessageBox.Showを呼び出すためのwrapperです。
    /// </summary>
    public class ThreadSafeMsgBox
    {
        /// <summary>
        /// staticコンストラクタです。
        /// </summary>
        static ThreadSafeMsgBox()
        {
            EnableMessageBox = true;
        }

        /// <summary>
        /// メッセージボックス表示の有効/無効を取得または設定します。
        /// </summary>
        public static bool EnableMessageBox { get; set; }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(string text)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result = System.Windows.Forms.MessageBox.Show(text),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="caption">表示するキャプション</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(string text, string caption)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result = System.Windows.Forms.MessageBox.Show(text, caption),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="caption">表示するキャプション</param>
        /// <param name="buttons">表示するボタンの組み合わせを示すenum値</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(
            string text,
            string caption,
            MessageBoxButtons buttons)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result =
                    System.Windows.Forms.MessageBox.Show(
                    text,
                    caption,
                    buttons),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="caption">表示するキャプション</param>
        /// <param name="buttons">表示するボタンの組み合わせを示すenum値</param>
        /// <param name="icon">表示するアイコンを示すenum値</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(
            string text,
            string caption,
            MessageBoxButtons buttons,
            MessageBoxIcon icon)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result =
                    System.Windows.Forms.MessageBox.Show(
                    text,
                    caption,
                    buttons,
                    icon),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="caption">表示するキャプション</param>
        /// <param name="buttons">表示するボタンの組み合わせを示すenum値</param>
        /// <param name="icon">表示するアイコンを示すenum値</param>
        /// <param name="defaultButton">デフォルトで選択状態になるボタンを示すenum値</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(
            string text,
            string caption,
            MessageBoxButtons buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result =
                    System.Windows.Forms.MessageBox.Show(
                    text,
                    caption,
                    buttons,
                    icon,
                    defaultButton),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="caption">表示するキャプション</param>
        /// <param name="buttons">表示するボタンの組み合わせを示すenum値</param>
        /// <param name="icon">表示するアイコンを示すenum値</param>
        /// <param name="defaultButton">デフォルトで選択状態になるボタンを示すenum値</param>
        /// <param name="options">MessageBoxの表示オプションを示すenum値</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(
            string text,
            string caption,
            MessageBoxButtons buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton,
            MessageBoxOptions options)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result =
                    System.Windows.Forms.MessageBox.Show(
                    text,
                    caption,
                    buttons,
                    icon,
                    defaultButton,
                    options),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="caption">表示するキャプション</param>
        /// <param name="buttons">表示するボタンの組み合わせを示すenum値</param>
        /// <param name="icon">表示するアイコンを示すenum値</param>
        /// <param name="defaultButton">デフォルトで選択状態になるボタンを示すenum値</param>
        /// <param name="options">MessageBoxの表示オプションを示すenum値</param>
        /// <param name="displayHelpButton">ヘルプボタンを表示するか否かを示すbool値</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult Show(
            string text,
            string caption,
            MessageBoxButtons buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton,
            MessageBoxOptions options,
            bool displayHelpButton)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result =
                    System.Windows.Forms.MessageBox.Show(
                    text,
                    caption,
                    buttons,
                    icon,
                    defaultButton,
                    options,
                    displayHelpButton),
                null);
            return result;
        }

        /// <summary>
        /// MessageBox.Show()のwrapperです。
        /// </summary>
        /// <param name="text">表示するメッセージ</param>
        /// <param name="buttons">表示するボタンの組み合わせを示すenum値</param>
        /// <param name="icon">表示するアイコンを示すenum値</param>
        /// <param name="defaultButton">デフォルトで選択状態になるボタンを示すenum値</param>
        /// <returns>MessageBox.Show()の返り値(コンテキスト未設定の場合はNone)
        /// <see cref="DialogResult"/>.</returns>
        public static DialogResult ShowEx(
            string text,
            MessageBoxButtonsEx buttons,
            MessageBoxIcon icon,
            MessageBoxDefaultButton defaultButton)
        {
            if (EnableMessageBox == false)
            {
                return DialogResult.Cancel;
            }

            MoveMainFormToTopMost();

            var result = DialogResult.None;
            UISynchronizationContextHolder.SynchronizationContext.Send(
                state => result =
                    EffectMaker.UIDialogs.MessageDialogs.MessageBox.Show(
                    text,
                    buttons,
                    icon,
                    defaultButton),
                null);
            return result;
        }

        /// <summary>
        /// MainFormを１番上に表示させる
        /// </summary>
        private static void MoveMainFormToTopMost()
        {
            Application.OpenForms["MainForm"].TopMost = true;
            Application.OpenForms["MainForm"].TopMost = false;
        }
    }

    /// <summary>
    /// メッセージボックスの表示を無効にするクラスです。
    /// </summary>
    public class DisableMessageBoxBlock : IDisposable
    {
        /// <summary>
        /// 変更前の表示状態です。
        /// </summary>
        private bool enableMessageBox;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DisableMessageBoxBlock()
        {
            this.enableMessageBox = ThreadSafeMsgBox.EnableMessageBox;

            ThreadSafeMsgBox.EnableMessageBox = false;
        }

        /// <summary>
        /// デストラクタです。
        /// </summary>
        public void Dispose()
        {
            ThreadSafeMsgBox.EnableMessageBox = this.enableMessageBox;
        }
    }
}
