﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics.CodeAnalysis;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;
using EffectMaker.UIDialogs.Properties;

namespace EffectMaker.UIDialogs.MessageDialogs
{
    /// <summary>
    /// アセットファイルのコピー先を選択するダイアログを表示します
    /// </summary>
    public partial class CopyAssetDialog : Form
    {
        /// <summary>
        /// テクスチャファイルコピー先の実体
        /// </summary>
        private string textureDestinationPath;

        /// <summary>
        /// プリミティブファイルコピー先の実体
        /// </summary>
        private string primitiveDestinationPath;

        /// <summary>
        /// コンバイナシェーダファイルコピー先の実体
        /// </summary>
        private string combinerDestinationPath;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="emitterPath">
        /// エミッタセットの保存先
        /// </param>
        /// <param name="sourceFiles">
        /// コピー元のファイルリスト
        /// </param>
        /// <param name="texDstPaths">
        /// テクスチャファイルのコピー先リスト
        /// </param>
        /// <param name="priDstPaths">
        /// プリミティブファイルのコピー先リスト
        /// </param>
        /// <param name="ecsDstPaths">
        /// コンバイナシェーダファイルのコピー先リスト
        /// </param>
        public CopyAssetDialog(
            string emitterPath,
            IEnumerable<string> sourceFiles,
            IEnumerable<string> texDstPaths,
            IEnumerable<string> priDstPaths,
            IEnumerable<string> ecsDstPaths)
        {
            this.InitializeComponent();

            // ローカライズされたテキストをセット
            this.Text = Resources.CopyAssetDialogCaption;
            this.label_description.Text = Resources.CopyAssetDialogDescription;
            this.label_source.Text = Resources.CopyAssetDialogSource;
            this.groupBox_SavePath.Text = Resources.CopyAssetDialogSavePath;
            this.label_destination.Text = Resources.CopyAssetDialogDestination;
            this.btnCopy.Text = Resources.DialogButtonCopy;
            this.btnCancel.Text = Resources.DialogButtonCancel;

            this.label_emiterPath.Text = emitterPath;

            // コピー元のファイルリストを表示
            foreach (string filepath in sourceFiles)
            {
                this.listBox_src.Items.Add(filepath);
            }

            if (texDstPaths != null)
            {
                // テクスチャ用ラベル
                var texLabel = new Label
                {
                    AutoSize = true,
                    Location = new System.Drawing.Point(0, 0),
                    Name = "label_TextureDst",
                    Size = new System.Drawing.Size(88, 16),
                    Text = Resources.CopyAssetDialogTextureDestination
                };
                this.panel_TextureDestination.Controls.Add(texLabel);

                // コレクション内のパスをラジオボタンに展開
                this.LayoutRadioButtons(texDstPaths, this.panel_TextureDestination, 2);
            }
            else
            {
                this.panel_TextureDestination.Height = 0;
            }

            // プリミティブパネルの位置を調整
            int prmY = this.panel_TextureDestination.Height + 3;
            this.panel_PrimitiveDestination.Location = new Point(0, prmY == 3 ? 0 : prmY);

            if (priDstPaths != null)
            {
                // プリミティブ用ラベル
                var priLabel = new Label
                {
                    AutoSize = true,
                    Location = new System.Drawing.Point(0, 0),
                    Name = "label_PrimitiveDst",
                    Size = new System.Drawing.Size(88, 16),
                    Text = Resources.CopyAssetDialogPrimitiveDestination
                };
                this.panel_PrimitiveDestination.Controls.Add(priLabel);

                // コレクション内のパスをラジオボタンに展開
                this.LayoutRadioButtons(priDstPaths, this.panel_PrimitiveDestination, 3);
            }
            else
            {
                this.panel_PrimitiveDestination.Height = 0;
            }

            // コンバイナシェーダパネルの位置を調整
            int cmbY = this.panel_PrimitiveDestination.Location.Y + this.panel_PrimitiveDestination.Height + 3;
            this.panel_CombinerDestination.Location = new Point(0, cmbY == 3 ? 0 : cmbY);

            if (ecsDstPaths != null)
            {
                // コンバイナ用ラベル
                var priLabel = new Label
                {
                    AutoSize = true,
                    Location = new System.Drawing.Point(0, 0),
                    Name = "label_CombinerDst",
                    Size = new System.Drawing.Size(88, 16),
                    Text = Resources.CopyAssetDialogCombinerDestination
                };
                this.panel_CombinerDestination.Controls.Add(priLabel);

                // コレクション内のパスをラジオボタンに展開
                this.LayoutRadioButtons(ecsDstPaths, this.panel_CombinerDestination, 4);
            }
        }

        /// <summary>
        /// テクスチャファイルコピー先の選択結果
        /// </summary>
        public string TextureDestnationPath
        {
            get
            {
                return this.textureDestinationPath;
            }
        }

        /// <summary>
        /// プリミティブファイルコピー先の選択結果
        /// </summary>
        public string PrimitiveDestnationPath
        {
            get
            {
                return this.primitiveDestinationPath;
            }
        }

        /// <summary>
        /// コンバイナシェーダファイルコピー先の選択結果
        /// </summary>
        public string CombinerDestnationPath
        {
            get
            {
                return this.combinerDestinationPath;
            }
        }

        /// <summary>
        /// コレクション内のパスをラジオボタンに展開する。
        /// </summary>
        /// <param name="dstPaths">
        /// 展開するパスのリスト
        /// </param>
        /// <param name="panel">
        /// 展開先のパネル
        /// </param>
        /// <param name="tabIndex">
        /// 割り当てるタブインデックス
        /// </param>
        private void LayoutRadioButtons(IEnumerable<string> dstPaths, Panel panel, int tabIndex)
        {
            int count = 0;
            foreach (string filepath in dstPaths)
            {
                var radio = new RadioButton
                {
                    AutoSize = true,
                    Location = new System.Drawing.Point(4, 16 * (count + 1)),
                    Name = "radio",
                    Size = new System.Drawing.Size(88, 16),
                    TabIndex = tabIndex,
                    TabStop = true,
                    Text = filepath,
                    UseVisualStyleBackColor = true
                };

                radio.CheckedChanged += (sender, e) =>
                {
                    var checkedRadio = (RadioButton)sender;
                    if (checkedRadio.Checked)
                    {
                        if (panel == this.panel_TextureDestination)
                        {
                            this.textureDestinationPath = checkedRadio.Text;
                        }
                        else if(panel == this.panel_PrimitiveDestination)
                        {
                            this.primitiveDestinationPath = checkedRadio.Text;
                        }
                        else
                        {
                            this.combinerDestinationPath = checkedRadio.Text;
                        }
                    }
                };

                if (count == 0)
                {
                    radio.Checked = true;
                }

                panel.Controls.Add(radio);
                count++;
            }
        }

        /// <summary>
        /// The btn copy_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void BtnCopy_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// The btn cancel_ click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void BtnCancel_Click(object sender, EventArgs e)
        {
            this.textureDestinationPath = string.Empty;
            this.primitiveDestinationPath = string.Empty;
            this.combinerDestinationPath = string.Empty;
            this.Close();
        }
    }
}
