﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIDialogs.EmbededViewerDialog
{
    /// <summary>
    /// ビューア埋め込みパネルです。
    /// </summary>
    public partial class EmbededViewerPanel : UIUserControl
    {
        /// <summary>
        /// ビューア埋め込み用のコントロールに設定するビューモデルです。
        /// </summary>
        private HwndPresenterViewModel hwndPresenterViewModel;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EmbededViewerPanel()
        {
            this.InitializeComponent();

            // グリッドのサイズのコンボボックスに 1 ～ 10 の選択肢を設定する
            {
                for (int i = 1; i <= 10; ++i)
                {
                    ComboboxItem<int> sizeItem = new ComboboxItem<int>(i.ToString(), i);
                    this.cmbFloorSize.Items.Add(sizeItem);
                }

                this.cmbFloorSize.SelectedIndex = 0;
            }

            // グリッドの分割数のコンボボックスに 1 と 10 の選択肢を設定する
            {
                ComboboxItem<int>[] divisionItems = new ComboboxItem<int>[]
                {
                    new ComboboxItem<int>("1", 1),
                    new ComboboxItem<int>("10", 10)
                };

                this.cmbFloorDivisionNumber.Items.AddRange(divisionItems);
                this.cmbFloorDivisionNumber.SelectedIndex = 0;
            }

            // グレアの強さのコンボボックスに 0 ～ 6 の選択肢を設定する
            {
                for (int i = 0; i <= 6; ++i)
                {
                    ComboboxItem<int> glareItem = new ComboboxItem<int>(i.ToString(), i);
                    this.cmbGlareLevel.Items.Add(glareItem);
                }

                this.cmbGlareLevel.SelectedIndex = 0;
            }

            // ビューア埋め込み用のコントロールを作成
            {
                HwndPresenderControl control = new HwndPresenderControl();

                this.hwndPresenterViewModel = new HwndPresenterViewModel();
                control.DataContext = this.hwndPresenterViewModel;

                this.elementHost.Child = control;
            }

            // ツールバー向けのバインディングを追加
            this.AddBinding(nameof(this.FloorType), "FloorType").IssueCommand = false;  // FloorType については ViewModel 側でプロパティが実装されているためコマンドの発行は不要
            this.AddBinding(nameof(this.FloorSize), "GridSize").IssueCommand = true;
            this.AddBinding(nameof(this.FloorDivisionNumber), "GridDivisionNumber").IssueCommand = true;
            this.AddBinding(nameof(this.EnabledFloorSize), "EnabledGridSize");
            this.AddBinding(nameof(this.EnabledFloorDivisionNumber), "EnabledGridDivisionNumber");
            this.AddBinding(nameof(this.GlareLevel), "GlareLevel").IssueCommand = true;
            this.AddBinding(nameof(this.DisplayEmitterShape), "DisplayParentEmitterShape").IssueCommand = true;
            this.AddBinding(nameof(this.DisplayChildEmitterShape), "DisplayChildEmitterShape").IssueCommand = true;
            this.AddBinding(nameof(this.DisplayFieldShape), "DisplayFieldShape").IssueCommand = true;
            this.AddBinding(nameof(this.IsOverDrawMode), "IsOverDrawMode").IssueCommand = true;
            this.AddBinding(nameof(this.DisplayDebugInformation), "DisplayDebugInformation").IssueCommand = true;
        }

        #region Binding Properties

        /// <summary>
        /// FloorType プロパティのバッキングフィールドです。
        /// </summary>
        private int floorType;

        /// <summary>
        /// 床の表示タイプを取得または設定します。
        /// </summary>
        public int FloorType
        {
            get
            {
                return this.floorType;
            }

            set
            {
                if (value != this.floorType)
                {
                    this.floorType = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();

                    this.UpdateFloorTypeButtonImage();
                }
            }
        }

        /// <summary>
        /// 床の種類の選択ボタンの画像を更新します。
        /// </summary>
        private void UpdateFloorTypeButtonImage()
        {
            Bitmap image = null;

            switch (this.FloorType)
            {
                case 0:
                    image = Properties.Resources.Icon_FloorDisplay_Off;
                    break;
                case 1:
                case 2:
                    image = Properties.Resources.Icon_FloorDisplay_Grid;
                    break;
                case 3:
                    image = Properties.Resources.Icon_FloorDisplay_Block;
                    break;
            }

            this.btnFloorType.Image = image;
        }

        /// <summary>
        /// 床のサイズを取得または設定します。
        /// </summary>
        public int FloorSize
        {
            get
            {
                ComboboxItem<int> selectedItem = this.cmbFloorSize.SelectedItem as ComboboxItem<int>;
                return selectedItem.Value;;
            }

            set
            {
                ComboboxItem<int> selectedItem = this.cmbFloorSize.SelectedItem as ComboboxItem<int>;

                if (selectedItem == null || value != selectedItem.Value)
                {
                    ComboboxItem<int> newItem = this.cmbFloorSize.Items.Cast<ComboboxItem<int>>().FirstOrDefault(x => x.Value == value);

                    if (newItem != null)
                    {
                        this.cmbFloorSize.SelectedItem = newItem;
                        this.LogicalTreeElementExtender.NotifyPropertyChanged();
                    }
                }
            }
        }

        /// <summary>
        /// 床のサイズが有効かどうか取得または設定します。
        /// </summary>
        public bool EnabledFloorSize
        {
            get
            {
                return this.cmbFloorSize.Enabled;
            }

            set
            {
                this.lblFloorSize.Enabled = value;
                this.cmbFloorSize.Enabled = value;
            }
        }

        /// <summary>
        /// 床の分割数を取得または設定します。
        /// </summary>
        public int FloorDivisionNumber
        {
            get
            {
                ComboboxItem<int> selectedItem = this.cmbFloorDivisionNumber.SelectedItem as ComboboxItem<int>;
                return selectedItem.Value;
            }

            set
            {
                ComboboxItem<int> selectedItem = this.cmbFloorDivisionNumber.SelectedItem as ComboboxItem<int>;

                if (selectedItem == null || value != selectedItem.Value)
                {
                    ComboboxItem<int> newItem = this.cmbFloorDivisionNumber.Items.Cast<ComboboxItem<int>>().FirstOrDefault(x => x.Value == value);

                    if (newItem != null)
                    {
                        this.cmbFloorDivisionNumber.SelectedItem = newItem;
                        this.LogicalTreeElementExtender.NotifyPropertyChanged();
                    }
                }
            }
        }

        /// <summary>
        /// 床の分割数が有効かどうか取得または設定します。
        /// </summary>
        public bool EnabledFloorDivisionNumber
        {
            get
            {
                return this.cmbFloorDivisionNumber.Enabled;
            }

            set
            {
                this.lblFloorDivisionNumber.Enabled = value;
                this.cmbFloorDivisionNumber.Enabled = value;
            }
        }

        /// <summary>
        /// グレアの強さを取得または設定します。
        /// </summary>
        public int GlareLevel
        {
            get
            {
                ComboboxItem<int> selectedItem = this.cmbGlareLevel.SelectedItem as ComboboxItem<int>;
                return selectedItem.Value;
            }

            set
            {
                ComboboxItem<int> selectedItem = this.cmbGlareLevel.SelectedItem as ComboboxItem<int>;

                if (selectedItem == null || value != selectedItem.Value)
                {
                    ComboboxItem<int> newItem = this.cmbGlareLevel.Items.Cast<ComboboxItem<int>>().FirstOrDefault(x => x.Value == value);

                    if (newItem != null)
                    {
                        this.cmbGlareLevel.SelectedItem = newItem;
                        this.LogicalTreeElementExtender.NotifyPropertyChanged();
                    }
                }
            }
        }

        /// <summary>
        /// エミッタ形状表示の On/Off を取得または設定します。
        /// </summary>
        public bool DisplayEmitterShape
        {
            get
            {
                return this.btnDisplayEmitterShape.Checked;
            }

            set
            {
                if (value != this.btnDisplayEmitterShape.Checked)
                {
                    this.btnDisplayEmitterShape.Checked = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();
                }
            }
        }

        /// <summary>
        /// チャイルドエミッタ形状表示の On/Off を取得または設定します。
        /// </summary>
        public bool DisplayChildEmitterShape
        {
            get
            {
                return this.btnDisplayChildEmitterShape.Checked;
            }

            set
            {
                if (value != this.btnDisplayChildEmitterShape.Checked)
                {
                    this.btnDisplayChildEmitterShape.Checked = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();
                }
            }
        }

        /// <summary>
        /// フィールド形状表示の On/Off を取得または設定します。
        /// </summary>
        public bool DisplayFieldShape
        {
            get
            {
                return this.btnDisplayFieldShape.Checked;
            }

            set
            {
                if (value != this.btnDisplayFieldShape.Checked)
                {
                    this.btnDisplayFieldShape.Checked = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();
                }
            }
        }

        /// <summary>
        /// IsOverDrawMode プロパティのバッキングフィールドです。
        /// </summary>
        private bool isOverDrawMode;

        /// <summary>
        /// パーティクルの重なりチェックの On/Off を取得または設定します。
        /// </summary>
        public bool IsOverDrawMode
        {
            get
            {
                return this.isOverDrawMode;
            }

            set
            {
                if (value != this.isOverDrawMode)
                {
                    this.isOverDrawMode = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();

                    this.btnOverDrawMode.Image = this.isOverDrawMode ?
                        Properties.Resources.Icon_LapCheck_On :
                        Properties.Resources.Icon_LapCheck_Off;
                }
            }
        }

        /// <summary>
        /// displayDebugInformation プロパティのバッキングフィールドです。
        /// </summary>
        private bool displayDebugInformation;

        /// <summary>
        /// デバッグ情報表示の On/Off を取得または設定します。
        /// </summary>
        public bool DisplayDebugInformation
        {
            get
            {
                return this.displayDebugInformation;
            }

            set
            {
                if (value != this.displayDebugInformation)
                {
                    this.displayDebugInformation = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();

                    this.btnDisplayDebugInformation.Image = this.displayDebugInformation ?
                        Properties.Resources.Icon_DebugInfo_On :
                        Properties.Resources.Icon_DebugInfo_Off;
                }
            }
        }

        #endregion

        #region Toolbar Event Methods

        /// <summary>
        /// 「床を非表示」メニューが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void mniFloorHidden_Click(object sender, EventArgs e)
        {
            this.FloorType = 0;
        }

        /// <summary>
        /// 「床をグリッドで表示」メニューが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void mniFloorGrid_Click(object sender, EventArgs e)
        {
            this.FloorType = 1;
        }

        /// <summary>
        /// 「床をブロックで表示」メニューが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void mniFloorBlock_Click(object sender, EventArgs e)
        {
            this.FloorType = 3;
        }

        /// <summary>
        /// 「床のサイズ」コンボボックスの選択が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void cmbFloorSize_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, nameof(this.FloorSize));
        }

        /// <summary>
        /// 「床の分割数」コンボボックスの選択が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void cmbFloorDivisionNumber_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, nameof(this.FloorDivisionNumber));
        }

        /// <summary>
        /// 「グレアの強さ」コンボボックスの選択が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void cmbGlareLevel_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, nameof(this.GlareLevel));
        }

        /// <summary>
        /// 「エミッタを表示」メニューのチェック状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void btnDisplayEmitterShape_CheckedChanged(object sender, EventArgs e)
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, nameof(this.DisplayEmitterShape));
        }

        /// <summary>
        /// 「チャイルドエミッタを表示」メニューのチェック状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void btnDisplayChildEmitterShape_CheckedChanged(object sender, EventArgs e)
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, nameof(this.DisplayChildEmitterShape));
        }

        /// <summary>
        /// 「フィールドを表示」メニューのチェック状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void btnDisplayFieldShape_CheckedChanged(object sender, EventArgs e)
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged,nameof(this.DisplayFieldShape));
        }

        /// <summary>
        /// 「パーティクルの重なりチェック」ボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void btnOverDrawMode_Click(object sender, EventArgs e)
        {
            this.IsOverDrawMode = !this.IsOverDrawMode;
        }

        /// <summary>
        /// 「デバッグ情報の表示」ボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void btnDisplayDebugInformation_Click(object sender, EventArgs e)
        {
            this.DisplayDebugInformation = !this.DisplayDebugInformation;
        }

        /// <summary>
        /// 「ヘルプ」ボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void btnHelp_Click(object sender, EventArgs e)
        {
            UILogic.Manager.HelpManager.ShowHelp("92121926");
        }

        #endregion

        /// <summary>
        /// 埋め込むビューアのプロセスを取得または設定します。
        /// </summary>
        public Process ViewerProcess
        {
            get { return this.hwndPresenterViewModel.ViewerProcess; }
            set { this.hwndPresenterViewModel.ViewerProcess = value; }
        }

        /// <summary>
        /// 埋め込まれたビューアのプロセスを停止します。
        /// </summary>
        public void KillViewer()
        {
            this.hwndPresenterViewModel.KillViewer();
        }

        #region ComboboxItem Class

        /// <summary>
        /// コンボボックスアイテムです。
        /// </summary>
        /// <typeparam name="T">値の型</typeparam>
        private class ComboboxItem<T>
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="key">キー</param>
            /// <param name="value">値</param>
            public ComboboxItem(string key, T value)
            {
                this.Key = key;
                this.Value = value;
            }

            /// <summary>
            /// キーを取得します。
            /// </summary>
            public string Key { get; private set; }

            /// <summary>
            /// 値を取得します。
            /// </summary>
            public T Value { get; private set; }

            /// <summary>
            /// 現在のオブジェクトを表す文字列を返します。
            /// この値がコンボボックスのドロップダウンリストに表示されるため、キーを返します。
            /// </summary>
            /// <returns>現在のオブジェクトを表す文字列を返します。</returns>
            public override string ToString()
            {
                return this.Key;
            }
        }

        #endregion
    }
}
