﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using System.Xaml;

using EffectMaker.Foundation.Log;

namespace EffectMaker.UIControls.Xaml
{
    /// <summary>
    /// A loader that manages the XAML engine.
    /// </summary>
    public static class XamlLoader
    {
        /// <summary>
        /// Stores the SynchronizationContext instance used to
        /// marshal threaded code to the proper thread.
        /// </summary>
        private static SynchronizationContext synchContext;

        /// <summary>
        /// Sets the SynchronizationContext instance used to
        /// marshal threaded code to the proper thread.
        /// </summary>
        /// <param name="synchronizationContext">The SynchronizationContext instance used to
        /// marshal threaded code to the proper thread.</param>
        public static void SetSynchronizationContext(
            SynchronizationContext synchronizationContext)
        {
            synchContext = synchronizationContext;
        }

        /// <summary>
        /// Load XAML from a string.
        /// </summary>
        /// <param name="content">The string that contains the XAML to load.</param>
        /// <param name="encoding">Encoding of the content.
        /// If this parameter is null, then UTF8 is used as default encoding.</param>
        /// <returns>Returns the root control defined by the XAML, or null if failed.</returns>
        public static Control Parse(string content, Encoding encoding = null)
        {
            if (string.IsNullOrWhiteSpace(content))
            {
                throw new ArgumentException("content");
            }

            encoding = encoding ?? Encoding.UTF8;

            var data = encoding.GetBytes(content);
            var ms = new MemoryStream(data, false);

            return Load(ms);
        }

        /// <summary>
        /// Load XAML from a stream.
        /// </summary>
        /// <param name="xamlContent">The stream that contains the XAML to load.</param>
        /// <param name="sourceAssembly">The assembly in which the
        /// XAML resource is located.</param>
        /// <returns>Returns the root control defined by the XAML, or null if failed.</returns>
        public static Control Load(Stream xamlContent, Assembly sourceAssembly = null)
        {
            if (sourceAssembly == null)
            {
                sourceAssembly = Assembly.GetCallingAssembly();
            }

            var xamlSchemaContext = new EffectMakerXamlSchemaContext(sourceAssembly);

            var xamlReader = new XamlXmlReader(xamlContent, xamlSchemaContext);
            return LoadInternal(xamlReader);
        }

        /// <summary>
        /// Load XAML from a file.
        /// </summary>
        /// <param name="xamlFilename">The file that contains the XAML to load.</param>
        /// <returns>Returns the root control defined by the XAML, or null if failed.</returns>
        public static Control Load(string xamlFilename)
        {
            var xamlSchemaContext = new EffectMakerXamlSchemaContext(
                Assembly.GetCallingAssembly());

            var xamlReader = new XamlXmlReader(xamlFilename, xamlSchemaContext);
            return LoadInternal(xamlReader);
        }

        /// <summary>
        /// Loads the XAML content from XamlReader.
        /// </summary>
        /// <param name="xamlReader">The XamlReader representing the XAML content.</param>
        /// <returns>Returns a loaded control, or null if failed.</returns>
        private static Control LoadInternal(XamlReader xamlReader)
        {
            var xamlWriter = new ThreadSafeXamlObjectWriter(
                synchContext,
                xamlReader.SchemaContext as EffectMakerXamlSchemaContext);

            try
            {
                XamlServices.Transform(xamlReader, xamlWriter, true);
            }
            catch (Exception e)
            {
                Logger.Log(LogLevels.Error, "XamlLoader.LoadInternal : Exception occurred, {0}", e.Message);
            }

            return xamlWriter.Result as Control;
        }
    }
}
