﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Xaml;
using EffectMaker.Foundation.Extensions;

namespace EffectMaker.UIControls.Xaml
{
    /// <summary>
    /// A thread-safe XamlObjectWriter wrapper.
    /// </summary>
    public class ThreadSafeXamlObjectWriter : XamlObjectWriter
    {
        /// <summary>
        /// Stores the SynchronizationContext instance used to marshal
        /// threaded code to the proper thread.
        /// </summary>
        private SynchronizationContext synchronizationContext;

        /// <summary>
        /// Initializes the ThreadSafeXamlObjectWriter instance.
        /// </summary>
        /// <param name="synchronizationContext">The SynchronizationContext instance used to
        /// marshal threaded code to the proper thread.</param>
        /// <param name="schemaContext">XamlSchemaContext instance provided
        /// by the XAML engine.</param>
        public ThreadSafeXamlObjectWriter(
            SynchronizationContext synchronizationContext,
            XamlSchemaContext schemaContext)
            : base(schemaContext)
        {
            this.synchronizationContext = synchronizationContext;
        }

        /// <summary>
        /// WriteNamespace: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="namespaceDeclaration">namespaceDeclaration: See XamlObjectWriter
        /// documentation for more information.</param>
        public override void WriteNamespace(NamespaceDeclaration namespaceDeclaration)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteNamespace(namespaceDeclaration));
            }
            else
            {
                base.WriteNamespace(namespaceDeclaration);
            }
        }

        /// <summary>
        /// WriteStartObject: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="xamlType">xamlType: See XamlObjectWriter
        /// documentation for more information.</param>
        public override void WriteStartObject(XamlType xamlType)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteStartObject(xamlType));
            }
            else
            {
                base.WriteStartObject(xamlType);
            }
        }

        /// <summary>
        /// WriteStartMember: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="property">property: See XamlObjectWriter
        /// documentation for more information.</param>
        public override void WriteStartMember(XamlMember property)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteStartMember(property));
            }
            else
            {
                base.WriteStartMember(property);
            }
        }

        /// <summary>
        /// WriteGetObject: See XamlObjectWriter documentation for more information.
        /// </summary>
        public override void WriteGetObject()
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteGetObject());
            }
            else
            {
                base.WriteGetObject();
            }
        }

        /// <summary>
        /// WriteValue: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="value">See XamlObjectWriter documentation for more information.</param>
        public override void WriteValue(object value)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteValue(value));
            }
            else
            {
                base.WriteValue(value);
            }
        }

        /// <summary>
        /// WriteEndMember: See XamlObjectWriter documentation for more information.
        /// </summary>
        public override void WriteEndMember()
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteEndMember());
            }
            else
            {
                base.WriteEndMember();
            }
        }

        /// <summary>
        /// WriteEndObject: See XamlObjectWriter documentation for more information.
        /// </summary>
        public override void WriteEndObject()
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.WriteEndObject());
            }
            else
            {
                base.WriteEndObject();
            }
        }

        /// <summary>
        /// OnAfterBeginInit: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="value">See XamlObjectWriter documentation for more information.</param>
        protected override void OnAfterBeginInit(object value)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.OnAfterBeginInit(value));
            }
            else
            {
                base.OnAfterBeginInit(value);
            }
        }

        /// <summary>
        /// OnAfterEndInit: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="value">See XamlObjectWriter documentation for more information.</param>
        protected override void OnAfterEndInit(object value)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.OnAfterEndInit(value));
            }
            else
            {
                base.OnAfterEndInit(value);
            }
        }

        /// <summary>
        /// OnBeforeProperties: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="value">See XamlObjectWriter documentation for more information.</param>
        protected override void OnBeforeProperties(object value)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.OnBeforeProperties(value));
            }
            else
            {
                base.OnBeforeProperties(value);
            }
        }

        /// <summary>
        /// OnAfterProperties: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="value">See XamlObjectWriter documentation for more information.</param>
        protected override void OnAfterProperties(object value)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.OnAfterProperties(value));
            }
            else
            {
                base.OnAfterProperties(value);
            }
        }

        /// <summary>
        /// OnSetValue: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="eventSender">eventSender: See XamlObjectWriter
        /// documentation for more information.</param>
        /// <param name="member">member: See XamlObjectWriter
        /// documentation for more information.</param>
        /// <param name="value">value: See XamlObjectWriter
        /// documentation for more information.</param>
        /// <returns>return: See XamlObjectWriter documentation for more information.</returns>
        protected override bool OnSetValue(object eventSender, XamlMember member, object value)
        {
            if (this.synchronizationContext != null)
            {
                return this.synchronizationContext.ProduceSafely(
                    () => base.OnSetValue(eventSender, member, value));
            }
            else
            {
                return base.OnSetValue(eventSender, member, value);
            }
        }

        /// <summary>
        /// Dispose: See XamlObjectWriter documentation for more information.
        /// </summary>
        /// <param name="disposing">disposing: See XamlObjectWriter
        /// documentation for more information.</param>
        protected override void Dispose(bool disposing)
        {
            if (this.synchronizationContext != null)
            {
                this.synchronizationContext.Send(() => base.Dispose(disposing));
            }
            else
            {
                base.Dispose(disposing);
            }
        }
    }
}
