﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Markup;
using System.Xaml;

namespace EffectMaker.UIControls.Xaml
{
    /// <summary>
    /// Represent a resource access.
    /// </summary>
    public class Resource : System.Windows.Markup.MarkupExtension
    {
        /// <summary>
        /// Stores the name of the resource to access.
        /// </summary>
        private string resourceName;

        /// <summary>
        /// Initializes the Resource instance.
        /// </summary>
        /// <param name="resourceName">The name of the resource to access.</param>
        public Resource(string resourceName)
        {
            this.resourceName = resourceName;
        }

        /// <summary>
        /// Provides a resource value.
        /// </summary>
        /// <param name="serviceProvider">ServiceProvider given by the XAML engine.</param>
        /// <returns>Returns a resource value, or null if failed to obtain.</returns>
        public override object ProvideValue(IServiceProvider serviceProvider)
        {
            IControl control = this.GetParentInstance(serviceProvider);

            if (control == null)
            {
                control = this.GetGrandParentInstance(serviceProvider);
            }

            bool found = false;

            if (control != null)
            {
                object value = this.GetResourceValue(control, out found);
                if (found)
                {
                    return value;
                }
            }

            control = this.GetRootInstance(serviceProvider);
            if (control != null)
            {
                return this.GetResourceValue(control, out found);
            }

            return null;
        }

        /// <summary>
        /// Gets the resource value for a given control.
        /// </summary>
        /// <param name="control">The control to lookup the resources.</param>
        /// <param name="found">Tells whether the resource as been found or not.</param>
        /// <returns>Returns the resource value, or null if failed to obtain.</returns>
        private object GetResourceValue(IControl control, out bool found)
        {
            if (control == null)
            {
                found = false;
                return null;
            }

            object value;

            if (control.Resources.TryGetValue(this.resourceName, out value))
            {
                found = true;
                return value;
            }

            return this.GetResourceValue(control.Parent as IControl, out found);
        }

        /// <summary>
        /// Gets the root instance of the XAML tree.
        /// </summary>
        /// <param name="serviceProvider">ServiceProvider given by the XAML engine.</param>
        /// <returns>Returns the root instance, or null if not available.</returns>
        private IControl GetRootInstance(IServiceProvider serviceProvider)
        {
            var rootObjectProvider = (IRootObjectProvider)serviceProvider.GetService(
                typeof(IRootObjectProvider));

            return rootObjectProvider.RootObject as IControl;
        }

        /// <summary>
        /// Gets the parent instance of the current Resource instance.
        /// </summary>
        /// <param name="serviceProvider">ServiceProvider given by the XAML engine.</param>
        /// <returns>Returns the parent instance, or null if not available.</returns>
        private IControl GetParentInstance(IServiceProvider serviceProvider)
        {
            var provideValueTarget = (IProvideValueTarget)serviceProvider.GetService(
                typeof(IProvideValueTarget));

            return provideValueTarget.TargetObject as IControl;
        }

        /// <summary>
        /// Gets the grand-parent instance of the current Resource instance.
        /// </summary>
        /// <param name="serviceProvider">ServiceProvider given by the XAML engine.</param>
        /// <returns>Returns the grand-parent instance, or null if not available.</returns>
        private IControl GetGrandParentInstance(IServiceProvider serviceProvider)
        {
            IXamlSchemaContextProvider service = serviceProvider.GetService(
                typeof(IXamlSchemaContextProvider)) as IXamlSchemaContextProvider;
            if (service == null)
            {
                return null;
            }

            FieldInfo field = service.GetType().GetField(
                "_xamlContext",
                BindingFlags.NonPublic | BindingFlags.Instance);
            if (field == null)
            {
                return null;
            }

            object fieldValue = field.GetValue(service);
            if (fieldValue == null)
            {
                return null;
            }

            PropertyInfo property = fieldValue.GetType().GetProperty(
                "GrandParentInstance");
            if (property == null)
            {
                return null;
            }

            return property.GetValue(fieldValue) as IControl;
        }
    }
}
