﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Reflection;
using System.Xaml;

namespace EffectMaker.UIControls.Xaml
{
    /// <summary>
    /// An extended XamlSchemaContext class.
    /// </summary>
    public class EffectMakerXamlSchemaContext : XamlSchemaContext
    {
        /// <summary>
        /// Cache for types already made XamlType.
        /// </summary>
        private Dictionary<Type, EffectMakerXamlType> convertedTypes =
            new Dictionary<Type, EffectMakerXamlType>();

        /// <summary>
        /// Initializes the EffectMakerXamlSchemaContext instance.
        /// </summary>
        /// <param name="sourceAssembly">The assembly in which the XAML file
        /// is set as embedded resource.</param>
        public EffectMakerXamlSchemaContext(Assembly sourceAssembly)
        {
            this.SourceAssembly = sourceAssembly;
        }

        /// <summary>
        /// Gets the assembly in which the XAML file is defined.
        /// </summary>
        public Assembly SourceAssembly { get; private set; }

        /// <summary>
        /// Provides a XamlType instance.
        /// </summary>
        /// <param name="type">The related runtime type.</param>
        /// <returns>Returns an EffectMakerXamlType if type is of type IControl,
        /// or the result of the base implementation otherwise.</returns>
        public override XamlType GetXamlType(Type type)
        {
            EffectMakerXamlType result;

            if (typeof(IBindable).IsAssignableFrom(type) == false)
            {
                return base.GetXamlType(type);
            }

            if (this.convertedTypes.TryGetValue(type, out result) == false)
            {
                result = new EffectMakerXamlType(type, this);
                this.convertedTypes.Add(type, result);
            }

            return result;
        }
    }
}
