﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Forms.Layout;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// An extended LayoutEngine with stacking capabilities.
    /// </summary>
    public class StackLayoutEngine : LayoutEngineBase
    {
        /// <summary>
        /// Backing field for Orientation property.
        /// </summary>
        private Orientation orientation = Orientation.Vertical;

        /// <summary>
        /// 前回計算時のコンテナサイズ.
        /// </summary>
        private Size cachedSize = new Size(-1, -1);

        /// <summary>
        /// Gets or sets the orientation of the control.
        /// </summary>
        public Orientation Orientation
        {
            get
            {
                return this.orientation;
            }

            set
            {
                if (this.orientation != value)
                {
                    this.orientation = value;
                    this.OnRequestLayout(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Perform the layout.
        /// </summary>
        /// <param name="container">Container control.</param>
        /// <param name="containerParentSize">Size of the parent of the current container.</param>
        /// <param name="layoutEventArgs">Layout event argument.</param>
        /// <returns>Returns true if the parent must perform layout too,
        /// false otherwise.</returns>
        protected override bool OnLayout(
            Control container,
            Size containerParentSize,
            LayoutEventArgs layoutEventArgs)
        {
            // Use DisplayRectangle so that parent.Padding is honored
            Rectangle parentDisplayRectangle = container.DisplayRectangle;

            return this.ArrangeControls(container, parentDisplayRectangle.Size);
        }

        /// <summary>
        /// Arrange controls in a stack layout.
        /// </summary>
        /// <param name="container">Container control.</param>
        /// <param name="availableSize">Container size.</param>
        /// <returns>Returns true if the parent must perform layout too,
        /// false otherwise.</returns>
        private bool ArrangeControls(Control container, Size availableSize)
        {
            Size containerSize = new Size();

            if (this.Orientation == Orientation.Horizontal)
            {
                int left = container.Padding.Left;

                foreach (ILayoutElement child in container.GetNonCollapsedControls())
                {
                    Size size = child.GetElementDisplaySize(availableSize);

                    var box = new Rectangle(
                        left,
                        container.Padding.Top,
                        size.Width,
                        availableSize.Height);

                    child.ArrangeLayoutElement(box);

                    size = child.GetElementDisplaySize(availableSize);

                    left += size.Width;

                    // コンテナのサイズを計算
                    if (size.Height > containerSize.Height)
                    {
                        containerSize.Height = size.Height;
                    }

                    containerSize.Width = left;
                }
            }
            else if (this.Orientation == Orientation.Vertical)
            {
                int top = container.Padding.Top;

                foreach (ILayoutElement child in container.GetNonCollapsedControls())
                {
                    Size size = child.GetElementDisplaySize(availableSize);

                    var box = new Rectangle(
                        container.Padding.Left,
                        top,
                        availableSize.Width,
                        size.Height);

                    child.ArrangeLayoutElement(box);

                    size = child.GetElementDisplaySize(availableSize);

                    top += size.Height;

                    // コンテナのサイズを計算
                    if (size.Width > containerSize.Width)
                    {
                        containerSize.Width = size.Width;
                    }

                    containerSize.Height = top;
                }
            }
            else
            {
                throw new ArgumentException("Invalid 'Orientation' property.", "Orientation");
            }

            // コンテナのサイズが変わったときだけtrueを返す
            bool updateParent = false;

            if (containerSize != this.cachedSize)
            {
                this.cachedSize = containerSize;
                updateParent = true;
            }

            return updateParent;
        }
    }
}
