﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Windows.Forms.Layout;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// Type of grid cell sizing.
    /// </summary>
    public enum CellSizeType
    {
        /// <summary>
        /// A fixed amount of pixels.
        /// </summary>
        Fixed,

        /// <summary>
        /// The less size possible.
        /// </summary>
        Auto,

        /// <summary>
        /// The maximum size possible, dispatched among the other stared-size columns.
        /// </summary>
        Star,
    }

    /// <summary>
    /// Represent the sizing of a column or a row.
    /// </summary>
    [TypeConverter(typeof(CellSizeTypeConverter))]
    public class CellSize
    {
        /// <summary>
        /// Initializes the CellSize instance with Auto sizing.
        /// </summary>
        public CellSize()
        {
            this.Type = CellSizeType.Auto;
        }

        /// <summary>
        /// Initializes the CellSize instance.
        /// </summary>
        /// <param name="type">Sizing type. If Auto is used, then value is ignored.</param>
        /// <param name="value">The value related to the sizing type.
        /// Effective only for Fixed or Star.</param>
        public CellSize(CellSizeType type, int value)
        {
            this.Type = type;
            this.Value = value;
        }

        /// <summary>
        /// Gets the value related to the sizing type.
        /// </summary>
        public int Value { get; private set; }

        /// <summary>
        /// Gets the sizing type.
        /// </summary>
        public CellSizeType Type { get; private set; }

        /// <summary>
        /// Provides textual representation.
        /// </summary>
        /// <returns>Returns textual representation of the CellSize.</returns>
        public override string ToString()
        {
            switch (this.Type)
            {
                case CellSizeType.Auto: return "Auto";
                case CellSizeType.Fixed: return string.Format("{0}px", this.Value);
                case CellSizeType.Star: return string.Format("{0}*", this.Value);
            }

            return base.ToString();
        }
    }

    /// <summary>
    /// Type converter to transform string sizing representation to a CellSize instance.
    /// </summary>
    public class CellSizeTypeConverter : TypeConverter
    {
        /// <summary>
        /// Tells whether the source type can be converted to a CellSize type.
        /// </summary>
        /// <param name="context">The type descriptor context.</param>
        /// <param name="sourceType">The type to convert from.</param>
        /// <returns>Returns true if the conversion can be done, false otherwise.</returns>
        public override bool CanConvertFrom(ITypeDescriptorContext context, Type sourceType)
        {
            if (sourceType == typeof(string))
            {
                return true;
            }

            return base.CanConvertFrom(context, sourceType);
        }

        /// <summary>
        /// Converts a value from a given type to an instance of CellSize.
        /// </summary>
        /// <param name="context">The type descriptor context.</param>
        /// <param name="culture">The culture used for conversion.</param>
        /// <param name="value">The source value to convert from.</param>
        /// <returns>Returns a CellSize instance. Throws an exception if it fails.</returns>
        public override object ConvertFrom(
            ITypeDescriptorContext context,
            CultureInfo culture,
            object value)
        {
            if ((value is string) == false)
            {
                return base.ConvertFrom(context, culture, value);
            }

            var strValue = (string)value;
            if (string.IsNullOrWhiteSpace(strValue))
            {
                throw new Exception("Invalid CellSize format"); // TODO: Make it localizable.
            }

            strValue = strValue.Trim();

            if (string.Equals(strValue, "Auto", StringComparison.OrdinalIgnoreCase))
            {
                return new CellSize();
            }
            else
            {
                var cellSizeType = CellSizeType.Fixed;

                if (strValue.EndsWith("*"))
                {
                    // set cell size type to star.
                    cellSizeType = CellSizeType.Star;

                    // remove the ending star symbol
                    strValue = strValue.Substring(0, strValue.Length - 1);
                    if (strValue.Length == 0)
                    {
                        return new CellSize(CellSizeType.Star, 1);
                    }
                }

                int intValue;
                if (int.TryParse(strValue, out intValue) == false || intValue < 0)
                {
                    throw new Exception("Invalid CellSize format"); // TODO: Make it localizable.
                }

                return new CellSize(cellSizeType, intValue);
            }
        }
    }

    /// <summary>
    /// Represent the definition of a grid row.
    /// </summary>
    public class RowDefinition
    {
        /// <summary>
        /// Initializes the RowDefinition instance.
        /// </summary>
        public RowDefinition()
        {
            this.Height = new CellSize();
        }

        /// <summary>
        /// Gets or sets the height of the row.
        /// </summary>
        public CellSize Height { get; set; }

        /// <summary>
        /// Provides a textual representation of the RowDefinition.
        /// </summary>
        /// <returns>Returns a textual representation.</returns>
        public override string ToString()
        {
            return string.Format("Height: {0}", this.Height);
        }
    }

    /// <summary>
    /// Represent the definition of a grid column.
    /// </summary>
    public class ColumnDefinition
    {
        /// <summary>
        /// Initializes the ColumnDefinition instance.
        /// </summary>
        public ColumnDefinition()
        {
            this.Width = new CellSize();
        }

        /// <summary>
        /// Gets or sets the width of the row.
        /// </summary>
        public CellSize Width { get; set; }

        /// <summary>
        /// Provides a textual representation of the ColumnDefinition.
        /// </summary>
        /// <returns>Returns a textual representation.</returns>
        public override string ToString()
        {
            return string.Format("Width: {0}", this.Width);
        }
    }

    /// <summary>
    /// Represent the definition of a grid cell.
    /// </summary>
    public class CellInfoBase
    {
        /// <summary>
        /// Initializes the CellInfoBase instance.
        /// </summary>
        public CellInfoBase()
        {
            this.RowSpan = 1;
            this.ColumnSpan = 1;
        }

        /// <summary>
        /// Initializes the CellInfoBase instance using values from another one.
        /// </summary>
        /// <param name="source">The source CellInfoBase to copy values from.</param>
        public CellInfoBase(CellInfoBase source)
        {
            if (source == null)
            {
                throw new ArgumentNullException("source");
            }

            this.Column = source.Column;
            this.Row = source.Row;
            this.ColumnSpan = source.ColumnSpan;
            this.RowSpan = source.RowSpan;
        }

        /// <summary>
        /// Gets or sets the row number.
        /// </summary>
        public int Row { get; set; }

        /// <summary>
        /// Gets or sets the column number.
        /// </summary>
        public int Column { get; set; }

        /// <summary>
        /// Gets or sets the number of rows to span.
        /// </summary>
        public int RowSpan { get; set; }

        /// <summary>
        /// Gets or sets the number of columns to span.
        /// </summary>
        public int ColumnSpan { get; set; }

        /// <summary>
        /// Gets whether the CellInfoBase contains valid information or not.
        /// </summary>
        public virtual bool IsValid
        {
            get
            {
                return this.Row >= 0 && this.Column >= 0 &&
                    this.RowSpan >= 1 && this.ColumnSpan >= 1;
            }
        }
    }

    /// <summary>
    /// Represent the definition of a grid cell. (for XAML)
    /// </summary>
    public class CellDefinition : CellInfoBase
    {
        /// <summary>
        /// Initializes the CellDefinition instance.
        /// </summary>
        /// <param name="control">The control that belong to the current cell.</param>
        public CellDefinition(Control control)
        {
            this.Control = control;
        }

        /// <summary>
        /// Gets the control.
        /// </summary>
        public Control Control { get; private set; }

        /// <summary>
        /// Gets whether the CellDefinition contains valid information or not.
        /// </summary>
        public override bool IsValid
        {
            get
            {
                return base.IsValid && this.Control != null;
            }
        }
    }

    /// <summary>
    /// An extended LayoutEngine providing grid layout capability.
    /// </summary>
    public class GridLayoutEngine : LayoutEngineBase
    {
        /// <summary>
        /// Initializes the GridLayoutEngine instance.
        /// </summary>
        public GridLayoutEngine()
        {
            this.RowDefinitions = this.CreateCollection<RowDefinition>();
            this.ColumnDefinitions = this.CreateCollection<ColumnDefinition>();
            this.CellDefinitions = this.CreateCollection<CellDefinition>();
        }

        /// <summary>
        /// Gets the list of row definitions.
        /// </summary>
        protected internal IList<RowDefinition> RowDefinitions { get; private set; }

        /// <summary>
        /// Gets the list of column definitions.
        /// </summary>
        protected internal ICollection<ColumnDefinition> ColumnDefinitions { get; private set; }

        /// <summary>
        /// Gets the list of cell definitions.
        /// </summary>
        protected internal ICollection<CellDefinition> CellDefinitions { get; private set; }

        /// <summary>
        /// Creates an typed observable collection.
        /// </summary>
        /// <typeparam name="T">The type of items.</typeparam>
        /// <returns>Returns an instance of observable collection.</returns>
        protected IList<T> CreateCollection<T>()
        {
            var collection = new ObservableCollection<T>();
            collection.CollectionChanged += this.OnCollectionChanged;
            return collection;
        }

        /// <summary>
        /// Perform the layout.
        /// </summary>
        /// <param name="container">Container control.</param>
        /// <param name="containerParentSize">Size of the parent of the current container.</param>
        /// <param name="layoutEventArgs">Layout event argument.</param>
        /// <returns>Returns true if the parent must perform layout too,
        /// false otherwise.</returns>
        protected override bool OnLayout(
            Control container,
            Size containerParentSize,
            LayoutEventArgs layoutEventArgs)
        {
            container.Dock = DockStyle.Fill;

            // Use DisplayRectangle so that parent.Padding is honored
            var parentDisplayRectangle = container.DisplayRectangle;

            var perColumns = this.CellDefinitions
                .ToLookup(x => x.Column);

            var perRows = this.CellDefinitions
                .ToLookup(x => x.Row);

            if (perColumns.Count == 0 || perRows.Count == 0)
            {
                return false;
            }

            var horizontalStartPoints = GetCellsStartPoints(
                this.ColumnDefinitions.Count,
                this.ColumnDefinitions.Select(c => c.Width),
                perColumns,
                c => c.ColumnSpan == 1,
                c => c.Control.GetPreferredSize(parentDisplayRectangle.Size).Width,
                parentDisplayRectangle.Width);

            var verticalStartPoints = GetCellsStartPoints(
                this.RowDefinitions.Count,
                this.RowDefinitions.Select(c => c.Height),
                perRows,
                c => c.RowSpan == 1,
                c => c.Control.GetPreferredSize(parentDisplayRectangle.Size).Height,
                parentDisplayRectangle.Height);

            foreach (var cell in this.CellDefinitions)
            {
                var left = 0;
                var top = 0;

                if (cell.Column > 0)
                {
                    left = horizontalStartPoints
                        .Take(cell.Column)
                        .Sum();
                }

                if (cell.Row > 0)
                {
                    top = verticalStartPoints
                        .Take(cell.Row)
                        .Sum();
                }

                var width = horizontalStartPoints
                    .Skip(cell.Column)
                    .Take(cell.ColumnSpan)
                    .Sum();

                var height = verticalStartPoints
                    .Skip(cell.Row)
                    .Take(cell.RowSpan)
                    .Sum();

                ((ILayoutElement)cell.Control).ArrangeLayoutElement(
                    new Rectangle(left, top, width, height));
            }

            return false;
        }

        /// <summary>
        /// Computes the start position of each row or column of a grid.
        /// </summary>
        /// <param name="cellCount">The number of row or column.</param>
        /// <param name="cells">The rows or columns.</param>
        /// <param name="cellGroups">The rows or columns grouped by row or column.</param>
        /// <param name="filterAutoCell">A predicate that let the Auto cells to pass.</param>
        /// <param name="controlSize">A function that tells the size
        /// of a given control. (width or height)</param>
        /// <param name="availableSize">The total available size. (width or height)</param>
        /// <returns>Returns an array of integer telling the
        /// start position of each row or column.</returns>
        private static int[] GetCellsStartPoints(
            int cellCount,
            IEnumerable<CellSize> cells,
            ILookup<int, CellDefinition> cellGroups,
            Func<CellDefinition, bool> filterAutoCell,
            Func<CellDefinition, int> controlSize,
            int availableSize)
        {
            var sizes = new int[cellCount];

            var i = 0;
            foreach (var cell in cells.Where(c => c != null))
            {
                if (cell.Type == CellSizeType.Fixed)
                {
                    sizes[i] = cell.Value;
                }
                else if (cell.Type == CellSizeType.Auto)
                {
                    var cellInfo = cellGroups[i];

                    if (cellInfo != null && cellInfo.Any())
                    {
                        sizes[i] = cellInfo
                            .Where(filterAutoCell)
                            .Max(controlSize);
                    }
                }

                i++;
            }

            var nonStarSize = sizes.Sum();

            var totalStarSize = availableSize - nonStarSize;

            var totalStar = (float)cells
                .Where(col => col.Type == CellSizeType.Star)
                .Select(col => col.Value)
                .Sum();

            i = 0;
            foreach (var cell in cells.Where(c => c != null))
            {
                if (cell.Type == CellSizeType.Star)
                {
                    sizes[i] = (int)Math.Round(totalStarSize * cell.Value / totalStar);
                }

                i++;
            }

            return sizes;
        }

        /// <summary>
        /// Called when a collection has changed.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.OnRequestLayout(e);
        }
    }
}
