﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Reflection;
using System.Windows.Forms;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UIControls.Input
{
    /// <summary>
    /// Extend an IExecutionSource class by compositing its features.
    /// </summary>
    public class ExecutionSourceExtender : IExecutionSource, IDisposable
    {
        /// <summary>
        /// Stores the extended Control instance.
        /// </summary>
        private Action<bool> enabled;

        /// <summary>
        /// Backing field for the Executable property.
        /// </summary>
        private IExecutable executable;

        /// <summary>
        /// Backing field for the ExecutionParameter property.
        /// </summary>
        private object executionParameter;

        /// <summary>
        /// Stores the extended control.
        /// </summary>
        private IExecutionSource extendedControl;

        /// <summary>
        /// Initialize the ExecutionSourceExtender object.
        /// </summary>
        /// <param name="extendedControl">The control to extend.</param>
        public ExecutionSourceExtender(IExecutionSource extendedControl)
        {
            if (extendedControl == null)
            {
                throw new ArgumentNullException("extendedControl");
            }

            this.extendedControl = extendedControl;

            PropertyInfo property = extendedControl.GetType().GetProperty("Enabled");
            if (property != null)
            {
                MethodInfo method = property.GetSetMethod(false);
                if (method != null)
                {
                    this.enabled = (Action<bool>)method.CreateDelegate(
                        typeof(Action<bool>),
                        extendedControl);
                }
            }
        }

        /// <summary>
        /// Fired before Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler BeforeExecute;

        /// <summary>
        /// Fired after Execute method of the IExecutionSource is called.
        /// </summary>
        public event EventHandler AfterExecute;

        /// <summary>
        /// Gets or sets the executable to execute.
        /// </summary>
        public IExecutable Executable
        {
            get
            {
                return this.executable;
            }

            set
            {
                if (this.executable == value)
                {
                    return;
                }

                if (this.executable != null)
                {
                    // unregister event on the previous executable
                    this.executable.CanExecuteChanged -= this.OnExecutableCanExecuteChanged;
                }

                this.executable = value;

                if (this.executable != null)
                {
                    // register event on the new executable
                    this.executable.CanExecuteChanged += this.OnExecutableCanExecuteChanged;

                    this.EvaluateControlAvailability();
                }
            }
        }

        /// <summary>
        /// Gets or sets a custom parameter to provide to the execution.
        /// </summary>
        public object ExecutionParameter
        {
            get
            {
                return this.executionParameter;
            }

            set
            {
                if (this.executionParameter == value)
                {
                    return;
                }

                this.executionParameter = value;

                this.EvaluateControlAvailability();
            }
        }

        /// <summary>
        /// Execute or cancel based on a given event information.
        /// </summary>
        /// <param name="sender">The caller of the event.</param>
        /// <param name="e">The event argument.</param>
        /// <param name="skipIfCancelled">Skip if cancelled from the event.</param>
        public void ExecuteOnEvent(object sender, EventArgs e, bool skipIfCancelled = false)
        {
            if (skipIfCancelled)
            {
                var cancellable = e as CancelEventArgs;
                if (cancellable != null && cancellable.Cancel)
                {
                    // cancelled
                    return;
                }
            }

            if (this.executable != null && this.executable.CanExecute(this.executionParameter))
            {
                this.OnBeforeExecute(sender, e);
                this.executable.Execute(this.executionParameter);
                this.OnAfterExecute(sender, e);
            }
        }

        /// <summary>
        /// Dispose the ExecutionSourceExtender.
        /// </summary>
        public void Dispose()
        {
            if (this.executable != null)
            {
                this.executable.CanExecuteChanged -= this.OnExecutableCanExecuteChanged;
            }
        }

        /// <summary>
        /// Fired before Execute method of the IExecutionSource is called.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event arguments.</param>
        protected virtual void OnBeforeExecute(object sender, EventArgs e)
        {
            EventHandler handler = this.BeforeExecute;

            if (handler != null)
            {
                handler(sender, e);
            }
        }

        /// <summary>
        /// Fired after Execute method of the IExecutionSource is called.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event arguments.</param>
        protected virtual void OnAfterExecute(object sender, EventArgs e)
        {
            EventHandler handler = this.AfterExecute;

            if (handler != null)
            {
                handler(sender, e);
            }
        }

        /// <summary>
        /// Raised when the executable availability changes.
        /// </summary>
        /// <param name="sender">The caller of the event.</param>
        /// <param name="e">The event argument.</param>
        private void OnExecutableCanExecuteChanged(object sender, EventArgs e)
        {
            this.EvaluateControlAvailability();
        }

        /// <summary>
        /// Evaluate the executable availability and update the extended control accordingly.
        /// </summary>
        private void EvaluateControlAvailability()
        {
            if (this.executable != null && this.enabled != null)
            {
                this.enabled(this.executable.CanExecute(this.ExecutionParameter));
            }
        }
    }
}
