﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.UIControls.Behaviors
{
    /// <summary>
    /// A collection of behavior.
    /// </summary>
    public class BehaviorCollection : ICollection<Behavior>, ILogicalTreeElement, IDisposable
    {
        /// <summary>
        /// Stores the control that owns the behavior collection.
        /// </summary>
        private IControl control;

        /// <summary>
        /// Backing field for the LogicalTreeElementExtender property.
        /// </summary>
        private LogicalTreeElementExtender logicalTreeElementExtender;

        /// <summary>
        /// Stores the behaviors instances.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> behaviors;

        /// <summary>
        /// Initializes the BehaviorCollection instance.
        /// </summary>
        /// <param name="control">The control that owns the behaviors.</param>
        public BehaviorCollection(IControl control)
        {
            if (control == null)
            {
                throw new ArgumentNullException("control");
            }

            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.control = control;
            this.behaviors = new IndexableCollection<ILogicalTreeElement>();

            if (control is Control)
            {
                ((Control)control).Disposed += this.OnControlDisposed;
            }
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Gets the number of behaviors in the collection.
        /// </summary>
        public int Count
        {
            get { return this.behaviors.Count; }
        }

        /// <summary>
        /// Gets whether the collection is readonly or not.
        /// It always return false.
        /// </summary>
        public bool IsReadOnly
        {
            get { return false; }
        }

        /// <summary>
        /// Gets the LogicalTreeElementExtender instance.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets of sets the data context.
        /// </summary>
        public object DataContext
        {
            get
            {
                return this.logicalTreeElementExtender.DataContext;
            }

            set
            {
                this.logicalTreeElementExtender.DataContext = value;
            }
        }

        /// <summary>
        /// Gets the parent element.
        /// </summary>
        public ILogicalTreeElement Parent
        {
            get { return this.control; }
        }

        /// <summary>
        /// Gets the child elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get { return this.behaviors; }
        }

        /// <summary>
        /// Add a behavior to the collection.
        /// </summary>
        /// <param name="item">The behavior to add.</param>
        public void Add(Behavior item)
        {
            if (item == null)
            {
                throw new ArgumentNullException("item");
            }

            this.behaviors.Add(item);

            item.Attach(this.control);
        }

        /// <summary>
        /// Removes a behavior from the collection.
        /// </summary>
        /// <param name="item">The behavior to remove.</param>
        /// <returns>Returns true if the behavior could be removed,
        /// false otherwise.</returns>
        public bool Remove(Behavior item)
        {
            if (item == null)
            {
                throw new ArgumentNullException("item");
            }

            item.Detach();

            return this.behaviors.Remove(item);
        }

        /// <summary>
        /// Clears the behaviors.
        /// </summary>
        public void Clear()
        {
            this.behaviors.OfType<Behavior>().ForEach(b => b.Detach());
            this.behaviors.Clear();
        }

        /// <summary>
        /// Check whether a behavior exists in the collection or not.
        /// </summary>
        /// <param name="item">The behavior to check existence upon.</param>
        /// <returns>Returns true if the behavior exists, false otherwise.</returns>
        public bool Contains(Behavior item)
        {
            if (item == null)
            {
                throw new ArgumentNullException("item");
            }

            return this.behaviors.Contains(item);
        }

        /// <summary>
        /// Copies the behaviors to an external array.
        /// </summary>
        /// <param name="array">The target external array.</param>
        /// <param name="arrayIndex">The start index in the target external array.</param>
        public void CopyTo(Behavior[] array, int arrayIndex)
        {
            this.behaviors.CopyTo(array, arrayIndex);
        }

        /// <summary>
        /// Detaches all the behaviors.
        /// </summary>
        public void Dispose()
        {
            foreach (Behavior behavior in this.behaviors)
            {
                behavior.Detach();
            }

            this.behaviors.Clear();

            if (this.control is Control)
            {
                ((Control)this.control).Disposed -= this.OnControlDisposed;
            }
        }

        /// <summary>
        /// Get a behavior enumerator.
        /// </summary>
        /// <returns>Returns a behavior enumerator.</returns>
        public IEnumerator<Behavior> GetEnumerator()
        {
            return this.behaviors.OfType<Behavior>().GetEnumerator();
        }

        /// <summary>
        /// Get a behavior enumerator.
        /// </summary>
        /// <returns>Returns a behavior enumerator.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return this.behaviors.GetEnumerator();
        }

        /// <summary>
        /// Clears the data context.
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Called when the attached control is disposed.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnControlDisposed(object sender, EventArgs e)
        {
            this.Dispose();
        }
    }
}
