﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended TreeView class.
    /// </summary>
    public class UITreeView : TreeView, IItemsControl, ILayoutElement
    {
        /// <summary>Constant message type for setting double buffer for the tree view.</summary>
        private const int TvmSetExtendedStyle = 0x1100 + 44;

        /// <summary>Constant message type for setting double buffer for the tree view.</summary>
        private const int TvmGetExtendedStyle = 0x1100 + 45;

        /// <summary>Constant message type for setting double buffer for the tree view.</summary>
        private const int TvsExDoubleBuffer = 0x0004;

        /// <summary>
        /// Constant that defines the maximum time span of a slow double click in milliseconds.
        /// Two continuous clicks on the same tree node that is slower than
        /// a double click and is faster than this constant is a slow double click.
        /// </summary>
        private const double MaxTimeSpanSlowDoubleClick = 1000.0;

        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Stores the items control extender.
        /// </summary>
        private ItemsControlExtender itemsControlExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Stores the node that was selected before selection changed.
        /// </summary>
        private TreeNode previouslySelectedNode;

        /// <summary>
        /// Stores the tree node under mouse when left button is clicked.
        /// </summary>
        private TreeNode mouseDownTreeNode;

        /// <summary>
        /// Stores the position of the mouse when left button is clicked.
        /// </summary>
        private Point mouseDownPosition;

        /// <summary>
        /// Tells whether the mouse left button is pressed or not.
        /// </summary>
        private bool isLeftButtonPressed;

        /// <summary>
        /// Tells whether the mouse is dragging a node or not.
        /// </summary>
        private bool isDragging;

        /// <summary>
        /// Stores the tree node currently being dragged.
        /// </summary>
        private TreeNode draggingNode;

        /// <summary>
        /// Backing field for the SelectedItem property.
        /// </summary>
        private object selectedItem;

        /// <summary>
        /// Backing field for the SelectedContainer property.
        /// </summary>
        private ILogicalTreeElement selectedContainer;

        /// <summary>
        /// 最後にノードをクリックしたときの時間です。
        /// NodeEarlyDoubleClick/NodeSlowDoubleClickイベントを発生させるために使います。
        /// </summary>
        private DateTime lastNodeClickTime = new DateTime();

        /// <summary>
        /// 最後にクリックしたノードです。
        /// NodeEarlyDoubleClick/NodeSlowDoubleClickイベントを発生させるために使います。
        /// </summary>
        private TreeNode lastClickedNode;

        /// <summary>
        /// Initializes the UITreeView instance.
        /// </summary>
        public UITreeView()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.itemsControlExtender = new ItemsControlExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);

            this.DrawMode = TreeViewDrawMode.OwnerDrawAll;
            this.DoubleBuffered = true;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Raised when the selected node changes.
        /// </summary>
        public event EventHandler<ValueChangedEventArgs<TreeNode>> SelectedNodeChanged;

        /// <summary>
        /// Raised just before DoDragDrop method is called, in the MouseMove event handler.
        /// </summary>
        public event EventHandler<BeforeDragEventArgs> BeforeDoDragDrop;

        /// <summary>
        /// Raised just after DoDragDrop method is called, in the MouseMove event handler.
        /// </summary>
        public event EventHandler AfterDoDragDrop;

        /// <summary>
        /// ノードの描画後に呼び出されるハンドラです。
        /// </summary>
        public event EventHandler<DrawTreeNodeEventArgs> AfterRenderNode;

        /// <summary>
        /// Raised when double click on tree nodes.
        /// This double click event will be triggered before BeforeExpand/BeforeCollipse,
        /// that's why it's called NodeEarlyDoubleClick.
        /// </summary>
        public event TreeNodeMouseClickEventHandler NodeEarlyDoubleClick;

        /// <summary>
        /// Raised when slow double click(twice click) on tree nodes.
        /// </summary>
        public event TreeNodeMouseClickEventHandler NodeSlowDoubleClick;

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// Gets the selected data item.
        /// </summary>
        public object SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.controlExtender.SetValue(ref this.selectedItem, value))
                {
                    if (value != null)
                    {
                        UITreeNode foundNode = this.GetAllNodes().FirstOrDefault(
                                n => ControlExtensions.GetDataItemFromContainer(n) == value);
                        if (foundNode != null)
                        {
                            foundNode.IsSelected = true;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Gets the selected container.
        /// </summary>
        public ILogicalTreeElement SelectedContainer
        {
            get
            {
                return this.selectedContainer;
            }

            private set
            {
                this.controlExtender.SetValue(ref this.selectedContainer, value);
            }
        }

        /// <summary>
        /// Gets whether a tree node is being dragged or not.
        /// </summary>
        public bool IsDragging
        {
            get
            {
                return this.isDragging;
            }

            private set
            {
                this.controlExtender.SetValue(ref this.isDragging, value);
            }
        }

        /// <summary>
        /// Gets the tree node being dragged.
        /// </summary>
        public TreeNode DraggingNode
        {
            get
            {
                return this.draggingNode;
            }

            private set
            {
                var oldNode = this.draggingNode as UITreeNode;

                if (this.controlExtender.SetValue(ref this.draggingNode, value))
                {
                    if (oldNode != null)
                    {
                        oldNode.LogicalTreeElementExtender.NotifyPropertyChanged(
                            propertyName: "IsDragging");
                    }

                    var newNode = this.draggingNode as UITreeNode;
                    if (newNode != null)
                    {
                        newNode.LogicalTreeElementExtender.NotifyPropertyChanged(
                            propertyName: "IsDragging");
                    }
                }
            }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new TreeNodeCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// Gets or sets the children items source.
        /// </summary>
        public IEnumerable ItemsSource
        {
            get { return this.itemsControlExtender.ItemsSource; }
            set { this.itemsControlExtender.ItemsSource = value; }
        }

        /// <summary>
        /// Gets or sets the type of item container to construct.
        /// If this is null, the ItemContainerSelector is used.
        /// </summary>
        public Type ItemContainerType
        {
            get { return this.itemsControlExtender.ItemContainerType; }
            set { this.itemsControlExtender.ItemContainerType = value; }
        }

        /// <summary>
        /// Gets or sets the item container selector.
        /// If this is null, no item container is instanced.
        /// </summary>
        public IItemContainerSelector ItemContainerSelector
        {
            get { return this.itemsControlExtender.ItemContainerSelector; }
            set { this.itemsControlExtender.ItemContainerSelector = value; }
        }

        /// <summary>
        /// 階層を辿って全てのツリーノードを取得します.
        /// </summary>
        /// <returns>全てのツリーノードを返します.</returns>
        public IEnumerable<UITreeNode> GetAllNodes()
        {
            foreach (UITreeNode node in this.Nodes)
            {
                yield return node;
                foreach (UITreeNode sub in node.GetAllNodes())
                {
                    yield return sub;
                }
            }
        }

        /// <summary>
        /// Compares two child controls for sorting.
        /// </summary>
        /// <param name="a">First control.</param>
        /// <param name="b">Second control.</param>
        /// <returns>Returns 0 if equal, 1 or -1 if different depending on order.</returns>
        public virtual int CompareControls(IControl a, IControl b)
        {
            return 0;
        }

        /// <summary>
        /// Retrieve the data item corresponding to the container.
        /// </summary>
        /// <param name="container">The container to look for its data item.</param>
        /// <returns>Returns the data item, or null if not found.</returns>
        public object GetDataItemFromContainer(IControl container)
        {
            return this.itemsControlExtender.GetDataItemFromContainer(container);
        }

        /// <summary>
        /// Retrieve the container corresponding to the data item.
        /// </summary>
        /// <param name="dataItem">The data item to look for its data.</param>
        /// <returns>Returns the container, or null if not found.</returns>
        public IControl GetContainerFromDataItem(object dataItem)
        {
            return this.itemsControlExtender.GetContainerFromDataItem(dataItem);
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Called when a child node is added.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected internal virtual void OnChildNodeAdded(TreeViewEventArgs e)
        {
        }

        /// <summary>
        /// Called when a child node is removed.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected internal virtual void OnChildNodeRemoved(TreeViewEventArgs e)
        {
        }

        /// <summary>
        /// Handle HandleCreated event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnHandleCreated(EventArgs e)
        {
            EffectMaker.Foundation.Win32.Functions.SendMessage(
                this.Handle,
                TvmSetExtendedStyle,
                (IntPtr)TvsExDoubleBuffer,
                (IntPtr)TvsExDoubleBuffer);

            base.OnHandleCreated(e);
        }

        /// <summary>
        /// Called after a node selection changes.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnAfterSelect(TreeViewEventArgs e)
        {
            base.OnAfterSelect(e);
            this.UpdateSelections();

            this.previouslySelectedNode = this.SelectedNode;
        }

        /// <summary>
        /// Called when the selected node changed.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnSelectedNodeChanged(ValueChangedEventArgs<TreeNode> e)
        {
            var handler = this.SelectedNodeChanged;

            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Called when owner draing node is required.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnDrawNode(DrawTreeNodeEventArgs e)
        {
            base.OnDrawNode(e);

            var node = e.Node as UITreeNode;
            if (node == null)
            {
                e.DrawDefault = true;
            }
            else
            {
                try
                {
                    e.DrawDefault = !node.Render(e.Bounds, e.State, e.Graphics);
                    if (this.AfterRenderNode != null)
                    {
                        this.AfterRenderNode(this, e);
                    }
                }
                catch (NullReferenceException)
                {
                    // ProjectTreeNode.DrawHierarchy()のthis.NextNode == nullで発生する
                    // NullReferenceExceptionをキャッチ
                    // (Emitterノードを選択した状態でEmitterSetを作成したときなどに起きる)
                }
            }
        }

        /// <summary>
        /// Processes Windows messages.
        /// </summary>
        /// <param name="m">The Windows Message to process.</param>
        protected override void WndProc(ref Message m)
        {
            switch ((Foundation.Win32.WM)m.Msg)
            {
                case Foundation.Win32.WM.WM_LBUTTONDOWN:
                    {
                        Point mousePos = this.PointToClient(MousePosition);
                        this.OnMouseDown(new MouseEventArgs(
                            MouseButtons, 1, mousePos.X, mousePos.Y, 0));

                        return;
                    }

                case Foundation.Win32.WM.WM_LBUTTONUP:
                    {
                        Point mousePos = this.PointToClient(MousePosition);
                        this.OnMouseUp(new MouseEventArgs(MouseButtons, 1, mousePos.X, mousePos.Y, 0));

                        return;
                    }

                case Foundation.Win32.WM.WM_LBUTTONDBLCLK:
                    {
                        Point mousePos = this.PointToClient(MousePosition);
                        this.OnMouseDoubleClick(new MouseEventArgs(MouseButtons, 1, mousePos.X, mousePos.Y, 0));

                        return;
                    }
            }

            try
            {
                base.WndProc(ref m);
            }
            catch
            {
                // ハンドルの作成がカスタム描画で間に合わないことがあるので握りつぶし
            }
        }

        /// <summary>
        /// Raised when the mouse is moved.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            if (this.isLeftButtonPressed && this.mouseDownTreeNode != null)
            {
                int dx = this.mouseDownPosition.X - e.X;
                int dy = this.mouseDownPosition.Y - e.Y;
                int dist = this.mouseDownTreeNode.Bounds.Height / 2;

                if (this.IsDragging == false && (Math.Abs(dx) >= dist || Math.Abs(dy) >= dist))
                {
                    this.DraggingNode = this.mouseDownTreeNode;
                    this.IsDragging = true;
                    this.SelectedNode = null;

                    this.OnDoDragDrop(this.mouseDownTreeNode);

                    this.DraggingNode = null;
                    this.IsDragging = false;
                    this.SelectedNode = this.mouseDownTreeNode;

                    this.mouseDownTreeNode = null;
                    this.isLeftButtonPressed = false;
                }
            }
        }

        /// <summary>
        /// Raised when a mouse button is down.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            UITreeNode node = this.GetNodeAt(e.Location) as UITreeNode;

            if (e.Button == MouseButtons.Left)
            {
                // Get the current time and current node.
                DateTime now  = DateTime.Now;

                // Should we trigger early/slow double click event?
                if (node != null && node.IsSelected == true && node == this.lastClickedNode)
                {
                    double doubleClickDuration = (now - this.lastNodeClickTime).TotalMilliseconds;

                    if (doubleClickDuration <= SystemInformation.DoubleClickTime)
                    {
                        if (this.NodeEarlyDoubleClick != null)
                        {
                            var args = new TreeNodeMouseClickEventArgs(
                                node,
                                e.Button,
                                e.Clicks,
                                e.X,
                                e.Y);
                            this.NodeEarlyDoubleClick(this, args);
                        }
                    }
                    else if (doubleClickDuration <= MaxTimeSpanSlowDoubleClick)
                    {
                        if (this.NodeSlowDoubleClick != null)
                        {
                            var args = new TreeNodeMouseClickEventArgs(
                                node,
                                e.Button,
                                e.Clicks,
                                e.X,
                                e.Y);
                            this.NodeEarlyDoubleClick(this, args);
                        }
                    }
                }

                // Update the variables.
                this.lastNodeClickTime = now;
                this.lastClickedNode = node;
            }

            // Ctrlキーによる複数選択
            if ((Control.ModifierKeys & Keys.Control) == Keys.Control
                && e.Button == MouseButtons.Left)
            {
                // 本選択ノードは複数選択対象にできない
                if (node != null && node != this.SelectedNode)
                {
                    node.ToggleMultiSelection();
                }

                return;
            }

            // Shiftキーによる複数選択
            if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift
                && e.Button == MouseButtons.Left)
            {
                this.ProcRangeSelection(node);
                return;
            }

            base.OnMouseDown(e);

            if (e.Button == MouseButtons.Left || e.Button == MouseButtons.Right)
            {
                this.isLeftButtonPressed = e.Button == MouseButtons.Left;
                this.DraggingNode = null;
                this.IsDragging = false;
                this.mouseDownTreeNode = node;
                this.mouseDownPosition = e.Location;

                if (node != null)
                {
                    // 同じノードをクリックした時は複数選択を解除
                    if (node == this.SelectedNode)
                    {
                        foreach (var n in this.Nodes.OfType<UITreeNode>())
                        {
                            this.ClearMultiSelection(n);
                        }
                    }

                    // Raise a mouse down event.
                    node.OnMouseDown(e);

                    // Update a selected node.
                    this.SelectedNode = node;
                }
            }
        }

        /// <summary>
        /// Raised when a mouse button is double clicked.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);

            if (e.Button == System.Windows.Forms.MouseButtons.Left)
            {
                // Raise a mouse double click event.
                var node = this.GetNodeAt(e.X, e.Y) as UITreeNode;
                if (node != null)
                {
                    node.OnMouseDoubleClick(e);
                }
            }
        }

        /// <summary>
        /// Called just before the DoDragDrop method is called, in the MouseMove handler.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnBeforeDoDragDrop(BeforeDragEventArgs e)
        {
            EventHandler<BeforeDragEventArgs> handler = this.BeforeDoDragDrop;

            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Called just after the DoDragDrop method is called, in the MouseMove handler.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnAfterDoDragDrop(EventArgs e)
        {
            EventHandler handler = this.AfterDoDragDrop;

            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Called when a dragging process is being performed.
        /// </summary>
        /// <param name="draggingNode">The tree node that is being dragged.</param>
        protected virtual void OnDoDragDrop(TreeNode draggingNode)
        {
            var e = new BeforeDragEventArgs();
            this.OnBeforeDoDragDrop(e);
            this.DoDragDrop(draggingNode, e.AllowedEffects);
            this.OnAfterDoDragDrop(EventArgs.Empty);
        }

        /// <summary>
        /// コントロールがフォーカスを失ったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);

            // マウスの左ボタンが押された状態で OnMouseDown イベントを飛ばして OnMouseMove イベントが呼ばれることがあり、
            // そのときに選択ノードが古い mouseDownTreeNode に切り替わってしまうため、フォーカスが外れたときは
            // マウスの操作情報をクリアして選択状態が切り替わらないようにしておく
            this.mouseDownTreeNode = null;
            this.mouseDownPosition = new Point();
        }

        /// <summary>
        /// 複数選択を解除します。
        /// </summary>
        /// <param name="clickedNode">基点ノード</param>
        private void ClearMultiSelection(UITreeNode clickedNode)
        {
            clickedNode.IsMultiSelected = false;
            foreach (var n in clickedNode.Nodes.OfType<UITreeNode>())
            {
                this.ClearMultiSelection(n);
            }
        }

        /// <summary>
        /// 範囲複数選択を処理します。
        /// </summary>
        /// <param name="clickedNode">クリックしたノード</param>
        private void ProcRangeSelection(UITreeNode clickedNode)
        {
            // 見えてるノードリストを作る
            var visibleList = this.GetAllNodes().Where(n => n.IsVisible).ToList();

            // 見えてるリスト内での、選択済みノードとクリックノードのインデックスを取得
            var addedIndex = visibleList.IndexOf(clickedNode);
            var selectedIndex = visibleList.IndexOf((UITreeNode)this.SelectedNode);

            if (addedIndex == -1 || selectedIndex == -1)
            {
                // 見えてるノードにクリックしたものや選択したものが含まれてないのは例外モノ
                return;
            }

            // 大小関係に応じて操作の開始と終了インデックスを設定
            int startIndex, endIndex;
            if (addedIndex > selectedIndex)
            {
                startIndex = selectedIndex + 1;
                endIndex = addedIndex;
            }
            else if (addedIndex < selectedIndex)
            {
                startIndex = addedIndex;
                endIndex = selectedIndex - 1;
            }
            else
            {
                return;
            }

            // 伝統的なfor文で1つずつトグル(実際にトグルするかどうかはメソッドで判定)
            for (int i = startIndex; i <= endIndex; ++i)
            {
                if (!visibleList[i].IsMultiSelected)
                {
                    visibleList[i].ToggleMultiSelection();
                }
            }
        }

        /// <summary>
        /// Check all selected state and raise events accordingly.
        /// </summary>
        private void UpdateSelections()
        {
            this.FireSelectionEvents(this.previouslySelectedNode);
            this.FireSelectionEvents(this.SelectedNode);

            var node = this.SelectedNode as UITreeNode;
            this.SelectedContainer = node;
            this.SelectedItem = ControlExtensions.GetDataItemFromContainer(node);

            // tells that the value of the SelectedNode property has changed
            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedNode");

            // tells everybody that the selected node has changed
            this.OnSelectedNodeChanged(new ValueChangedEventArgs<TreeNode>(
                    this.previouslySelectedNode,
                    this.SelectedNode));

            this.Focus();
        }

        /// <summary>
        /// Check whether the tree node is of type UITreeNode,
        /// and raise selection events accordingly.
        /// </summary>
        /// <param name="treenode">The tree node that selection changed.</param>
        private void FireSelectionEvents(TreeNode treenode)
        {
            var node = treenode as UITreeNode;

            if (node == null)
            {
                return;
            }

            // IsSelectedをアップデートするとSelectedNodeがnullにされるバグ
            var tmp = this.SelectedNode;

            // raise a property changed event on the unselected node
            node.LogicalTreeElementExtender.NotifyPropertyChanged(propertyName: "IsSelected");

            // とりあえず値を保持して戻すように修正
            if (this.SelectedNode == null)
            {
                this.SelectedNode = tmp;
            }

            // tells the node that it is getting unselected
            node.OnSelectionChanged(new ValueChangedEventArgs<bool>(!node.IsSelected, node.IsSelected));
        }

        /// <summary>
        /// Event arguments related to drag and drop feature.
        /// </summary>
        public class BeforeDragEventArgs : EventArgs
        {
            /// <summary>
            /// Initializes the BeforeDragEventArgs instance.
            /// </summary>
            public BeforeDragEventArgs()
            {
                this.AllowedEffects = DragDropEffects.None;
            }

            /// <summary>
            /// Gets and sets the allowed effects for the drag and drop.
            /// </summary>
            public DragDropEffects AllowedEffects { get; set; }
        }
    }
}
