﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Layout;
using ControlExtensions = EffectMaker.UIControls.Extensions.ControlExtensions;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended TabControl class.
    /// </summary>
    public class UITabControl : TabControl, IItemsControl, ILayoutElement, IDocumentLinker
    {
        /// <summary>
        /// Backing field for the ControlExtender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// Stores the ItemsControlExtender instance.
        /// </summary>
        private ItemsControlExtender itemsControlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// Backing field for the SelectedContainer property.
        /// </summary>
        private ILogicalTreeElement selectedContainer;

        /// <summary>
        /// Stores a reference to the previously selected page.
        /// </summary>
        private TabPage previousPage;

        /// <summary>
        /// Backing field for the SelectedItem property.
        /// </summary>
        private object selectedItem;

        /// <summary>
        /// 初回フォーカス検知フラグ
        /// </summary>
        private bool firstFocus = true;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UITabControl()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.itemsControlExtender = new ItemsControlExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// Gets or sets the sequence of items used as input data source.
        /// </summary>
        public IEnumerable ItemsSource
        {
            get
            {
                return this.itemsControlExtender.ItemsSource;
            }

            set
            {
                this.controlExtender.SkipPropertyChangedEvent = true;
                this.itemsControlExtender.ItemsSource = value;
                this.controlExtender.SkipPropertyChangedEvent = false;
            }
        }

        /// <summary>
        /// Gets or sets the type of container to create for data items.
        /// </summary>
        public Type ItemContainerType
        {
            get { return this.itemsControlExtender.ItemContainerType; }
            set { this.itemsControlExtender.ItemContainerType = value; }
        }

        /// <summary>
        /// Gets or sets the ItemContainerSelector instance.
        /// </summary>
        public IItemContainerSelector ItemContainerSelector
        {
            get { return this.itemsControlExtender.ItemContainerSelector; }
            set { this.itemsControlExtender.ItemContainerSelector = value; }
        }

        /// <summary>
        /// Gets or sets the selected item container.
        /// </summary>
        public ILogicalTreeElement SelectedContainer
        {
            get
            {
                return this.selectedContainer;
            }

            set
            {
                if (this.controlExtender.SetValue(ref this.selectedContainer, value))
                {
                    if (value is TabPage)
                    {
                        this.SelectedTab = value as TabPage;
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the selected data item.
        /// </summary>
        public object SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.controlExtender.SetValue(ref this.selectedItem, value))
                {
                    if (value != null)
                    {
                        IControl container = null;
                        if (this.ItemsSource == null)
                        {
                            foreach (TabPage page in this.TabPages)
                            {
                                var myPage = page as UITabPage;
                                if (myPage != null && myPage.DataContext == value)
                                {
                                    container = myPage;
                                    break;
                                }
                            }
                        }
                        else
                        {
                            container = ControlExtensions.GetContainerFromDataItem(this, value);
                        }

                        this.SelectedContainer = container;
                    }
                }
            }
        }

        /// <summary>
        /// ヘルプとしてリンクするページのIDを取得または設定します。
        /// </summary>
        public string DocumentId { get; set; }

        /// <summary>
        /// Gets a CreateParams that contains the required creation parameters
        /// when the handle to the control is created.
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                cp.ExStyle |= 0x02000000; // WS_EX_COMPOSITED
                return cp;
            }
        }

        /// <summary>
        /// Compares two child controls for sorting.
        /// </summary>
        /// <param name="a">First control.</param>
        /// <param name="b">Second control.</param>
        /// <returns>Returns 0 if equal, 1 or -1 if different depending on order.</returns>
        public virtual int CompareControls(IControl a, IControl b)
        {
            return 0;
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Retrieve the data item corresponding to the container.
        /// </summary>
        /// <param name="container">The container to look for its data item.</param>
        /// <returns>Returns the data item, or null if not found.</returns>
        public object GetDataItemFromContainer(IControl container)
        {
            return this.itemsControlExtender.GetDataItemFromContainer(container);
        }

        /// <summary>
        /// Retrieve the container corresponding to the data item.
        /// </summary>
        /// <param name="dataItem">The data item to look for its data.</param>
        /// <returns>Returns the container, or null if not found.</returns>
        public IControl GetContainerFromDataItem(object dataItem)
        {
            return this.itemsControlExtender.GetContainerFromDataItem(dataItem);
        }

        /// <summary>
        /// Called when a control is added.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnControlAdded(ControlEventArgs e)
        {
            base.OnControlAdded(e);

            if (this.Controls.Count == 1)
            {
                this.previousPage = this.TabPages[0];
                if (this.previousPage != null && this.previousPage is UITabPage)
                {
                    ((UITabPage)this.previousPage).OnSelected();
                }
            }
        }

        /// <summary>
        /// Testing OnSelected event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnSelected(TabControlEventArgs e)
        {
            base.OnSelected(e);

            if (this.previousPage != null && this.previousPage is UITabPage)
            {
                ((UITabPage)this.previousPage).OnUnselected();
            }

            var selectedPage = e.TabPage as IControl;

            if (selectedPage != null && e.TabPageIndex != -1)
            {
                this.SelectedContainer = selectedPage;
                if (!EnumerableExtensions.IsNullOrEmpty(this.ItemsSource))
                {
                    this.SelectedItem = this.ItemsSource
                        .Cast<object>()
                        .ElementAt(e.TabPageIndex);
                }
                else
                {
                    this.SelectedItem = null;
                }

                if (selectedPage is UITabPage)
                {
                    ((UITabPage)selectedPage).OnSelected();
                }
            }
            else
            {
                this.SelectedContainer = null;
                this.SelectedItem = null;
            }

            this.previousPage = this.SelectedTab;
        }

        /// <summary>
        /// Handle MouseUp event for the tab control.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            if (e.Button == MouseButtons.Right)
            {
                for (int i = 0; i < this.TabPages.Count; ++i)
                {
                    var rect = this.GetTabRect(i);
                    if (rect.Contains(e.Location) == true)
                    {
                        var page = this.TabPages[i] as UITabPage;
                        if (page != null && page.TabHeaderContextMenuStrip != null)
                        {
                            page.TabHeaderContextMenuStrip.Show(this, e.Location);
                        }

                        break;
                    }
                }
            }
        }

        /// <summary>
        /// タブコントロールに初めてフォーカスが当たった場合、サブタブなら親タブにフォーカスします。
        /// </summary>
        /// <param name="e">event</param>
        protected override void OnGotFocus(EventArgs e)
        {
            if (this.firstFocus)
            {
                var parent = this.Parent as UITabPage;
                if (parent != null)
                {
                    var paretnTabCtl = parent.Parent as UITabControl;
                    if (paretnTabCtl != null)
                    {
                        paretnTabCtl.Focus();
                    }
                }

                this.firstFocus = false;
            }

            base.OnGotFocus(e);
        }
    }
}
