﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Linq;
using EffectMaker.BusinessLogic.Options;
#if true
using System;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.Windows.Forms;
using System.Collections.Generic;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;
using EffectMaker.Foundation.Utility;
using EffectMaker.Foundation.Win32;
using System.Runtime.InteropServices;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// ＵＩ数値アップダウンクラス。
    /// </summary>
    public class UINumericUpDown : NumericUpDown
    {
        /// <summary>
        /// インポート関数：SendMessage
        /// </summary>
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        public static extern IntPtr SendMessage(IntPtr hWnd, int msg, IntPtr wParam, IntPtr lParam);

        // 内部ボタンコントロール
        private readonly Control _internalButtons;
        // 内部テキストボックス
        private readonly TextBox _internalTextBox;
        // 大増減値。
        private decimal _largeIncrement = new decimal(10);
        // リターン入力有効フラグ
        private bool _wantReturn = true;
        // 自動範囲拡張フラグ
        private bool _autoRangeExpansion = false;
        // 最小値変更中フラグ
        private bool _minimumFixed = false;
        // 値変更中フラグ
        private bool _valueChanging = false;

        private Point point = new Point(-1,-1);
        //
        private bool m_bAllTextSelected = false;

        /// <summary>
        /// Default RoundNumber
        /// </summary>
        public const int DefaultRoundNumber = 3;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UINumericUpDown()
        {
            base.TextAlign = HorizontalAlignment.Right;

            // 内部コントロール取得
            this._internalButtons = base.Controls[0];
            this._internalTextBox = this.Controls.OfType<TextBox>().First();
            this.Click += (s, e) =>
            {
                if (!this._internalButtons.DisplayRectangle.Contains(
                    this._internalButtons.PointToClient(Control.MousePosition)))
                {
                    if (OptionStore.RootOptions.Interface.ClickToSelectAllInTextBox)
                    {
                        this._internalTextBox.SelectAll();
                    }
                }
            };
            this.GotFocus += (s, e) =>
            {
                if (UIUserControl.DisableMenuShortcut != null)
                {
                    UIUserControl.DisableMenuShortcut();
                }
            };
            this.LostFocus += (s, e) =>
            {
                if (UIUserControl.EnableMenuShortcut != null)
                {
                    UIUserControl.EnableMenuShortcut();
                }
            };
            this.MouseMove += new MouseEventHandler(OnMouseMove);
            this.AllowDrop = false;
            this.RoundNumber = DefaultRoundNumber;
        }

        #region プロパティ

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(typeof(decimal), "10")]
        [Description("大増減値。PageUp、PageDown入力時に使用します。")]
        public decimal LargeIncrement
        {
            get { return _largeIncrement; }
            set { _largeIncrement = value; }
        }

        /// <summary>
        /// リターン入力有効フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Description("リターン入力を有効にするかどうかを示します。")]
        public bool WantReturn
        {
            get { return _wantReturn; }
            set { _wantReturn = value; }
        }

        /// <summary>
        /// 自動範囲拡張フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("範囲外の値が設定された時に範囲を自動で拡張するかどうかを示します。")]
        public bool AutoRangeExpantion
        {
            get { return _autoRangeExpansion; }
            set { _autoRangeExpansion = value; }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(false)]
        [Description("最小値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public bool MinimumFixed
        {
            get { return _minimumFixed; }
            set { _minimumFixed = value; }
        }

        /// <summary>
        /// 小数点以下の丸め込む桁数
        /// </summary>
        [DefaultValue(DefaultRoundNumber)]
        [Description("小数点以下の丸め込む桁数")]
        public int RoundNumber { get; set; }

        /// <summary>
        /// Skip mouse wheel event processing.
        /// </summary>
        [DefaultValue(false)]
        public bool SkipMouseWheel { get; set; }

        #endregion

        #region イベント

        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler =
                (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }

        #endregion

        /// <summary>
        /// 全選択。
        /// </summary>
        public void SelectAll()
        {
            _internalTextBox.SelectAll();
        }

        /// <summary>
        /// 内部ボタンコントロール。
        /// </summary>
        protected Control InternalButtons
        {
            get { return _internalButtons; }
        }

        /// <summary>
        /// 内部テキストボックス。
        /// </summary>
        public TextBox InternalTextBox
        {
            get { return _internalTextBox; }
        }

        /// <summary>
        /// 現在の値を取得。
        /// </summary>
        protected decimal GetCurrentValue()
        {
            decimal value = decimal.Zero;

            // UserEdit == true 時は base.Value で
            // ValidateEditText が呼ばれてしまう
            bool userEdit = base.UserEdit;
            base.UserEdit = false;
            {
                value = Math.Round(base.Value, this.RoundNumber);
            }
            base.UserEdit = userEdit;

            return value;
        }

        /// <summary>
        /// 数値テキストを取得。
        /// </summary>
        protected string GetNumberText(decimal value)
        {
            // NumericUpDown の private GetNumberText() とまったく同じ
            // 公開されていないので、そのまま持ってきてます
            if (base.Hexadecimal)
            {
                long number = (long)value;
                return number.ToString("X", CultureInfo.InvariantCulture);
            }
            return value.ToString((base.ThousandsSeparator ? "N" : "F") +
                base.DecimalPlaces.ToString(CultureInfo.CurrentCulture), CultureInfo.CurrentCulture);
        }

        /// <summary>
        /// 小数部の有効桁数を取得。
        /// </summary>
        public static int GetDecimalPlaces(decimal value)
        {
            // 一旦文字列にする
            string valueString = value.ToString();

            // 小数点位置
            int periodIndex = valueString.LastIndexOf('.');
            if (periodIndex == -1)
            {
                return 0;
            }

            // 小数部文字列
            string decimalString = valueString.Substring(periodIndex + 1);

            // 右側から「０」の部分を除いた文字数を計測
            int result = decimalString.Length;
            for (int i = decimalString.Length - 1; i >= 0; i--)
            {
                if (decimalString[i] == '0')
                {
                    result--;
                }
                else
                {
                    break;
                }
            }
            return result;
        }

        /// <summary>
        /// 値を増減させる。
        /// </summary>
        protected virtual void IncrementValue( bool up,
                                               decimal increment,
                                               bool startValueChanging )
        {
            decimal value  = GetCurrentValue();
            decimal result = value;
            if (increment > decimal.Zero)
            {
                // 増加
                if (up)
                {
                    result = Math.Floor((value + increment) / increment) * increment;
                }
                // 減少
                else
                {
                    result = Math.Ceiling((value - increment) / increment) * increment;
                }

                if ( (Control.ModifierKeys & Keys.Shift)==0 )
                {
                    int iDigits =
                        MathUtility.FindMostSignificantDigit( (float)this.Increment );
                    result = Math.Round( result, iDigits );
                }
                else
                {
                    int iDigits =
                        MathUtility.FindMostSignificantDigit( (float)this.LargeIncrement );
                    result = Math.Round( result, iDigits );
                }

                if ( result>base.Maximum )
                    result = base.Maximum;
                if ( result<base.Minimum )
                    result = base.Minimum;

                if (result != value)
                {
                    if (startValueChanging)
                    {
                        _valueChanging = true;
                    }
                    base.Value = result;
                }
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="curValue">設定したい値</param>
        protected virtual void SetValue(decimal curValue)
        {
            decimal fixValue = curValue;

            // 値を修正
            if ( this.Minimum>fixValue )
                fixValue = this.Minimum;

            if ( this.Maximum<fixValue )
                fixValue = this.Maximum;


            // 値を更新する
            try
            {
                base.Value = fixValue;
            }
            catch (Exception/* ex*/)
            {
#if false // TODO:コード復帰
                DebugConsole.WriteLine(ex.Message);
#endif
            }
        }

        #region オーバーライド

        private bool? overridenOnFocusSelectAllEnabled;

        /// <summary>
        /// Set an overriden value for the OnFocusSelectAllEnabled property.
        /// </summary>
        /// <param name="value">New value of the OnFocusSelectAllEnabled property.</param>
        public void SetOnFocusSelectAllEnabledOverride(bool value)
        {
            overridenOnFocusSelectAllEnabled = value;
        }

        /// <summary>
        /// Clears the overriden value of the OnFocusSelectAllEnabled property and
        /// revert it back to Option-based global value.
        /// </summary>
        public void ClearOnFocusSelectAllEnabledOverride()
        {
            overridenOnFocusSelectAllEnabled = null;
        }


        /// <summary>
        /// フォーカス時に全て選択するか。
        /// </summary>
        public bool OnFocusSelectAllEnabled
        {
            get
            {
                if (overridenOnFocusSelectAllEnabled.HasValue)
                {
                    return overridenOnFocusSelectAllEnabled.Value;
                }

                return false;

#if false // TODO:コード復帰
                if ( Config.Data==null ||
                     Config.Data.Option==null )
                {
                    return false;
                }

                return Config.Data.Option.TextBoxClickSelectAll;
#endif
            }
            set { } // Do nothing, this property is not accessible from here.
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
#if false // TODO:コード復帰
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
#endif
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(80, base.DefaultSize.Height); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (_wantReturn)
            {
                if (keyData == Keys.Return)
                {
                    // OnLostFocus() と同じ処理をする
                    if (base.UserEdit)
                    {
                        UpdateEditText();
                    }
                    return true;
                }
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスホイール
                case (int)EffectMaker.Foundation.Win32.WM.WM_MOUSEWHEEL:
                    {
                        // フォーカスあり
                        if (SkipMouseWheel == false && this.Focused)
                        {
                            // コントロール領域内なら増減させる
                            Point point = PointToClient(EffectMaker.Foundation.Win32.Utility.LParamToPoint(m.LParam));
                            if (this.ClientRectangle.Contains(point))
                            {
                                int delta = (int)EffectMaker.Foundation.Win32.Utility.SignedHIWORD(m.WParam);

                                if ( (Control.ModifierKeys & Keys.Shift)==0 )
                                    IncrementValue( delta > 0, this.Increment, false );
                                else
                                    IncrementValue( delta > 0, this.LargeIncrement, false );

                                return;
                            }
                        }

                        // 親に委譲する
                        UINumericUpDown.SendMessage(this.Parent.Handle, m.Msg, m.WParam, m.LParam);
                    }
                    return;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateEditText()
        {
            base.UpdateEditText();

            // 全選択
            if (this.Focused)
            {
                _internalTextBox.SelectAll();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void UpButton()
        {
            if ( (Control.ModifierKeys & Keys.Shift)==0 )
                IncrementValue( true, base.Increment, true );
            else
                IncrementValue( true, this.LargeIncrement, true );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void DownButton()
        {
            if ( (Control.ModifierKeys & Keys.Shift)==0 )
                IncrementValue( false, base.Increment, true );
            else
                IncrementValue( false, this.LargeIncrement, true );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            // 全選択（TABキー移動時のみ）
            if (Control.MouseButtons == MouseButtons.None)
            {
                SelectAll();
            }
            base.OnGotFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);

            // 値変更イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));

            // NumericUpDown.OnValueChanged は CanRaiseEvents が
            // 効かないので、呼び出し条件を直接判断する
            if (this.CanRaiseEvents)
            {
                base.OnValueChanged(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnValueChanged(EventArgs e)
        {
            // 値変更イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(_valueChanging));

            // NumericUpDown.OnValueChanged は CanRaiseEvents が
            // 効かないので、呼び出し条件を直接判断する
            if (this.CanRaiseEvents)
            {
                base.OnValueChanged(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // 最小値
            if (e.KeyData == Keys.Home)
            {
                base.Value = base.Minimum;
                e.Handled = true;
            }
            // 最大値
            else if (e.KeyData == Keys.End)
            {
                base.Value = base.Maximum;
                e.Handled = true;
            }
            // 大増加
            else if (e.KeyData == Keys.PageUp)
            {
                IncrementValue(true, _largeIncrement, true);
                e.Handled = true;
            }
            // 大減少
            else if (e.KeyData == Keys.PageDown)
            {
                IncrementValue(false, _largeIncrement, true);
                e.Handled = true;
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            // 値変更中解除
            if (e.KeyData == Keys.Up || e.KeyData == Keys.Down ||
                e.KeyData == Keys.PageUp || e.KeyData == Keys.PageDown)
            {
                if (_valueChanging)
                {
                    _valueChanging = false;
                    OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                }
            }

            if (e.KeyCode == Keys.Tab)
            {
                m_bAllTextSelected = false;
            }
            else if (e.KeyCode == Keys.Enter)
            {
                this.SelectAll();
                m_bAllTextSelected = true;
            }

            base.OnKeyUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected void OnMouseMove(object sender, MouseEventArgs e)
        {
            int iUpdateSpeed = 1;
            if (e.Button == MouseButtons.Left)
            {
                Point curPoint = e.Location;
                int iYDiff = (point.Y - curPoint.Y);

                decimal increment = iUpdateSpeed;
                if ( (Control.ModifierKeys & Keys.Shift)==0 )
                    increment *= this.Increment;
                else
                    increment *= this.LargeIncrement;

                if (iYDiff > 0)
                    IncrementValue( true, increment, true );
                else if (iYDiff < 0)
                    IncrementValue( false, increment, true );

                point = e.Location;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                point = e.Location;

        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // 値変更中解除
            if (e.Button == MouseButtons.Left)
            {
                if (_valueChanging)
                {
                    _valueChanging = false;
                    OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                }
            }

            if ( this.OnFocusSelectAllEnabled==true &&
                 m_bAllTextSelected==true )
            {
                if ((base.Controls[1] as TextBox).SelectionLength <= 0)
                    this.SelectAll();
                m_bAllTextSelected = false;
            }

            base.OnMouseUp(e);
        }


        /// <summary>
        ///
        /// </summary>
        /// <param name="e"></param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            if ( this.OnFocusSelectAllEnabled==true )
            {
                this.SelectAll();
                m_bAllTextSelected = true;
            }
        }

        #endregion

        #region デザイナ制御

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool Hexadecimal
        {
            get { return base.Hexadecimal; }
            set { base.Hexadecimal = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool InterceptArrowKeys
        {
            get { return base.InterceptArrowKeys; }
            set { base.InterceptArrowKeys = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool ReadOnly
        {
            get { return base.ReadOnly; }
            set { base.ReadOnly = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Right)]
        public new HorizontalAlignment TextAlign
        {
            get { return base.TextAlign; }
            set { base.TextAlign = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new LeftRightAlignment UpDownAlign
        {
            get { return base.UpDownAlign; }
            set { base.UpDownAlign = value; }
        }

        #endregion

        private void InitializeComponent()
        {
            ((System.ComponentModel.ISupportInitialize)(this)).BeginInit();
            this.SuspendLayout();
            ((System.ComponentModel.ISupportInitialize)(this)).EndInit();
            this.ResumeLayout(false);

        }
    }

    //----------------------------------------------------------------------------
    // 値編集用
    #region IntUpDown
    /// <summary>
    /// 整数値アップダウンクラス。
    /// </summary>
    public class IntUpDown : UINumericUpDown
    {
        // Use floating point numbers to prevent underflowing
        private double _fSmallIncrement = 0.0;
        private double _fLargeIncrement = 0.0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntUpDown()
        {
            _fSmallIncrement = (double)base.Increment;
            _fLargeIncrement = (double)base.LargeIncrement;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Description("値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Value
        {
            get { return Convert.ToInt32(base.Value); }
            set
            {
                // 範囲拡張
                if (this.AutoRangeExpantion)
                {
                    IntRangeProperty range = this.RangeProperty;
                    if (!range.Contains(value))
                    {
                        while (!range.Contains(value))
                        {
                            if (!range.Scale(true))
                            {
                                break;
                            }
                        }
                        this.Minimum        = range.Minimum;
                        this.Maximum        = range.Maximum;

                        this.IncrementReal      = range.ChangeS;
                        this.LargeIncrementReal = range.ChangeL;
                    }
                }

                // 値を更新
                SetValue(Convert.ToDecimal(value));
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0)]
        [Description("最小値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Minimum
        {
            get { return Convert.ToInt32(base.Minimum); }
            set { base.Minimum = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100)]
        [Description("最大値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Maximum
        {
            get { return Convert.ToInt32(base.Maximum); }
            set { base.Maximum = Convert.ToDecimal(value); }
        }

        ///// <summary>
        ///// 最小値。
        ///// </summary>
        //[DefaultValue(false)]
        //[Description("最小値。int型で再定義しています。")]
        //[RefreshProperties(RefreshProperties.All)]
        //public bool MinimumFixed
        //{
        //    get;
        //    set;
        //}

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1)]
        [Description("増減値。int型で再定義しています。")]
        public new int Increment
        {
            get { return Convert.ToInt32(base.Increment); }
            set
            {
                base.Increment   = Convert.ToDecimal(value);
                _fSmallIncrement = (double)value;

                if ( base.Increment<1 )
                    base.Increment = 1;
            }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10)]
        [Description("大増減値。int型で再定義しています。")]
        public new int LargeIncrement
        {
            get { return Convert.ToInt32(base.LargeIncrement); }
            set
            {
                base.LargeIncrement = Convert.ToDecimal(value);
                _fLargeIncrement    = (double)value;

                if ( base.LargeIncrement<1 )
                    base.LargeIncrement = 1;
            }
        }

        /// <summary>
        /// 増減値。
        /// Floating point number to prevent underflowing by range scaling
        /// </summary>
        [Browsable(false)]
        public double IncrementReal
        {
            get { return _fSmallIncrement; }
            set
            {
                _fSmallIncrement = value;
                base.Increment   = Convert.ToDecimal(value);

                if ( base.Increment<1 )
                    base.Increment = 1;
            }
        }

        /// <summary>
        /// 大増減値。
        /// Floating point number to prevent underflowing by range scaling
        /// </summary>
        [Browsable(false)]
        public double LargeIncrementReal
        {
            get { return _fLargeIncrement; }
            set
            {
                _fLargeIncrement    = value;
                base.LargeIncrement = Convert.ToDecimal(value);

                if ( base.LargeIncrement<1 )
                    base.LargeIncrement = 1;
            }
        }

        /// <summary>
        /// 範囲プロパティ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IntRangeProperty RangeProperty
        {
            get
            {
                return new IntRangeProperty( this.Minimum,
                                             this.Maximum,
                                             this.IncrementReal,
                                             this.LargeIncrementReal );
            }
        }

        /// <summary>
        /// 値を増減させる。
        /// </summary>
        protected override void IncrementValue( bool up,
                                                decimal increment,
                                                bool startValueChanging )
        {
            base.IncrementValue( up,
                                 System.Math.Round(base.Increment, 0),
                                 startValueChanging );
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int DecimalPlaces
        {
            get { return base.DecimalPlaces; }
            set { base.DecimalPlaces = value; }
        }
        #endregion
    }
    #endregion

    #region FloatUpDown
    /// <summary>
    /// 実数値アップダウンクラス。
    /// </summary>
    public class FloatUpDown : UINumericUpDown
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatUpDown()
        {
            base.DecimalPlaces = 1;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Value
        {
            get { return Convert.ToSingle(base.Value); }
            set
            {
                // 範囲拡張
                if (this.AutoRangeExpantion)
                {
                    FloatRangeProperty range = this.RangeProperty;
                    if (!range.Contains(value))
                    {
                        while (!range.Contains(value))
                        {
                            if (!range.Scale(true))
                            {
                                break;
                            }
                        }
                        this.Minimum        = range.Minimum;
                        this.Maximum        = range.Maximum;
                        this.Increment      = range.ChangeS;
                        this.LargeIncrement = range.ChangeL;
                    }
                }

                // 値を更新
                SetValue( Convert.ToDecimal(value) );

                // 小数部桁数に応じて表示桁数を変更する
                int decimalPlaces = GetDecimalPlaces( GetCurrentValue() );
                if ( decimalPlaces == 0 )
                    decimalPlaces = 1;

                if ( base.DecimalPlaces != decimalPlaces )
                    base.DecimalPlaces = decimalPlaces;
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("最小値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Minimum
        {
            get { return Convert.ToSingle(base.Minimum); }
            set { base.Minimum = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100.0f)]
        [Description("最大値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Maximum
        {
            get { return Convert.ToSingle(base.Maximum); }
            set { base.Maximum = Convert.ToDecimal(value); }
        }

        ///// <summary>
        ///// 最小値Fixed。
        ///// </summary>
        //[DefaultValue(false)]
        //[Description("最小値。float型で再定義おできない")]
        //[RefreshProperties(RefreshProperties.All)]
        //public bool MinimumFixed
        //{
        //    get;
        //    set;
        //}

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1.0f)]
        [Description("増減値。float型で再定義しています。")]
        public new float Increment
        {
            get { return Convert.ToSingle(base.Increment); }
            set { base.Increment = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10.0f)]
        [Description("大増減値。float型で再定義しています。")]
        public new float LargeIncrement
        {
            get { return Convert.ToSingle(base.LargeIncrement); }
            set { base.LargeIncrement = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 範囲プロパティ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FloatRangeProperty RangeProperty
        {
            get { return new FloatRangeProperty(
                this.Minimum, this.Maximum, this.Increment, this.LargeIncrement); }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(1)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new int DecimalPlaces
        {
            get { return base.DecimalPlaces; }
            set { base.DecimalPlaces = value; }
        }
        #endregion

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnValueChanged(EventArgs e)
        {
            // 小数部桁数に応じて表示桁数を変更する
            int decimalPlaces = GetDecimalPlaces( GetCurrentValue() );
            if ( decimalPlaces == 0 )
                decimalPlaces = 1;

            if ( base.DecimalPlaces != decimalPlaces )
                base.DecimalPlaces = decimalPlaces;

            base.OnValueChanged(e);
        }
    }
    #endregion

    #region IntRangeProperty
    /// <summary>
    /// 整数値範囲プロパティクラス。
    /// </summary>
    public sealed class IntRangeProperty
    {
        // 最小値
        private int _minimum = 0;
        // 最大値
        private int _maximum = 0;

        // Use floating point numbers to prevent underflowing
        // 小変化量
        private double _fRealChangeS = 0;
        // 大変化量
        private double _fRealChangeL = 0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntRangeProperty( int minimum,
                                 int maximum,
                                 double changeS,
                                 double changeL )
        {
            _minimum = minimum;
            _maximum = maximum;

            _fRealChangeS = changeS;
            _fRealChangeL = changeL;

            MinimumFixed = false;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public int Minimum
        {
            get { return _minimum; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public int Maximum
        {
            get { return _maximum; }
        }

        /// <summary>
        /// 最小値不變。
        /// </summary>
        public bool MinimumFixed { get; set; }

        /// <summary>
        /// 小変化量。
        /// </summary>
        public double ChangeS
        {
            get { return _fRealChangeS; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        public double ChangeL
        {
            get { return _fRealChangeL; }
        }

        /// <summary>
        /// 指定した値が含まれるか。
        /// </summary>
        public bool Contains(int value)
        {
            return value >= _minimum && value <= _maximum;
        }

        /// <summary>
        /// スケーリング。
        /// </summary>
        public bool Scale(bool scaleUp)
        {
            //--------------------------------------------------------------------------
            // FloatRangeProperty とまったく同じなので変更時は両方修正すること！
            //--------------------------------------------------------------------------
            int calcMinimum = _minimum;
            int calcMaximum = _maximum;

            // 拡大
            if (scaleUp)
            {
                const int smallEnd = -10000000; // 1000万
                const int largeEnd =  10000000; // 1000万

                if (!MinimumFixed)
                    calcMinimum *= 10;

                calcMaximum *= 10;

                // 限界値で止める
                if (_minimum < 0 && calcMinimum < smallEnd) { calcMinimum = smallEnd; }
                if (_minimum > 0 && calcMinimum > largeEnd) { calcMinimum = largeEnd; }
                if (_maximum < 0 && calcMaximum < smallEnd) { calcMaximum = smallEnd; }
                if (_maximum > 0 && calcMaximum > largeEnd) { calcMaximum = largeEnd; }

                // 設定
                if (calcMinimum != calcMaximum &&
                    (_minimum != calcMinimum || _maximum != calcMaximum))
                {
                    _minimum = calcMinimum;
                    _maximum = calcMaximum;
                    _fRealChangeS *= 10.0;
                    _fRealChangeL *= 10.0;
                    return true;
                }
            }
            // 縮小
            else
            {
                const int smallEnd = -1;
                const int largeEnd =  1;

                if (!MinimumFixed)
                    calcMinimum /= 10;

                calcMaximum /= 10;

                // 限界値で止める
                if (_minimum < 0 && calcMinimum > smallEnd) { calcMinimum = smallEnd; }
                if (_minimum > 0 && calcMinimum < largeEnd) { calcMinimum = largeEnd; }
                if (_maximum < 0 && calcMaximum > smallEnd) { calcMaximum = smallEnd; }
                if (_maximum > 0 && calcMaximum < largeEnd) { calcMaximum = largeEnd; }

                // 設定
                if (calcMinimum != calcMaximum &&
                    (_minimum != calcMinimum || _maximum != calcMaximum))
                {
                    _minimum = calcMinimum;
                    _maximum = calcMaximum;
                    _fRealChangeS /= 10.0;
                    _fRealChangeL /= 10.0;
                    return true;
                }
            }
            return false;
        }
    }
    #endregion

    #region FloatRangeProperty
    /// <summary>
    /// 実数値範囲プロパティクラス。
    /// </summary>
    public sealed class FloatRangeProperty
    {
        // 最小値
        private float _minimum = 0;
        // 最大値
        private float _maximum = 0;
        // 小変化量
        private float _changeS = 0;
        // 大変化量
        private float _changeL = 0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatRangeProperty(float minimum, float maximum, float changeS, float changeL)
        {
            _minimum = minimum;
            _maximum = maximum;
            _changeS = changeS;
            _changeL = changeL;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public float Minimum
        {
            get { return _minimum; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public float Maximum
        {
            get { return _maximum; }
        }

        /// <summary>
        /// 小変化量。
        /// </summary>
        public float ChangeS
        {
            get { return _changeS; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        public float ChangeL
        {
            get { return _changeL; }
        }

        /// <summary>
        /// 指定した値が含まれるか。
        /// </summary>
        public bool Contains(float value)
        {
            return value >= _minimum && value <= _maximum;
        }

        /// <summary>
        /// スケーリング。
        /// </summary>
        public bool Scale(bool scaleUp)
        {
            //--------------------------------------------------------------------------
            // IntRangeProperty とまったく同じなので変更時は両方修正すること！
            //--------------------------------------------------------------------------
            float calcMinimum = _minimum;
            float calcMaximum = _maximum;

            // 拡大
            if (scaleUp)
            {
                const int smallEnd = -10000000; // 1000万
                const int largeEnd =  10000000; // 1000万

                calcMinimum *= 10;
                calcMaximum *= 10;

                // 限界値で止める
                if (_minimum < 0 && calcMinimum < smallEnd) { return false; }
                if (_minimum > 0 && calcMinimum > largeEnd) { return false; }
                if (_maximum < 0 && calcMaximum < smallEnd) { return false; }
                if (_maximum > 0 && calcMaximum > largeEnd) { return false; }

                // 設定
                if (calcMinimum != calcMaximum &&
                    (_minimum != calcMinimum || _maximum != calcMaximum))
                {
                    _minimum = calcMinimum;
                    _maximum = calcMaximum;
                    _changeS *= 10;
                    _changeL *= 10;
                    return true;
                }
            }
            // 縮小
            else
            {
                const int smallEnd = -1;
                const int largeEnd =  1;

                calcMinimum /= 10;
                calcMaximum /= 10;

                // 限界値で止める
                if (_minimum < 0 && calcMinimum > smallEnd) { return false; }
                if (_minimum > 0 && calcMinimum < largeEnd) { return false; }
                if (_maximum < 0 && calcMaximum > smallEnd) { return false; }
                if (_maximum > 0 && calcMaximum < largeEnd) { return false; }

                // 設定
                if (calcMinimum != calcMaximum &&
                    (_minimum != calcMinimum || _maximum != calcMaximum))
                {
                    _minimum = calcMinimum;
                    _maximum = calcMaximum;
                    _changeS /= 10;
                    _changeL /= 10;
                    return true;
                }
            }
            return false;
        }
    }
    #endregion
}
#endif
