﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// メッセージリストを表示するUIです.
    /// </summary>
    public partial class UIMessageListView : UIUserControl
    {
        /// <summary>
        /// メッセージのサイズを更新するときtrue
        /// </summary>
        private bool doUpdateSize;

        /// <summary>
        /// メッセージのサイズ.
        /// </summary>
        private Size messageSize;

        /// <summary>
        /// メッセージリスト.
        /// </summary>
        private IEnumerable<Tuple<string, Color, Font>> messageList;

        /// <summary>
        /// 行間隔.
        /// </summary>
        private int lineSpacing;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public UIMessageListView()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// メッセージリストを取得または設定します.
        /// </summary>
        public IEnumerable<Tuple<string, Color, Font>> MessageList
        {
            get
            {
                return this.messageList;
            }

            set
            {
                // リストの内容だけが変化したときも処理するため
                // valueのチェックは行わない
                this.messageList = value;
                this.doUpdateSize = true;

                this.Invalidate();
                this.PerformLayout();
            }
        }

        /// <summary>
        /// 行間隔を取得または設定します.
        /// </summary>
        public int LineSpacing
        {
            get
            {
                return this.lineSpacing;
            }

            set
            {
                if (value != this.lineSpacing)
                {
                    this.lineSpacing = value;
                    this.doUpdateSize = true;

                    this.Invalidate();
                    this.PerformLayout();
                }
            }
        }

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.doUpdateSize)
            {
                this.UpdateMessageSize();
            }

            return new Size(
                this.Padding.Horizontal + this.messageSize.Width,
                this.Padding.Vertical + this.messageSize.Height);
        }

        /// <summary>
        /// Handle paint event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            // メッセージが登録されていないとき描画終了
            if (this.messageList == null)
            {
                return;
            }

            Graphics g = e.Graphics;
            SolidBrush brush = new SolidBrush(this.ForeColor);
            float posX = this.Padding.Left;
            float posY = this.Padding.Top;

            // メッセージを描画
            foreach (Tuple<string, Color, Font> message in this.messageList)
            {
                if (string.IsNullOrEmpty(message.Item1))
                {
                    continue;
                }

                Font font = message.Item3 != null ? message.Item3 : this.Font;

                SizeF strSize = g.MeasureString(message.Item1, font);

                brush.Color = message.Item2;
                g.DrawString(message.Item1, font, brush, posX, posY);

                posY += (float)Math.Ceiling(strSize.Height) + this.LineSpacing;
            }

            brush.Dispose();
        }

        /// <summary>
        /// メッセージサイズを更新します.
        /// </summary>
        private void UpdateMessageSize()
        {
            this.doUpdateSize = false;
            this.messageSize = new Size();

            if (this.messageList == null)
            {
                return;
            }

            Graphics g = this.CreateGraphics();

            // メッセージの文字サイズを計算
            foreach (Tuple<string, Color, Font> message in this.messageList)
            {
                if (string.IsNullOrEmpty(message.Item1))
                {
                    continue;
                }

                Font font = message.Item3 != null ? message.Item3 : this.Font;

                SizeF strSize = g.MeasureString(message.Item1, font);
                this.messageSize.Width   = Math.Max((int)Math.Ceiling(strSize.Width), this.messageSize.Width);
                this.messageSize.Height += (int)Math.Ceiling(strSize.Height) + this.LineSpacing;
            }

            if (this.messageSize.Height > 0)
            {
                this.messageSize.Height -= this.LineSpacing;
            }

            g.Dispose();
        }
    }
}
