﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An extended ProjectTreeNodeBase for Workspace implementation.
    /// </summary>
    public class WorkspaceTreeNode : ProjectTreeNodeBase
    {
        /// <summary>
        /// コンテキストメニューに追加したプリセットアイテム.
        /// </summary>
        private List<ToolStripItem> addedItems = new List<ToolStripItem>();

        /// <summary>
        /// Initializes the WorkspaceTreeNode instance.
        /// </summary>
        public WorkspaceTreeNode()
        {
            this.EnableDisplayedIcon = false;
            this.IsDrawModifyMark = false;
        }

        /// <summary>
        /// Render the node of workspace.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal override bool Render(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // The tree node might already be removed between the draw message is dispatched
            // and the actual rendering.
            // (the message dispatches, and the rendering happens on the next application loop)
            if (((TreeNode)this).Parent == null &&
                (this.TreeView == null || this.TreeView.Nodes.Contains(this) == false))
            {
                return true;
            }

            // ノード作成時に無効な領域の描画呼び出しが発生する
            if (bounds.Width == 0 || bounds.Height == 0)
            {
                return true;
            }

            this.DrawBackground(bounds, state, gr);

            int locationX = ProjectTreeNodeBase.ShowButtonWidth +
                            ProjectTreeNodeBase.ShowButtonMarginRight +
                            bounds.Left;

            locationX = this.DrawHierarchy(bounds, state, gr, locationX);

            locationX = this.DrawNodeTypeIcon(bounds, state, gr, locationX, Properties.Resources.Icon_Effect);

            this.DrawText(bounds, state, gr, locationX, Properties.Resources.WorkspaceRootNodeName, this.NodeFont);

            return true;
        }

        /// <summary>
        /// コンテキストメニューのセットアップ.
        /// </summary>
        protected override void SetupContextMenu()
        {
            var docMenu = this.ContextMenuStrip as UIContextMenuStrip;

            WorkspaceRootViewModel workspaceRootViewModel = null;
            var dataContext = this.DataContext as IHierarchyObject;
            if (dataContext == null)
            {
                Logger.Log(
                    LogLevels.Error,
                    "EmitterSetTreeNode.SetupContextMenu : can not to get WorkspaceRootViewModel.");
                return;
            }

            workspaceRootViewModel = dataContext.FindFarthestParentOfType<WorkspaceRootViewModel>();
            WorkspaceController wc = this.GetWorkspaceController();

            // プリセット対応
            var addPresetMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddFromPreset,
                Image = Properties.Resources.Icon_Effect_Preset,
                ImageTransparentColor = Color.White,
            };
            var dummyItem = new UIToolStripMenuItem();
            this.addedItems.Add(dummyItem);
            addPresetMenu.DropDownItems.Add(dummyItem);

            // エミッタセットを追加.
            var addEmitterSetMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddEmitterSet,
                Image = Properties.Resources.Icon_EmitterSet,
                ImageTransparentColor = Color.White,
                ShortcutKeys = Keys.Control | Keys.N,
            };
            addEmitterSetMenu.Click += (s, e) => wc.CreateNewEmitterSet();

            // 閉じる
            var closeMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameCloseAll,
                ShortcutKeys = Keys.Control | Keys.Alt | Keys.W,
            };
            closeMenu.Click += (ss, ee) => workspaceRootViewModel.OnCloseAll(new object[] { null, false });

            // メニューオープン時にプリセットが有効か無効かをチェック
            docMenu.Opening += (s, e) =>
            {
                var hasPreset = PresetUtility.HasPreset;
                if (hasPreset)
                {
                    PresetUtility.SetupPresetMenu(
                        addPresetMenu,
                        this.addedItems,
                        false,
                        PresetMode.EmitterSet);
                }

                addPresetMenu.Visible = hasPreset;
                addEmitterSetMenu.Enabled = workspaceRootViewModel.CanCreateNewEmitterSet;
                closeMenu.Enabled = workspaceRootViewModel.CanRemoveAnyNode;
            };

            // コンテキストメニューをセット.
            docMenu.Items.AddRange(
                new ToolStripItem[]
                {
                    addPresetMenu,
                    addEmitterSetMenu,
                    closeMenu,
                });
        }
    }
}
