﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An IItemContainerSelector class that creates item containers for the project tree view.
    /// </summary>
    public class ProjectTreeItemContainerSelector : IItemContainerSelector
    {
        /// <summary>Dictionary to cache the created item containers.</summary>
        private static readonly Dictionary<object, IControl> ItemContainerCache =
            new Dictionary<object, IControl>();

        /// <summary>
        /// Instances tree node item containers for each node view model.
        /// </summary>
        /// <param name="dataItem">View model related to
        /// the node item container to create.</param>
        /// <returns>Returns an appropriate item container for the given view model.</returns>
        public IControl SelectItemContainer(object dataItem)
        {
            // First find cached item container with the data item.
            IControl itemContainer = null;
            if (ItemContainerCache.TryGetValue(dataItem, out itemContainer) == true)
            {
                return itemContainer;
            }

            // This is a new data item, create an item container for it.
            itemContainer = this.CreateItemContainer(dataItem);
            if (itemContainer == null)
            {
                return null;
            }

            // Save the created item container to our cache.
            ItemContainerCache[dataItem] = itemContainer;

            return itemContainer;
        }

        /// <summary>
        /// Create item container for the given data item.
        /// </summary>
        /// <param name="dataItem">The data item.</param>
        /// <returns>The item container.</returns>
        private IControl CreateItemContainer(object dataItem)
        {
            var workspace = dataItem as WorkspaceViewModel;
            if (workspace != null)
            {
                return new WorkspaceTreeNode();
            }

            var viewer = dataItem as ViewerViewModel;
            if (viewer != null)
            {
                return new ViewerTreeNode();
            }

            var model = dataItem as ModelViewModel;
            if (model != null)
            {
                return new ModelTreeNode();
            }

            var emitterSet = dataItem as EmitterSetViewModel;
            if (emitterSet != null)
            {
                return new EmitterSetTreeNode();
            }

            var emitter = dataItem as EmitterViewModel;
            if (emitter != null)
            {
                return new EmitterTreeNode();
            }

            var field = dataItem as FieldViewModel;
            if (field != null)
            {
                var node = new FieldTreeNode();
                node.SetIcon(field);
                return node;
            }

            var preview = dataItem as PreviewViewModel;
            if (preview != null)
            {
                return new PreviewTreeNode();
            }

            var reservedShader = dataItem as ReservedShaderNodeViewModel;
            if (reservedShader != null)
            {
                return new ReservedShaderTreeNode();
            }

            var customAction = dataItem as CustomActionViewModel;
            if (customAction != null)
            {
                return new CustomActionTreeNode();
            }

            var emitterExtParams = dataItem as EmitterExtParamsViewModel;
            if (emitterExtParams != null)
            {
                return new EmitterExtParamsTreeNode();
            }

            return null;
        }
    }
}
