﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.TreeNodes
{
    /// <summary>
    /// An extended ProjectTreeNodeBase for emitter set implementation.
    /// </summary>
    public class EmitterSetTreeNode : ProjectTreeNodeBase
    {
        /// <summary>
        /// リンク状態です。
        /// </summary>
        private bool isLinked;

        /// <summary>
        /// コンテキストメニューに追加したプリセットアイテム.
        /// </summary>
        private List<ToolStripItem> addedItems = new List<ToolStripItem>();

        /// <summary>
        /// Initializes the EmitterSetTreeNode instance.
        /// </summary>
        public EmitterSetTreeNode()
        {
            this.CanRename = true;

            this.IsDrawModifyMark = true;

            this.AddBinding("Text", "Name");
            this.AddBinding("IsDisplayed", "Displayed");
            this.AddBinding("IsLinked", "Linked");
        }

        /// <summary>
        /// リンク状態を取得または設定します。
        /// </summary>
        public bool IsLinked
        {
            get
            {
                return this.isLinked;
            }

            set
            {
                this.LogicalTreeElementExtender.SetValue(ref this.isLinked, value);

                if (this.TreeView != null)
                {
                    this.TreeView.Invalidate();
                }
            }
        }

        /// <summary>
        /// Render the node of viewer.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <returns>Returns true if it self render, or false to tall parent
        /// TreeView to perform default node rendering.</returns>
        protected internal override bool Render(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // The tree node might already be removed between the draw message is dispatched
            // and the actual rendering.
            // (the message dispatches, and the rendering happens on the next application loop)
            if (((TreeNode)this).Parent == null &&
                (this.TreeView == null || this.TreeView.Nodes.Contains(this) == false))
            {
                return true;
            }

            // ノード作成時に無効な領域の描画呼び出しが発生する
            if (bounds.Width == 0 || bounds.Height == 0)
            {
                return true;
            }

            // 背景を描画
            this.DrawBackground(bounds, state, gr);

            // 目のアイコンを描画
            this.DrawDisplayedIcon(bounds, state, gr);

            // 階層情報の表示位置を取得
            int locationX = ProjectTreeNodeBase.ShowButtonWidth +
                            ProjectTreeNodeBase.ShowButtonMarginRight +
                            bounds.Left;

            // 階層情報を描画
            locationX = this.DrawHierarchy(bounds, state, gr, locationX);

            // ノードアイコンを描画
            Bitmap icon = this.IsLinked ? Properties.Resources.Icon_BindMark : Properties.Resources.Icon_EmitterSet;
            locationX = this.DrawNodeTypeIcon(bounds, state, gr, locationX, icon);

            // 編集マークを描画
            locationX = this.DrawModifyMark(bounds, state, gr, locationX);

            // テキストを描画
            this.DrawText(bounds, state, gr, locationX, this.Text, this.NodeFont);

            return true;
        }

        /// <summary>
        /// コンテキストメニューのセットアップ.
        /// </summary>
        protected override void SetupContextMenu()
        {
            var docMenu = this.ContextMenuStrip as UIContextMenuStrip;

            WorkspaceRootViewModel workspaceRootViewModel = null;
            var dataContext = this.DataContext as IHierarchyObject;
            if (dataContext == null)
            {
                Logger.Log(
                    LogLevels.Error,
                    "EmitterSetTreeNode.SetupContextMenu : can not to get WorkspaceRootViewModel.");
                return;
            }

            workspaceRootViewModel = dataContext.FindFarthestParentOfType<WorkspaceRootViewModel>();
            WorkspaceController wc = this.GetWorkspaceController();

            // プリセット対応
            var addPresetMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddFromPreset,
                Image = Properties.Resources.Icon_EmitterSet_Preset,
                ImageTransparentColor = Color.White,
            };
            var dummyItem = new UIToolStripMenuItem();
            this.addedItems.Add(dummyItem);
            addPresetMenu.DropDownItems.Add(dummyItem);

            // メニューオープン時にプリセットが有効か無効かをチェック
            docMenu.Opening += (s, e) =>
            {
                var hasPreset = PresetUtility.HasPreset;
                if (hasPreset)
                {
                    PresetUtility.SetupPresetMenu(
                        addPresetMenu,
                        this.addedItems,
                        false,
                        PresetMode.EmitterSetAndEmitter);
                }

                addPresetMenu.Visible = hasPreset;
            };

            // エミッタセットを追加.
            var addEmitterSetMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddEmitterSet,
                Image = Properties.Resources.Icon_EmitterSet,
                ImageTransparentColor = Color.White,
                ShortcutKeys = Keys.Control | Keys.N,
            };
            addEmitterSetMenu.Click += (s, e) => wc.CreateNewEmitterSet();

            // エミッタを追加.
            var addEmitterMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddEmitter,
                Image = Properties.Resources.Icon_Emitter_Simple,
                ImageTransparentColor = Color.White,
                ShortcutKeys = Keys.Control | Keys.Shift | Keys.N,
            };
            addEmitterMenu.Click += (s, e) => wc.CreateNewChildEmitter();

            // プレビューを追加.
            var addPreviewMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameAddPreview,
                Image = Properties.Resources.Icon_PreviewSetting,
                ImageTransparentColor = Color.White,
            };
            addPreviewMenu.Bindings.Add(new UIControls.DataBinding.Binder(
                addPreviewMenu, "Enabled", "CanCreateNewPreview"));
            addPreviewMenu.DataContext = workspaceRootViewModel;
            addPreviewMenu.Click += (s, e) => wc.CreateNewPreview();

            // --セパレータ

            // エミッタを貼り付け
            var pasteEmitterMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNamePasteEmitter,
                ShortcutKeys = Keys.Control | Keys.V,
                Image = Properties.Resources.Icon_Paste,
            };

            // 複製
            var duplicateMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameDuplicate,
                ShortcutKeys = Keys.Control | Keys.D,
                Image = Properties.Resources.Icon_DuplicateNode,
            };

            // 名前を変更
            var renameMenu = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuItemNameRename,
                ShortcutKeys = Keys.F2,
                Image = Properties.Resources.Icon_ChangeName,
                ImageTransparentColor = Color.White,
            };
            renameMenu.Click += (s, e) =>
            {
                var treeView = this.TreeView as WorkspaceTreeView;
                if (treeView != null)
                {
                    treeView.BeginRename();
                }
            };

            // 保存
            var saveMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameSave,
                ShortcutKeys = Keys.Control | Keys.S,
                Image = Properties.Resources.Icon_FileSave,
            };

            saveMenu.Click += (ss, ee) => workspaceRootViewModel.OnFileSave(false);

            saveMenu.Bindings.Add(
                new UIControls.DataBinding.Binder(
                    saveMenu, "Enabled", "CanSaveSelectedNode"));
            saveMenu.DataContext = workspaceRootViewModel;

            // 別名で保存
            var saveAsMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameSaveAs,
                ShortcutKeys = Keys.Control | Keys.Shift | Keys.S,
                Image = Properties.Resources.Icon_SaveAs,
            };

            saveAsMenu.Click += (ss, ee) => workspaceRootViewModel.OnFileSave(true);

            saveAsMenu.Bindings.Add(
                new UIControls.DataBinding.Binder(
                    saveAsMenu, "Enabled", "CanSaveAsSelectedNode"));
            saveAsMenu.DataContext = workspaceRootViewModel;

            // 閉じる
            var closeMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameClose,
                ShortcutKeys = Keys.Control | Keys.W,
                Image = Properties.Resources.Icon_Delete,
            };
            closeMenu.Click += (ss, ee) => wc.Remove();

            // --セパレータ

            // ファイルを開き直す
            var reopenEsetMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameReopenEset,
                Image = Properties.Resources.TB_FileOpen,
            };

            var reopenMenuBinder = new Binder(reopenEsetMenu, "Enabled", "CanReopenSelectedNode");
            reopenEsetMenu.Bindings.Add(reopenMenuBinder);
            reopenEsetMenu.DataContext = workspaceRootViewModel;
            reopenEsetMenu.Click += (ss, ee) => WorkspaceRootViewModel.Instance.OnFileReopen(null);

            // エクスプローラで開く.
            var openExplorerMenu = new UIToolStripMenuItem
            {
                Text = Properties.Resources.MenuItemNameOpenExplorer,
                ShortcutKeys = Keys.Control | Keys.E,
                Image = Properties.Resources.Icon_Explorer,
            };

            // コンテキストメニューをセット.
            docMenu.Items.AddRange(
                new ToolStripItem[]
                {
                    addPresetMenu,
                    addEmitterSetMenu,
                    addEmitterMenu,
                    addPreviewMenu,
                    new ToolStripSeparator(),
                    pasteEmitterMenu,
                    duplicateMenu,
                    renameMenu,
                    saveMenu,
                    saveAsMenu,
                    closeMenu,
                    new ToolStripSeparator(),
                    reopenEsetMenu,
                    openExplorerMenu,
                });

            // !!! メニューへ追加した後でないと、正常に動作しない.

            // エミッタセットの複製
            duplicateMenu
               .AddBinding("Executable", "NodeDuplicateExecutable")
               .UpdateElement();
            Binder duplicateMenuBinder = duplicateMenu.AddBinding("Enabled", "CanCreateNewEmitterSet");
            this.EmitterSetContextMenuUpdaters += duplicateMenuBinder.UpdateElement;
            duplicateMenuBinder.UpdateElement();

            // エクスプローラで開く
            openExplorerMenu
                .AddBinding("Executable", "NodeOpenExlorerExecutable")
                .UpdateElement();

            // エミッタを貼り付け
            pasteEmitterMenu
                .AddBinding("Executable", "NodePasteEmitterExecutable")
                .UpdateElement();

            docMenu.AddBinding("ContextMenuOpeningExecutable", "EvaluateMenuItemsExecutable").UpdateElement();

            var addEmitterBinder = addEmitterMenu.AddBinding("Enabled", "CanCreateNewEmitter");
            var addEmitterSetBinder = addEmitterSetMenu.AddBinding("Enabled", "CanCreateNewEmitterSet");
            this.EmitterContextMenuUpdaters += addEmitterBinder.UpdateElement;
            this.EmitterSetContextMenuUpdaters += addEmitterSetBinder.UpdateElement;
            addEmitterBinder.UpdateElement();
            addEmitterSetBinder.UpdateElement();
        }

        /// <summary>
        /// Draw the background.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        protected override void DrawBackground(Rectangle bounds, TreeNodeStates state, Graphics gr)
        {
            // 描画領域全体をクリア。
            gr.FillRectangle(Brushes.White, bounds);

            // 表示アイコンの裏は背景を描画しない。
            bounds.X += ProjectTreeNodeBase.ShowButtonWidth;

            if ((state & TreeNodeStates.Selected) != 0 ||
                (state & TreeNodeStates.Focused) != 0)
            {
                // 選択時の背景を描画。
                gr.FillRectangle(Brushes.LightSteelBlue, bounds);
            }
            else if (this.IsDragging)
            {
                // ドラッグ中の背景を描画。
                gr.FillRectangle(Brushes.LightSkyBlue, bounds);
            }
            else
            {
                // デフォルトの背景を描画。
                LinearGradientBrush gradientBrush =
                    new LinearGradientBrush(
                        bounds,
                        Color.FromArgb(230, 230, 230),
                        Color.FromArgb(255, 255, 255),
                        LinearGradientMode.Vertical);

                gr.FillRectangle(gradientBrush, bounds);

                gradientBrush.Dispose();
            }
        }

        /// <summary>
        /// Draw the text.
        /// </summary>
        /// <param name="bounds">The rectangular region in whic to render.</param>
        /// <param name="state">The current node state.</param>
        /// <param name="gr">The graphics instance to render to.</param>
        /// <param name="locationX">The x-coordinate.</param>
        /// <param name="text">The text.</param>
        /// <param name="font">The font.</param>
        protected override void DrawText(
            Rectangle bounds,
            TreeNodeStates state,
            Graphics gr,
            int locationX,
            string text,
            Font font)
        {
            Brush textBrush = Brushes.Black;

            // フォントを設定。
            if (font == null)
            {
                font = TreeView.Font;
            }

            font = new Font(font, FontStyle.Bold);

            // ノード選択時はテキストの色を変える。
            if ((state & TreeNodeStates.Selected) != 0 ||
                (state & TreeNodeStates.Focused) != 0)
            {
                textBrush = Brushes.DarkBlue;
            }

            // テキストの描画位置を計算。
            SizeF textSize = gr.MeasureString(text, font);
            int locationY = this.Bounds.Top + ((this.Bounds.Height - (int)textSize.Height) / 2);

            gr.DrawString(
                text,
                font,
                textBrush,
                locationX,
                locationY);

            font.Dispose();
        }
    }
}
