﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Input;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using MouseEventArgs = System.Windows.Forms.MouseEventArgs;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// テクスチャ画像ビュー
    /// </summary>
    public class TextureImageView : ScrollableControl
    {
        /// <summary>
        ///     表示倍率テーブル
        /// </summary>
        private static readonly float[] MagnifyTable =
        {
            0.0100f, 0.0200f, 0.0300f, 0.0400f,
            0.0500f, 0.0625f, 0.0833f, 0.1250f,
            0.1660f, 0.2500f, 0.3330f, 0.5000f,
            0.6660f, 1.0000f, 2.0000f, 3.0000f,
            4.0000f, 5.0000f, 6.0000f, 7.0000f,
            8.0000f, 12.0000f, 16.0000f
        };

        /// <summary>
        /// カラーイメージ（ミップマップ付き）
        /// </summary>
        private readonly Dictionary<MipmapOrientationType, Bitmap> colorImageMM = new Dictionary<MipmapOrientationType, Bitmap>();

        /// <summary>
        /// アルファイメージ（ミップマップ付き）
        /// </summary>
        private readonly Dictionary<MipmapOrientationType, Bitmap> alphaImageMM = new Dictionary<MipmapOrientationType, Bitmap>();

        /// <summary>
        /// ミップマップ部表示フラグ
        /// </summary>
        private bool isShowMipmap;

        /// <summary>
        /// ミップマップ方向
        /// </summary>
        private MipmapOrientationType mipmapOrientation = MipmapOrientationType.Horizontal;

        /// <summary>
        /// チャンネル
        /// </summary>
        private ColorChannelFlags channel;

        /// <summary>
        /// 表示する深さ-1で全部表示
        /// </summary>
        private int depthIndex;

        /// <summary>
        /// 最後にポイントしたピクセル
        /// </summary>
        private Point lastPointedPixel = new Point(-2, -2);

        /// <summary>
        /// カラーイメージ
        /// </summary>
        private Bitmap colorImage;

        /// <summary>
        /// アルファイメージ
        /// </summary>
        private Bitmap alphaImage;

        /// <summary>
        /// 前ステップのカーソル位置
        /// </summary>
        private Point beforeLocation = new Point(-1, -1);

        /// <summary>
        /// 中ボタンの押し下げ状態
        /// </summary>
        private bool isMiddleDown = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="targetFilePath">対象テクスチャファイルパス</param>
        /// <param name="depthIndex">深さ</param>
        public TextureImageView(string targetFilePath, int depthIndex)
        {
            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;

            this.Magnify = 1.0f;
            this.DoubleBuffered = true;

            this.depthIndex = depthIndex;

            this.Target = BusinessLogic.Manager.TextureManager.Instance.LoadTexture(targetFilePath, true).TextureData;
            if (this.Target != null)
            {
                // this.channel = this.Target.HasAlpha ? ColorChannelFlags.Rgba : ColorChannelFlags.Rgb;

                // イメージ作成
                this.UpdateImages();

                // 初期設定
                this.InitSetteings();
            }
        }

        /// <summary>
        /// シンプルなビットマップを表示するためのコンストラクタ
        /// </summary>
        /// <param name="bitmap">対象ビットマップ</param>
        public TextureImageView(Bitmap bitmap)
        {
            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;

            this.Magnify = 1.0f;
            this.DoubleBuffered = true;

            this.depthIndex = -1;

            if (bitmap != null)
            {
                this.Target = new TextureData();
                this.Target.MipmapCount = 1;
            }

            if (this.Target != null)
            {
                this.colorImage = bitmap;
                this.alphaImage = bitmap;

                // 初期設定
                this.InitSetteings();
            }
        }

        /// <summary>
        /// MagnifyChanged
        /// </summary>
        public event EventHandler MagnifyChanged;

        /// <summary>
        /// ChannelChanged
        /// </summary>
        public event EventHandler ChannelChanged;

        /// <summary>
        /// PixelHint
        /// </summary>
        public event EventHandler<TpBitmapImagePixelHintEventArgs> PixelHint;

        /// <summary>
        /// テクスチャ/パレットビットマップイメージピクセルヒント
        /// </summary>
        public enum TpBitmapImagePixelHint
        {
            /// <summary>無効</summary>
            None,

            /// <summary>ピクセルカラー</summary>
            PixelColor,
        }

        /// <summary>
        /// ミップマップ表示方向
        /// </summary>
        public enum MipmapOrientationType
        {
            /// <summary>
            /// 水平
            /// </summary>
            Horizontal,

            /// <summary>
            /// 垂直
            /// </summary>
            Vertical
        }

        /// <summary>
        /// 対象データ
        /// </summary>
        public TextureData Target
        {
            get;
            private set;
        }

        /// <summary>
        /// チャンネル
        /// </summary>
        public ColorChannelFlags Channel
        {
            get
            {
                return this.channel;
            }

            set
            {
                if (this.channel != value)
                {
                    this.channel = value;
                    this.Invalidate();
                    this.OnChannelChanged();
                }
            }
        }

        /// <summary>
        /// ミップマップ部表示フラグ
        /// </summary>
        public bool ShowMipmap
        {
            get
            {
                return this.isShowMipmap;
            }

            set
            {
                if (this.isShowMipmap != value)
                {
                    this.isShowMipmap = value;

                    // スクロール関連の再設定
                    // Magnify の変更時と同じような処理になる
                    this.Invalidate();
                    this.AutoScrollMinSize = this.RenderingSize;
                    this.AutoScrollPosition = Point.Empty;
                }
            }
        }

        /// <summary>
        /// 描画サイズ
        /// </summary>
        public Size RenderingSize
        {
            get
            {
                // 基本サイズ
                var size = this.RenderingColorImage.Size;

                // 表示倍率をかける（最低でも１ピクセル）
                size.Width = (int)(size.Width * this.Magnify);
                size.Height = (int)(size.Height * this.Magnify);

                if (size.Width <= 1)
                {
                    size.Width = 1;
                }

                if (size.Height <= 1)
                {
                    size.Height = 1;
                }

                return size;
            }
        }

        /// <summary>
        /// ターゲットが有効か
        /// </summary>
        public bool IsTargetEnabled
        {
            get
            {
                return this.Target != null;
            }
        }

        /// <summary>
        /// 深さ
        /// </summary>
        public int DepthIndex
        {
            get
            {
                return this.depthIndex;
            }

            set
            {
                if (this.depthIndex != value)
                {
                    this.depthIndex = value;

                    this.UpdateImages();

                    // スクロール関連の再設定
                    // Magnify の変更時と同じような処理になる
                    this.Invalidate();
                    this.AutoScrollMinSize = this.RenderingSize;
                    this.AutoScrollPosition = Point.Empty;
                }
            }
        }

        /// <summary>
        /// ミップマップ方向
        /// </summary>
        public MipmapOrientationType MipmapOrientation
        {
            get
            {
                return this.mipmapOrientation;
            }

            set
            {
                if (this.mipmapOrientation != value)
                {
                    this.mipmapOrientation = value;

                    // スクロール関連の再設定
                    // Magnify の変更時と同じような処理になる
                    this.Invalidate();
                    this.AutoScrollMinSize = this.RenderingSize;
                    this.AutoScrollPosition = Point.Empty;
                }
            }
        }

        /// <summary>
        /// 表示倍率
        /// </summary>
        public float Magnify { get; set; }

        /// <summary>
        /// クライアント領域の中央
        /// </summary>
        private Point CenterOfClient
        {
            get
            {
                var clientSize = ClientSize;
                return new Point(clientSize.Width / 2, clientSize.Height / 2);
            }
        }

        /// <summary>
        /// 描画領域
        /// </summary>
        private Rectangle RenderingBounds
        {
            get
            {
                // 基本領域
                var clientRectangle = ClientRectangle;
                var rectangle = new Rectangle(this.AutoScrollPosition, this.RenderingSize);

                // 非スクロール時は領域中央に描画
                if (!this.HScroll)
                {
                    rectangle.X = (clientRectangle.Width - rectangle.Width) / 2;
                }

                if (!this.VScroll)
                {
                    rectangle.Y = (clientRectangle.Height - rectangle.Height) / 2;
                }

                return rectangle;
            }
        }

        /// <summary>
        /// 描画カラーイメージ
        /// </summary>
        private Bitmap RenderingColorImage
        {
            get { return this.isShowMipmap ? this.colorImageMM[this.MipmapOrientation] : this.colorImage; }
        }

        /// <summary>
        /// 描画カラーイメージ
        /// </summary>
        private Bitmap RenderingAlphaImage
        {
            get { return this.isShowMipmap ? this.alphaImageMM[this.MipmapOrientation] : this.alphaImage; }
        }

        /// <summary>
        /// 自動スクロール最大サイズ
        /// </summary>
        private Size AutoScrollMaxSize
        {
            get
            {
                var autoScrollMinSize = AutoScrollMinSize;
                var clientSize = ClientSize;

                // 幅
                int width = autoScrollMinSize.Width - clientSize.Width;
                if (width < 0)
                {
                    width = 0;
                }

                // 高さ
                int height = autoScrollMinSize.Height - clientSize.Height;
                if (height < 0)
                {
                    height = 0;
                }

                return new Size(width, height);
            }
        }

        /// <summary>
        /// ミップマップ用 Bitmap 配列を１つの Bitmap に結合します
        /// 下位レベルの画像は左下方向に向かって連続的に配置されます
        /// 結合条件を満たさない場合は NULL が返ります
        /// </summary>
        /// <param name="bitmaps">入力ビットマップ</param>
        /// <param name="mipmapMinSize">ミップマップ最小サイズ</param>
        /// <param name="mipmapOrientation">ミップマップ方向</param>
        /// <returns>結合済みビットマップ</returns>
        public static Bitmap CombineMipmapBitmaps(Bitmap[] bitmaps, Size mipmapMinSize, MipmapOrientationType mipmapOrientation)
        {
            //-----------------------------------------------------------------
            // 準備
            //-----------------------------------------------------------------
            // ２レベル以上必要
            if (bitmaps.Length == 1)
            {
                return (Bitmap)bitmaps[0].Clone();
            }

            // フォーマット確認
            var format = bitmaps[0].PixelFormat;
            switch (format)
            {
                // 対応するフォーマットのみ
                case PixelFormat.Format32bppArgb:
                    break;

                // 対応外のフォーマット
                default:
                    return null;
            }

            // サイズ確認
            var topLvSize = bitmaps[0].Size;
            var totalSize = topLvSize;
            var checkSize = topLvSize;
            for (int i = 1; i < bitmaps.Length; i++)
            {
                checkSize =
                    new Size(
                        Math.Max(mipmapMinSize.Width, checkSize.Width / 2),
                        Math.Max(mipmapMinSize.Height, checkSize.Height / 2));

                // 下位レベルがトップレベルと同じフォーマットか
                System.Diagnostics.Debug.Assert(bitmaps[i].PixelFormat == format, "bitmaps[i].PixelFormat == format");

                if (mipmapOrientation == MipmapOrientationType.Horizontal)
                {
                    totalSize.Width += checkSize.Width;
                }
                else
                {
                    totalSize.Height += checkSize.Height;
                }
            }

            //-----------------------------------------------------------------
            // 作成
            //-----------------------------------------------------------------
            var bitmap = new Bitmap(totalSize.Width, totalSize.Height, PixelFormat.Format32bppArgb);
            using (var bp = Graphics.FromImage(bitmap))
            {
                bp.Clear(Color.FromArgb(0x00, 0x00, 0x00, 0x00));
            }

            CombineBitmaps_32bppArgb(bitmap, bitmaps, mipmapOrientation);
            return bitmap;
        }

        /// <summary>
        /// 中心にフィットさせる
        /// </summary>
        public void FitCenter()
        {
            if (this.RenderingColorImage != null)
            {
                const float Margin = 16.0f;
                var scaleW = (float)(this.Width - Margin) / this.RenderingColorImage.Width;
                var scaleH = (float)(this.Height - Margin) / this.RenderingColorImage.Height;

                var scale = Math.Min(scaleW, scaleH);
                scale = Math.Max(Math.Min(scale, MagnifyTable.Last()), MagnifyTable.First());

                this.ChangeMagnifyCore(scale, this.CenterOfClient);
            }
        }

        /// <summary>
        /// 表示倍率を変更
        /// </summary>
        /// <param name="magnify">倍率</param>
        public void ChangeMagnify(float magnify)
        {
           // 倍率テーブルの範囲内のみ設定可能
            if (magnify >= MagnifyTable[0] && magnify <= MagnifyTable[MagnifyTable.Length - 1])
            {
                // 変更
                if (this.Magnify != magnify)
                {
                    this.ChangeMagnifyCore(magnify, this.CenterOfClient);
                }
            }
        }

        /// <summary>
        /// ズームイン
        /// </summary>
        public void ZoomIn()
        {
            this.ZoomInCore(this.CenterOfClient);
        }

        /// <summary>
        /// ズームアウト
        /// </summary>
        public void ZoomOut()
        {
            this.ZoomOutCore(this.CenterOfClient);
        }

        /// <summary>
        /// チャンネルを全て含んでいるか
        /// </summary>
        /// <param name="srcChannel">チャンネル</param>
        /// <returns>含んでいるか</returns>
        public bool Contains(ColorChannelFlags srcChannel)
        {
            return (srcChannel & ~this.channel) == 0;
        }

        /// <summary>
        /// チャンネルの切り替え
        /// </summary>
        /// <param name="srcChannel">チャンネル</param>
        public void ChangeChannel(ColorChannelFlags srcChannel)
        {
            if (this.Channel == srcChannel)
            {
                // 一致するときは切り替えない
                return;
            }

            if (srcChannel == ColorChannelFlags.Rgb)
            {
                if (this.Contains(ColorChannelFlags.Rgb))
                {
                    // OFF
                    this.Channel &= ~ColorChannelFlags.Rgb;
                }
                else
                {
                    // ON
                    this.Channel |= ColorChannelFlags.Rgb;
                }
            }
            else
            {
                // 反転
                this.Channel ^= srcChannel;
            }

            this.Invalidate();
            this.OnChannelChanged();
        }

        /// <summary>
        /// <see cref="T:System.Windows.Forms.Control"/> とその子コントロールが使用しているアンマネージ リソースを解放します。オプションで、マネージ リソースも解放します。
        /// </summary>
        /// <param name="disposing">マネージ リソースとアンマネージ リソースの両方を解放する場合は true。アンマネージ リソースだけを解放する場合は false。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// MouseDown
        /// </summary>
        /// <param name="e">e</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            this.Focus();
            if (e.Button == MouseButtons.Middle)
            {
                this.isMiddleDown = true;
                this.beforeLocation = e.Location;
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// MouseUp
        /// </summary>
        /// <param name="e">e</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Middle)
            {
                this.isMiddleDown = false;
            }

            base.OnMouseUp(e);
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseMove"/> イベントを発生させます
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.Windows.Forms.MouseEventArgs"/></param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.IsTargetEnabled)
            {
                // ALT+中ボタンドラッグによるズームイン・アウト
                if ((Control.ModifierKeys & Keys.Alt) == Keys.Alt && this.isMiddleDown)
                {
                    var magnify = (int)((this.Magnify * 100.0f) + 0.5f);

                    magnify += (e.X - this.beforeLocation.X) + (this.beforeLocation.Y - e.Y);

                    this.ChangeMagnify((float)magnify / 100.0f);
                    this.beforeLocation = e.Location;
                }

                var pixel = this.GetPixelLocation(e.Location);
                if (pixel != this.lastPointedPixel)
                {
                    var ftxTarget = this.Target as FtxTextureData;

                    // 領域内
                    if (pixel.X != -1 && pixel.Y != -1)
                    {
                        // int mipmapLevel = 0;
                        var mipmapLevel = -1;

                        // ミップマップ表示時の位置補正
                        var location = pixel;
                        if (this.Target.MipmapCount > 1 && this.isShowMipmap)
                        {
                            location = GetPixelLocationInMipmap(
                                pixel,
                                this.colorImage.Size,
                                this.Target.MipmapCount,
                                this.Target.MipmapMinSize,
                                this.mipmapOrientation,
                                out mipmapLevel);
                        }

                        // todo:未実装
#if false
                        if (mipmapLevel != -1)
                        {
                            // ピクセル位置をテクスチャの次元で補正する
                            location =
                                CorrectPixelLocation(
                                    location,
                                    target_.Size,
                                    mipmapLevel,
                                    Target.MipmapMinSize,
                                    target_.Data.texture_info.dimension,
                                    target_.Data.texture_info.depth,
                                    DepthIndex
                                );
                        }
#endif

                        // 補正後領域内
                        if (location.X != -1 && location.Y != -1)
                        {
                            var colorImage = this.RenderingColorImage;
                            var colorC = colorImage.GetPixel(pixel.X, pixel.Y);

                            var alphaImage = this.RenderingAlphaImage;
                            var colorA = alphaImage.GetPixel(pixel.X, pixel.Y);

                            // ピクセルカラー
                            var color = Color.FromArgb(colorA.R, colorC.R, colorC.G, colorC.B);

                            this.OnPixelHint(new TpBitmapImagePixelHintEventArgs(location, color));
                        }
                        else
                        {
                            // 補正後領域外（無効）
                            this.OnPixelHint(TpBitmapImagePixelHintEventArgs.Empty);
                        }
                    }
                    else
                    {
                        // 領域外（無効）
                        this.OnPixelHint(TpBitmapImagePixelHintEventArgs.Empty);
                    }
                }

                this.lastPointedPixel = pixel;
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// <see cref="E:System.Windows.Forms.Control.MouseLeave"/> イベントを発生させます
        /// </summary>
        /// <param name="e">イベント データを格納している <see cref="T:System.EventArgs"/></param>
        protected override void OnMouseLeave(EventArgs e)
        {
            // ピクセルヒントリセット
            this.ResetPixelHint();
            base.OnMouseLeave(e);
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            //-------------------------------------------------------------
            // 背景
            e.Graphics.Clear(Color.Silver);

            if (this.IsTargetEnabled)
            {
                //-------------------------------------------------------------
                // 描画領域
                var renderingBounds = this.RenderingBounds;

                //-------------------------------------------------------------
                // 枠線
                var frame = renderingBounds;
                frame.Inflate(1, 1);
                DrawRectangle(e.Graphics, Pens.Black, frame);

                //-------------------------------------------------------------
                // イメージ
                var colorImage = this.RenderingColorImage;
                var alphaImage = this.RenderingAlphaImage;
                {
#if false
                    // ミップマップまたはキューブマップ描画時
                    if (isShowMipmap
                            // || Target.Data.texture_info.dimension == texture_info_dimensionType.cube
                            // || target_.Data.texture_info.dimension == texture_info_dimensionType.cube_array
                       )
#else
                        // ミップマップまたはキューブマップ描画時
                        if (this.isShowMipmap)
#endif
                        {
                            // チェック模様の下地
                            using (var brush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.LightGray))
                            {
                                e.Graphics.RenderingOrigin = this.AutoScrollPosition;
                                e.Graphics.FillRectangle(brush, renderingBounds);
                            }
                        }

                    // 常にポイントサンプリング
                    var interpolationMode = e.Graphics.InterpolationMode;
                    var pixelOffsetMode = e.Graphics.PixelOffsetMode;
                    e.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor;
                    e.Graphics.PixelOffsetMode = PixelOffsetMode.Half;

                    // 拡大時はサンプリングポイントをずらす
                    var sampling = new PointF(0.0f, 0.0f);

                    if (this.channel == ColorChannelFlags.A)
                    {
                        // アルファのみ
                        e.Graphics.DrawImage(
                                alphaImage,
                                renderingBounds,
                                sampling.X,
                                sampling.Y,
                                alphaImage.Width,
                                alphaImage.Height,
                                GraphicsUnit.Pixel);
                    }
                    else
                    {
                        // カラー
                        // カラー要素別に行列作成
                        using (var ia = new ImageAttributes())
                        {
                            ia.SetColorMatrix(this.GetColorChannelMatrix());
                            e.Graphics.DrawImage(
                                    colorImage,
                                    renderingBounds,
                                    sampling.X,
                                    sampling.Y,
                                    colorImage.Width,
                                    colorImage.Height,
                                    GraphicsUnit.Pixel,
                                    ia);
                        }

                        // アルファ合成
                        if (this.Contains(ColorChannelFlags.A))
                        {
                            using (var ia = new ImageAttributes())
                            {
                                var cm = new ColorMatrix
                                {
                                    Matrix00 = 0.0f,
                                    Matrix10 = 0.0f,
                                    Matrix20 = 0.0f,
                                    Matrix30 = 0.0f,
                                    Matrix40 = 1.0f,

                                    Matrix01 = 0.0f,
                                    Matrix11 = 0.0f,
                                    Matrix21 = 0.0f,
                                    Matrix31 = 0.0f,
                                    Matrix41 = 0.0f,

                                    Matrix02 = 0.0f,
                                    Matrix12 = 0.0f,
                                    Matrix22 = 0.0f,
                                    Matrix32 = 0.0f,
                                    Matrix42 = 0.0f,

                                    Matrix03 = -0.5f,
                                    Matrix13 = 0.0f,
                                    Matrix23 = 0.0f,
                                    Matrix33 = 0.5f,
                                    Matrix43 = 0.0f
                                };

                                ia.SetColorMatrix(cm);
                                e.Graphics.DrawImage(
                                        alphaImage,
                                        renderingBounds,
                                        sampling.X,
                                        sampling.Y,
                                        alphaImage.Width,
                                        alphaImage.Height,
                                        GraphicsUnit.Pixel,
                                        ia);
                            }
                        }
                    }

                    // 補間モード復帰
                    e.Graphics.InterpolationMode = interpolationMode;
                    e.Graphics.PixelOffsetMode = pixelOffsetMode;
                }
            }

            base.OnPaint(e);
        }

        /// <summary>
        /// ホイールイベントをフックするためのプロシージャ
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        protected override void WndProc(ref Message m)
        {
            if ((Control.ModifierKeys & Keys.Alt) == Keys.Alt
                && m.Msg == (int)EffectMaker.Foundation.Win32.WM.WM_MOUSEWHEEL)
            {
                int d = EffectMaker.Foundation.Win32.Utility.HIWORD(m.WParam);
                d = d < 32768 ? 120 : -120;

                var magnify = (int)((this.Magnify * 100.0f) + 0.5f);

                if (d > 0)
                {
                    magnify += 2;
                }
                else
                {
                    magnify -= 2;
                }

                this.ChangeMagnify((float)magnify / 100.0f);
            }
            else
            {
                base.WndProc(ref m);
            }
        }

        /// <summary>
        /// 四角形を描画
        /// </summary>
        /// <param name="g">グラフィックオブジェクト</param>
        /// <param name="pen">ペン</param>
        /// <param name="rect">四角形</param>
        private static void DrawRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// ミップマップイメージ内でのピクセル位置を取得
        /// </summary>
        /// <param name="point">マウス位置</param>
        /// <param name="topLevelSize">トップレベルサイズ</param>
        /// <param name="mipmapCount">ミップマップ数</param>
        /// <param name="mipmapMinSize">ミップマップ最小サイズ</param>
        /// <param name="mipmapOrientation">ミップマップ方向</param>
        /// <param name="mipmapLevel">結果ミップマップレベル</param>
        /// <returns>ピクセル位置</returns>
        private static Point GetPixelLocationInMipmap(Point point, Size topLevelSize, int mipmapCount, Size mipmapMinSize, MipmapOrientationType mipmapOrientation, out int mipmapLevel)
        {
            var imageW = topLevelSize.Width;
            var imageH = topLevelSize.Height;

            if (mipmapOrientation == MipmapOrientationType.Vertical)
            {
                var heightPrev = 0;
                var heightNext = imageH;

                // ミップマップを追う
                for (var i = 0; i < mipmapCount; i++)
                {
                    // 縦方向にイメージ内
                    if (point.Y < heightNext)
                    {
                        // 横方向にイメージ内
                        if (point.X < imageW)
                        {
                            mipmapLevel = i;
                            return new Point(point.X, point.Y - heightPrev);
                        }

                        // 終了
                        break;
                    }

                    // 次のレベルへ
                    heightPrev += imageH;
                    {
                        imageW = Math.Max(mipmapMinSize.Width, imageW >> 1);
                        imageH = Math.Max(mipmapMinSize.Height, imageH >> 1);
                    }

                    heightNext += imageH;
                }
            }
            else
            {
                var widthPrev = 0;
                var widthNext = imageW;

                // ミップマップを追う
                for (var i = 0; i < mipmapCount; i++)
                {
                    // 横方向にイメージ内
                    if (point.X < widthNext)
                    {
                        // 縦方向にイメージ内
                        if (point.Y < imageH)
                        {
                            mipmapLevel = i;
                            return new Point(point.X - widthPrev, point.Y);
                        }

                        // 終了
                        break;
                    }

                    // 次のレベルへ
                    widthPrev += imageW;
                    {
                        imageW /= 2;
                        imageH /= 2;
                    }

                    widthNext += imageW;
                }
            }

            // 無効
            mipmapLevel = -1;
            return new Point(-1, -1);
        }

        /// <summary>
        /// ビットマップ結合（32bppArgb）
        /// </summary>
        /// <param name="dstBitmap">結果ビットマップ</param>
        /// <param name="srcBitmaps">入力ビットマップ</param>
        /// <param name="mipmapOrientation">ビットマップ方向</param>
        private static void CombineBitmaps_32bppArgb(Bitmap dstBitmap, Bitmap[] srcBitmaps, MipmapOrientationType mipmapOrientation)
        {
            var pixelOffset = 0;
            unsafe
            {
                // 先画像に
                using (var dstBp = new BitmapProcess(dstBitmap))
                {
                    var dstBmpPtr = (byte*)(void*)dstBp.BitmapData.Scan0;
                    var dstStride = dstBp.BitmapData.Stride;

                    // 元画像をレベル順にコピー
                    foreach (var srcBitmap in srcBitmaps)
                    {
                        using (var srcBp = new BitmapProcess(srcBitmap))
                        {
                            var srcBmpPtr = (byte*)(void*)srcBp.BitmapData.Scan0;
                            var srcStride = srcBp.BitmapData.Stride;
                            var rasterSize = srcBitmap.Width * 4;

                            if (mipmapOrientation == MipmapOrientationType.Horizontal)
                            {
                                Parallel.For(
                                    0,
                                    srcBitmap.Height,
                                    y =>
                                    {
                                        CopyMemory((IntPtr)(dstBmpPtr + ((dstStride * y) + (pixelOffset * 4))), (IntPtr)(srcBmpPtr + (srcStride * y)), rasterSize);
                                    });

                                pixelOffset += srcBitmap.Width;
                            }
                            else
                            {
                                Parallel.For(
                                    0,
                                    srcBitmap.Height,
                                    y =>
                                    {
                                        CopyMemory((IntPtr)(dstBmpPtr + (dstStride * (pixelOffset + y))), (IntPtr)srcBmpPtr + (srcStride * y), rasterSize);
                                    });

                                pixelOffset += srcBitmap.Height;
                            }
                        }
                    }
                }
            }
        }

        [DllImport("Kernel32.dll", EntryPoint = "RtlCopyMemory")]
        private static extern void CopyMemory(IntPtr destination, IntPtr source, [MarshalAs(UnmanagedType.U4)] int length);

        /// <summary>
        /// ピクセルヒントハンドラ
        /// </summary>
        /// <param name="e">イベント引数</param>
        private void OnPixelHint(TpBitmapImagePixelHintEventArgs e)
        {
            if (this.PixelHint != null)
            {
                this.PixelHint(this, e);
            }
        }

        /// <summary>
        /// ピクセル位置を取得
        /// </summary>
        /// <param name="point">マウス位置</param>
        /// <returns>ピクセル位置</returns>
        private Point GetPixelLocation(Point point)
        {
            // 描画領域
            var image = this.RenderingBounds;

            // 領域内を指している場合のみ
            var pixel = new Point(-1, -1);
            if (image.Contains(point))
            {
                var baseImage = this.RenderingColorImage.Size;
                pixel.X = (int)((point.X - image.X) * baseImage.Width  / image.Width);
                pixel.Y = (int)((point.Y - image.Y) * baseImage.Height / image.Height);
            }

            return pixel;
        }

        /// <summary>
        /// チャンネル変更ハンドラ
        /// </summary>
        private void OnChannelChanged()
        {
            if (this.ChannelChanged != null)
            {
                this.ChannelChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// カラーチャンネル行列を取得
        /// </summary>
        /// <returns>行列</returns>
        private ColorMatrix GetColorChannelMatrix()
        {
            var cm = new ColorMatrix();

            if (this.Contains(ColorChannelFlags.Rgb))
            {
                // ＲＧＢ合成
                // 単位行列
                cm.Matrix00 = 1.0f;
                cm.Matrix10 = 0.0f;
                cm.Matrix20 = 0.0f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.0f;
                cm.Matrix11 = 1.0f;
                cm.Matrix21 = 0.0f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.0f;
                cm.Matrix12 = 0.0f;
                cm.Matrix22 = 1.0f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.0f;
            }
            else
            {
                // 各成分の組み合わせ
                // 一旦ゼロ行列にする
                cm.Matrix00 = 0.0f;
                cm.Matrix10 = 0.0f;
                cm.Matrix20 = 0.0f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.0f;
                cm.Matrix11 = 0.0f;
                cm.Matrix21 = 0.0f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.0f;
                cm.Matrix12 = 0.0f;
                cm.Matrix22 = 0.0f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.0f;

                // Ｒ成分
                if (this.Contains(ColorChannelFlags.R))
                {
                    cm.Matrix00 = 1.0f;
                    cm.Matrix10 = 0.0f;
                    cm.Matrix20 = 0.0f;
                    cm.Matrix30 = 0.0f;
                    cm.Matrix40 = 0.0f;

                    // Ｒ成分のみ
                    if ((this.channel & ColorChannelFlags.Rgb) == ColorChannelFlags.R)
                    {
                        cm.Matrix01 = 1.0f;
                        cm.Matrix11 = 0.0f;
                        cm.Matrix21 = 0.0f;
                        cm.Matrix31 = 0.0f;
                        cm.Matrix41 = 0.0f;

                        cm.Matrix02 = 1.0f;
                        cm.Matrix12 = 0.0f;
                        cm.Matrix22 = 0.0f;
                        cm.Matrix32 = 0.0f;
                        cm.Matrix42 = 0.0f;
                    }
                }

                // Ｇ成分
                if (this.Contains(ColorChannelFlags.G))
                {
                    cm.Matrix01 = 0.0f;
                    cm.Matrix11 = 1.0f;
                    cm.Matrix21 = 0.0f;
                    cm.Matrix31 = 0.0f;
                    cm.Matrix41 = 0.0f;

                    // Ｇ成分のみ
                    if ((this.channel & ColorChannelFlags.Rgb) == ColorChannelFlags.G)
                    {
                        cm.Matrix00 = 0.0f;
                        cm.Matrix10 = 1.0f;
                        cm.Matrix20 = 0.0f;
                        cm.Matrix30 = 0.0f;
                        cm.Matrix40 = 0.0f;

                        cm.Matrix02 = 0.0f;
                        cm.Matrix12 = 1.0f;
                        cm.Matrix22 = 0.0f;
                        cm.Matrix32 = 0.0f;
                        cm.Matrix42 = 0.0f;
                    }
                }

                // Ｂ成分
                if (this.Contains(ColorChannelFlags.B))
                {
                    cm.Matrix02 = 0.0f;
                    cm.Matrix12 = 0.0f;
                    cm.Matrix22 = 1.0f;
                    cm.Matrix32 = 0.0f;
                    cm.Matrix42 = 0.0f;

                    // Ｂ成分のみ
                    if ((this.channel & ColorChannelFlags.Rgb) == ColorChannelFlags.B)
                    {
                        cm.Matrix00 = 0.0f;
                        cm.Matrix10 = 0.0f;
                        cm.Matrix20 = 1.0f;
                        cm.Matrix30 = 0.0f;
                        cm.Matrix40 = 0.0f;

                        cm.Matrix01 = 0.0f;
                        cm.Matrix11 = 0.0f;
                        cm.Matrix21 = 1.0f;
                        cm.Matrix31 = 0.0f;
                        cm.Matrix41 = 0.0f;
                    }
                }
            }

            return cm;
        }

        /// <summary>
        /// ズームイン（内部処理）
        /// </summary>
        /// <param name="center">中心</param>
        private void ZoomInCore(Point center)
        {
            for (var i = 0; i < MagnifyTable.Length; i++)
            {
                if (this.Magnify < i)
                {
                    float magnify = i;
                    this.ChangeMagnifyCore(magnify, center);
                    break;
                }
            }
        }

        /// <summary>
        /// ズームアウト（内部処理）
        /// </summary>
        /// <param name="center">中心</param>
        private void ZoomOutCore(Point center)
        {
            for (var i = MagnifyTable.Length - 1; i >= 0; i--)
            {
                if (this.Magnify > MagnifyTable[i])
                {
                    float magnify = MagnifyTable[i];
                    this.ChangeMagnifyCore(magnify, center);
                    break;
                }
            }
        }

        /// <summary>
        /// オーナーでアクティベイト時
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnOwnerDeactivate(object sender, EventArgs e)
        {
            this.ResetPixelHint();
        }

        /// <summary>
        /// ピクセルヒントをリセット
        /// </summary>
        private void ResetPixelHint()
        {
            // 保存値を既定値に戻す
            this.lastPointedPixel = new Point(-2, -2);

            // 無効ヒントイベント発行
            this.OnPixelHint(TpBitmapImagePixelHintEventArgs.Empty);
        }

        /// <summary>
        /// 表示倍率を変更（内部処理）
        /// </summary>
        /// <param name="magnify">倍率</param>
        /// <param name="center">中心</param>
        private void ChangeMagnifyCore(float magnify, Point center)
        {
            //-----------------------------------------------------------------
            // 倍率変更前に行う処理

            // 指定点をクライアント中心からと描画領域中心からの位置に変換
            var fromImageCenter = new PointF(0.0f, 0.0f);
            var fromClientCenter = new PointF(0.0f, 0.0f);
            {
                var renderingBounds = this.RenderingBounds;
                var centerOfClient = this.CenterOfClient;

                if (renderingBounds.X <= center.X && center.X < renderingBounds.Right)
                {
                    fromImageCenter.X = center.X - ((renderingBounds.X + renderingBounds.Right) / 2.0f);
                    fromClientCenter.X = center.X - centerOfClient.X;
                }

                if (renderingBounds.Y <= center.Y && center.Y < renderingBounds.Bottom)
                {
                    fromImageCenter.Y = center.Y - ((renderingBounds.Y + renderingBounds.Bottom) / 2.0f);
                    fromClientCenter.Y = center.Y - centerOfClient.Y;
                }
            }

            // 描画領域中心からの距離は変更率に応じてスケーリングする
            fromImageCenter.X *= magnify / this.Magnify;
            fromImageCenter.Y *= magnify / this.Magnify;

            //-----------------------------------------------------------------
            // 倍率変更
            this.Magnify = magnify;

            //-----------------------------------------------------------------
            // 倍率変更後に行う処理

            // 先に全領域を無効化
            // スクロール設定変更後に呼ぶとちらつく...
            this.Invalidate();

            // スクロール最小サイズを変更してスクロール位置を調整
            this.AutoScrollMinSize = this.RenderingSize;
            {
                var scrollMaxNew = this.AutoScrollMaxSize;

                // 水平方向
                var scrollX = 0;
                if (scrollMaxNew.Width > 0)
                {
                    scrollX = (int)((scrollMaxNew.Width * 0.5f) + (fromImageCenter.X - fromClientCenter.X));
                    if (scrollX < 0)
                    {
                        scrollX = 0;
                    }
                    else if (scrollX > scrollMaxNew.Width)
                    {
                        scrollX = scrollMaxNew.Width;
                    }
                }

                // 垂直方向
                int scrollY = 0;
                if (scrollMaxNew.Height > 0)
                {
                    scrollY = (int)((scrollMaxNew.Height * 0.5f) + (fromImageCenter.Y - fromClientCenter.Y));
                    if (scrollY < 0)
                    {
                        scrollY = 0;
                    }
                    else if (scrollY > scrollMaxNew.Height)
                    {
                        scrollY = scrollMaxNew.Height;
                    }
                }

                // スクロール位置
                this.AutoScrollPosition = new Point(scrollX, scrollY);
            }

            this.OnMagnifyChanged(EventArgs.Empty);
        }

        /// <summary>
        /// 表示倍率変更ハンドラ
        /// </summary>
        /// <param name="e">イベント引数</param>
        private void OnMagnifyChanged(EventArgs e)
        {
            if (this.MagnifyChanged != null)
            {
                this.MagnifyChanged(this, e);
            }
        }

        /// <summary>
        /// イメージの更新
        /// </summary>
        private void UpdateImages()
        {
            if (this.IsTargetEnabled == false)
            {
                return;
            }

            System.Diagnostics.Debug.Assert(this.Target is FtxTextureData, "this.Target is FtxTextureData");

            var ftxTarget = this.Target as FtxTextureData;
            var srcImages = Enumerable.Range(0, this.Target.MipmapCount).Select(x => this.Target.GeneratePreviewBitmap(x)).ToArray();
            var rgbMatrix = RenderUtility.CreateRgbColorMatrix(ftxTarget.CompSel, ftxTarget.PixelFormat);
            var alphaMatrix = RenderUtility.CreateAlphaColorMatrix(ftxTarget.CompSel, ftxTarget.PixelFormat);
            var colorImages = srcImages.Select(src => ColorUtility.ConvertBitmapWithGamma(
                src, rgbMatrix, true, this.Target.PixelFormat.IsSrgb())).ToArray();
            var alphaImages = srcImages.Select(src => ColorUtility.ConvertBitmapWithGamma(
                src, alphaMatrix, false, this.Target.PixelFormat.IsSrgb())).ToArray();

            this.colorImage = colorImages[0];
            this.alphaImage = alphaImages[0];

            if (this.DepthIndex == -1)
            {
                this.colorImageMM[MipmapOrientationType.Horizontal] = CombineMipmapBitmaps(colorImages, this.Target.MipmapMinSize, MipmapOrientationType.Horizontal);
                this.colorImageMM[MipmapOrientationType.Vertical] = CombineMipmapBitmaps(colorImages, this.Target.MipmapMinSize, MipmapOrientationType.Vertical);

                this.alphaImageMM[MipmapOrientationType.Horizontal] = CombineMipmapBitmaps(alphaImages, this.Target.MipmapMinSize, MipmapOrientationType.Horizontal);
                this.alphaImageMM[MipmapOrientationType.Vertical] = CombineMipmapBitmaps(alphaImages, this.Target.MipmapMinSize, MipmapOrientationType.Vertical);
            }
            else
            {
                System.Diagnostics.Debug.Assert(false, "未実装");
            }

            this.Invalidate();
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.UpdateImages();
        }

        /// <summary>
        /// 各種初期状態を設定します。
        /// </summary>
        private void InitSetteings()
        {
            // 初期表示倍率
            Size imageSize = this.RenderingColorImage.Size;
            {
                int size = Math.Max(imageSize.Width, imageSize.Height);

                // 512を超える場合のみ拡大、それ以下では等倍
                if (size > 512)
                {
                    this.Magnify = 0.5f;
                }
            }

            // 初期設定
            this.AutoScroll = true;
            this.AutoScrollMinSize = this.RenderingSize;
        }

        /// <summary>
        /// テクスチャ/パレットビットマップイメージピクセルヒントイベントデータ
        /// </summary>
        public sealed class TpBitmapImagePixelHintEventArgs : EventArgs
        {
            /// <summary>
            /// 既定のイベントデータ
            /// </summary>
            public static new readonly TpBitmapImagePixelHintEventArgs Empty = new TpBitmapImagePixelHintEventArgs();

            /// <summary>
            /// コンストラクタ
            /// ピクセルカラー情報用
            /// </summary>
            /// <param name="pixelLocation">ピクセル位置</param>
            /// <param name="color">色</param>
            public TpBitmapImagePixelHintEventArgs(Point pixelLocation, Color color)
            {
                this.PixelHint = TpBitmapImagePixelHint.PixelColor;
                this.PixelLocation = pixelLocation;
                this.Color = color;
            }

            /// <summary>
            /// コンストラクタ
            /// 無効情報用
            /// </summary>
            private TpBitmapImagePixelHintEventArgs()
            {
                this.PixelHint = TpBitmapImagePixelHint.None;
                this.PixelLocation = new Point(-1, -1);
                this.Color = Color.FromArgb(0, 0, 0, 0);
            }

            /// <summary>
            /// ピクセルヒント
            /// </summary>
            public TpBitmapImagePixelHint PixelHint { get; private set; }

            /// <summary>
            /// ピクセル位置
            /// </summary>
            public Point PixelLocation { get; private set; }

            /// <summary>
            /// カラー
            /// </summary>
            public Color Color { get; private set; }
        }

        /// <summary>
        /// ビットマップ加工処理クラス
        /// </summary>
        private sealed class BitmapProcess : IDisposable
        {
            /// <summary>
            /// ビットマップ
            /// </summary>
            private readonly Bitmap bitmap;

            /// <summary>
            /// ビットマップデータ
            /// </summary>
            private readonly BitmapData bitmapData;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="bitmap">ビットマップ</param>
            public BitmapProcess(Bitmap bitmap)
            {
                this.bitmap = bitmap;

                // ロック開始
                this.bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);
            }

            /// <summary>
            /// ビットマップデータ
            /// </summary>
            public BitmapData BitmapData
            {
                get { return this.bitmapData; }
            }

            /// <summary>
            /// インタフェース実装
            /// </summary>
            public void Dispose()
            {
                // ロック解除
                this.bitmap.UnlockBits(this.bitmapData);
            }
        }
    }
}
