﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.ClipboardDataTypes;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.Specifics.TableEditor
{
    /// <summary>
    /// TableEditor用DataGridViewクラス
    /// </summary>
    public class TableEditorDataGrid : DataGridView, IControl, ILayoutElement
    {
        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// IsLinearMode プロパティのバッキングフィールドです。
        /// </summary>
        private bool isLinearMode = false;

        /// <summary>
        /// The value table.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        private AnimationTableData valueTable = null;

        /// <summary>
        /// The backup value.
        /// </summary>
        private string backupValue;

        /// <summary>
        /// The last current row index.
        /// </summary>
        private int lastCurrentRowIndex = 0;

        /// <summary>
        /// 行数変更中のフラグ。
        /// </summary>
        private bool isRowsCountChanging = false;

        /// <summary>
        /// セル編集中のフラグ
        /// </summary>
        private bool isEditing = false;

        /// <summary>
        /// 編集用コントロール
        /// </summary>
        private Control editingControl = null;

        /// <summary>
        /// 選択セル変更イベント抑制フラグ
        /// </summary>
        private bool isSuppressSelectionChanged = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public TableEditorDataGrid()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);

            this.valueTable = new AnimationTableData();

            // 幅や高さに対するユーザーの変更をことごとく禁止する
            this.ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.DisableResizing;
            this.RowHeadersWidthSizeMode = DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            this.AllowUserToAddRows = false;
            this.AllowUserToDeleteRows = false;
            this.AllowUserToOrderColumns = false;
            this.AllowUserToResizeColumns = false;
            this.AllowUserToResizeRows = false;
            this.ImeMode = System.Windows.Forms.ImeMode.Off;
            this.MultiSelect = false;
            this.SelectionMode = System.Windows.Forms.DataGridViewSelectionMode.CellSelect;

            this.CellBeginEdit += this.TableEditorCellBeginEdit;
            this.CellEndEdit += this.TableEditorCellEndEdit;
            this.CellPainting += this.DataGridView_CellPainting;

            this.CurrentCellChanged += (s, e) =>
            {
                if (this.CurrentCell != null)
                {
                    this.LastCurrentRowIndex = Math.Max(0, this.CurrentCell.RowIndex);
                }

                if (this.isSuppressSelectionChanged)
                {
                    return;
                }

                if (this.OnCurrentCellChangedBlockable != null)
                {
                    this.OnCurrentCellChangedBlockable();
                }
            };

            this.EditingControlShowing += (s, e) =>
            {
                if (editingControl == null)
                {
                    e.Control.Enter += (ss, ee) =>
                    {
                        if (this.OnBeginCellEdit != null)
                        {
                            this.OnBeginCellEdit();
                        }
                    };
                    e.Control.Leave += (ss, ee) =>
                    {
                        if (!this.Focused)
                        {
                            this.CommitEdit();
                        }
                    };
                    this.editingControl = e.Control;
                }
            };

            this.MaxKeyNum = 32;
            this.Digit = 3;
            this.RowHeadersWidth = 60;
            this.AdjustSize();

            this.InitializeContextMenu();
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="controlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// リニア編集モードかどうか取得または設定します。
        /// </summary>
        public bool IsLinearMode
        {
            get
            {
                return this.isLinearMode;
            }

            set
            {
                if (value != this.isLinearMode)
                {
                    this.isLinearMode = value;

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// Gets or sets the columns of value.
        /// </summary>
        public int ColumnsOfValue
        {
            get
            {
                return this.ColumnCount - 1;
            }

            set
            {
                if (value < 1)
                {
                    return;
                }
                else if (value > 4)
                {
                    return;
                }

                this.Columns.Clear();
                for (int i = 0; i < value + 1; ++i)
                {
                    this.Columns.Add(new TableEditorColumn());
                }

                this.SyncData();
                this.AdjustSize();
            }
        }

        /// <summary>
        /// Gets or sets the values.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public AnimationTableData Values
        {
            get
            {
                return this.valueTable;
            }

            set
            {
                this.valueTable.Set(value);
                this.SyncView();
            }
        }

        /// <summary>
        /// 挿入や追加ができるかどうかを返すプロパティ
        /// </summary>
        public bool CanAdd
        {
            get
            {
                int a, b;
                return this.IsAddable(out a, out b);
            }
        }

        /// <summary>
        /// 削除が可能かどうかを返すプロパティ
        /// </summary>
        public bool CanDelete
        {
            get
            {
                return this.valueTable.Count > 1;
            }
        }

        /// <summary>
        /// The on begin cell edit.
        /// </summary>
        public Action OnBeginCellEdit { get; set; }

        /// <summary>
        /// The on end cell edit.
        /// </summary>
        public Action OnEndCellEdit { get; set; }

        /// <summary>
        /// セルの選択状態が変化した時の処理を取得または設定します。
        /// コントロールの状態に応じてトリガが抑制されます。
        /// </summary>
        public Action OnCurrentCellChangedBlockable { get; set; }

        /// <summary>
        /// 行番号カラムの描画モードを指定します。
        /// </summary>
        public int IndexDrawMode { get; set; }

        /// <summary>
        /// 行ヘッダーをクリックした際のイベントハンドラ
        /// </summary>
        public Action OnHeaderClicked { get; set; }

        /// <summary>
        /// 最後にカレントセルがいた行を返します。
        /// </summary>
        public int LastCurrentRowIndex
        {
            get
            {
                if (this.lastCurrentRowIndex < 0)
                {
                    this.lastCurrentRowIndex = 0;
                }
                else if (this.lastCurrentRowIndex >= this.Values.Count)
                {
                    this.lastCurrentRowIndex = this.Values.Count - 1;
                }

                return this.lastCurrentRowIndex;
            }

            private set
            {
                this.lastCurrentRowIndex = value;
            }
        }

        /// <summary>
        /// 最大キーフレーム数
        /// </summary>
        public int MaxKeyNum { get; set; }

        /// <summary>
        /// 小数点以下桁数
        /// </summary>
        public int Digit { get; set; }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="controlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// 規定値で埋めた行を追加・挿入します
        /// </summary>
        /// <param name="copying">
        /// コピーするならtrue、デフォルト値で作成するならfalseを渡します。
        /// </param>
        /// <returns>
        /// 追加や挿入ができた場合はtrue、できなかった場合はfalseを返します。
        /// </returns>
        public bool AddRow(bool copying)
        {
            int selectedIndex = -1;
            int nextKey = -1;

            if (this.IsAddable(out selectedIndex, out nextKey) == false)
            {
                return false;
            }

            this.isRowsCountChanging = true;

            // 追加行データを作成
            var row = new DataGridViewRow();

            DataGridViewCell keyCell = new DataGridViewTextBoxCell();
            keyCell.Value = nextKey;
            row.Cells.Add(keyCell);
            for (int i = 0; i < this.ColumnsOfValue; ++i)
            {
                DataGridViewCell cell = new DataGridViewTextBoxCell();
                if (copying && this.CurrentRow != null)
                {
                    cell.Value = this.Rows[this.CurrentRow.Index].Cells[i + 1].Value;
                }
                else
                {
                    var col = this.Columns[i + 1] as TableEditorColumn;
                    if (col != null)
                    {
                        cell.Value = col.DefaultCellValue;
                    }
                }

                row.Cells.Add(cell);
            }

            // 追加
            if (selectedIndex == -1 || selectedIndex + 1 == this.Rows.Count)
            {
                this.Rows.Add(row);
            }
            else
            {
                this.Rows.Insert(selectedIndex + 1, row);
            }

            // リストと同期
            this.SyncData();
            this.AdjustSize();

            // アクティブセルを追加した行に移動
            this.CurrentCell = row.Cells[0];

            this.isRowsCountChanging = false;

            return true;
        }

        /// <summary>
        /// The delete row.
        /// </summary>
        /// <returns>
        /// 実際に削除を行った場合はtrue、
        /// 行数が1行以下だったり、選択行が特定できない場合はfalseを返します。
        /// </returns>
        public bool DeleteRow()
        {
            if (this.CanDelete == false)
            {
                return false;
            }

            this.isRowsCountChanging = true;

            this.Rows.RemoveAt(this.SelectedCells.Count == 0 ? this.LastCurrentRowIndex : this.SelectedCells[0].RowIndex);

            this.SyncData();
            this.AdjustSize();

            this.OnCurrentCellChanged(null);

            this.isRowsCountChanging = false;

            return true;
        }

        /// <summary>
        /// The sync view.
        /// </summary>
        public void SyncView()
        {
            this.SuspendDrawing();

            int currentColumn = (this.CurrentCell != null) ? this.CurrentCell.ColumnIndex : 0;
            int currentRow = (this.CurrentCell != null) ? this.CurrentCell.RowIndex : 0;

            // データ同期時にいったんテーブルをクリアするため、セル変更イベントが走るのを抑制
            this.isSuppressSelectionChanged = true;

            this.Rows.Clear();

            foreach (var key in this.valueTable)
            {
                var row = new DataGridViewRow();

                {
                    DataGridViewCell cell = new DataGridViewTextBoxCell();
                    cell.Value = key.Frame;

                    row.Cells.Add(cell);
                }

                for (int j = 0; j < this.ColumnsOfValue; ++j)
                {
                    DataGridViewCell cell = new DataGridViewTextBoxCell();
                    cell.Value = key.Value[j];

                    row.Cells.Add(cell);
                }

                this.Rows.Add(row);
            }

            if (this.valueTable.Count == 0)
            {
                this.AddRow(false);
            }

            this.AdjustSize();

            // 抑制解除
            this.isSuppressSelectionChanged = false;

            this.CurrentCell =
                this.Rows[Math.Min(currentRow, this.RowCount - 1)]
                .Cells[Math.Min(currentColumn, this.ColumnCount - 1)];

            this.ResumeDrawing();
            this.Invalidate();
        }

        /// <summary>
        /// The adjust size.
        /// </summary>
        public void AdjustSize()
        {
            this.Width = 3 + this.RowHeadersWidth + this.Columns.Cast<TableEditorColumn>().Sum(column => column.Width);
            this.Height = 3 + this.ColumnHeadersHeight + this.Rows.Cast<DataGridViewRow>().Sum(row => row.Height);
        }

        /// <summary>
        /// 先頭のセルを選択します。
        /// </summary>
        public void FocusCellLeftTop()
        {
            if (this.isRowsCountChanging)
            {
                return;
            }

            this.CurrentCell = this.Rows[0].Cells[0];
            this.Select();
        }

        /// <summary>
        /// 終端のセルを選択します。
        /// </summary>
        public void FocusCellRightBottom()
        {
            if (this.isRowsCountChanging)
            {
                return;
            }

            this.CurrentCell = this.Rows[this.RowCount - 1].Cells[this.ColumnCount - 1];
            this.Select();
        }

        /// <summary>
        /// タブオーダーに準じたセルの選択を行います。
        /// </summary>
        /// <param name="isForward">true:順送り／false:逆送り</param>
        /// <returns>先頭か終端に到達したらtrue</returns>
        public bool FocusCellByOrder(bool isForward)
        {
            var cellIndex = this.CurrentCellAddress;

            if (isForward)
            {
                ++cellIndex.X;
                if (cellIndex.X >= this.ColumnCount)
                {
                    cellIndex.X = 0;
                    ++cellIndex.Y;
                    if (cellIndex.Y >= this.RowCount)
                    {
                        return true;
                    }
                }
            }
            else
            {
                --cellIndex.X;
                if (cellIndex.X < 0)
                {
                    cellIndex.X = this.ColumnCount - 1;
                    --cellIndex.Y;
                    if (cellIndex.Y < 0)
                    {
                        return true;
                    }
                }
            }

            this.CurrentCell = this.Rows[cellIndex.Y].Cells[cellIndex.X];
            this.Select();
            return false;
        }

        /// <summary>
        /// The commit edit.
        /// </summary>
        public void CommitEdit()
        {
            this.EndEdit();

            // カラーピッカーの履歴連続修正
            if (this.CurrentCell != null && this.isEditing)
            {
                this.OnCellEndEdit(new DataGridViewCellEventArgs(
                    this.CurrentCell.ColumnIndex, this.CurrentCell.RowIndex));
            }
        }

        /// <summary>
        /// EnterキーとTABキーが押された時の挙動制御
        /// </summary>
        /// <param name="keyData">
        /// The key data.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        [System.Security.Permissions.UIPermission(
            System.Security.Permissions.SecurityAction.LinkDemand,
            Window = System.Security.Permissions.UIPermissionWindow.AllWindows)]
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // Enterキーでの行移動を無効化
            if ((keyData & Keys.KeyCode) == Keys.Enter)
            {
                if (this.isEditing)
                {
                    this.CommitEdit();
                }

                return true;
            }

            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// 編集中のマウスダウンは編集打ち切りに
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (this.isEditing && this.CurrentCell != null)
            {
                var result = this.HitTest(e.X, e.Y);
                if (result.RowIndex != this.CurrentCell.RowIndex ||
                    result.ColumnIndex != this.CurrentCell.ColumnIndex)
                {
                    this.CommitEdit();
                    return;
                }
            }

            this.Focus();

            base.OnMouseDown(e);
        }

        /// <summary>
        /// 行ヘッダをクリックした際にセルを選択するように修正
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnMouseClick(MouseEventArgs e)
        {
            var result = this.HitTest(e.X, e.Y);
            if (result.RowIndex == -1)
            {
                return;
            }

            if (result.ColumnIndex == -1)
            {
                if (this.OnHeaderClicked != null)
                {
                    this.OnHeaderClicked();
                }

                this.CurrentCell = this.Rows[result.RowIndex].Cells[0];
                this.Focus();
            }

            base.OnMouseClick(e);
        }

        /// <summary>
        /// The is addable.
        /// </summary>
        /// <param name="index">
        /// The index.
        /// </param>
        /// <param name="key">
        /// The key.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool IsAddable(out int index, out int key)
        {
            int selectedIndex = -1;

            // 現在カーソルがある行数を取得
            if (this.SelectedCells.Count != 0)
            {
                selectedIndex = this.SelectedCells[0].RowIndex;
            }

            if (selectedIndex == -1)
            {
                selectedIndex = this.LastCurrentRowIndex;
            }

            int nextKey;

            // 追加する行の規定値となるキーフレームを取得
            if (selectedIndex == -1)
            {
                nextKey = this.valueTable.Count == 0
                    ? 0
                    : this.valueTable[this.valueTable.Count - 1].Frame + 1;
            }
            else
            {
                nextKey = this.valueTable[selectedIndex].Frame + 1;
                if (selectedIndex + 1 < this.valueTable.Count)
                {
                    if (this.valueTable[selectedIndex + 1].Frame == nextKey)
                    {
                        index = key = -1;
                        return false;
                    }
                }
            }

            index = selectedIndex;
            key = nextKey;

            if (this.valueTable.Count >= this.MaxKeyNum)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// セルの編集が始まるタイミングで現状の値をバックアップします。
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void TableEditorCellBeginEdit(object sender, DataGridViewCellCancelEventArgs e)
        {
            this.isEditing = true;
            this.backupValue = this.Rows[e.RowIndex].Cells[e.ColumnIndex].Value.ToString();
            this.LastCurrentRowIndex = e.RowIndex;

            if (this.OnBeginCellEdit != null)
            {
                this.OnBeginCellEdit();
            }
        }

        /// <summary>
        /// セルの編集が終了したタイミングで、適切な値かどうかを判断し、不適切な場合はバックアップした値に復元します。
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void TableEditorCellEndEdit(object sender, DataGridViewCellEventArgs e)
        {
            object cellValue = this.Rows[e.RowIndex].Cells[e.ColumnIndex].Value;
            if (cellValue == null)
            {
                this.FinishEdit(e.RowIndex);
                return;
            }

            string inputValue = cellValue.ToString();
            if (e.ColumnIndex == 0)
            {
                // 編集前の文字列を数値化
                int backupValueInt = 0;
                int.TryParse(this.backupValue, out backupValueInt);

                // キーフレームのカラム
                int key;
                if (int.TryParse(inputValue, out key) == false)
                {
                    // 整数に変換できない値だったら編集前に戻す
                    this.Rows[e.RowIndex].Cells[0].Value = backupValueInt;
                }
                else if (key < 0)
                {
                    // 負の値の場合でも編集前に戻す
                    this.Rows[e.RowIndex].Cells[0].Value = backupValueInt;
                }
                else
                {
                    // いったん入力前の値に戻す
                    this.Rows[e.RowIndex].Cells[0].Value = backupValueInt;

                    // 同じキーフレームの値がないかどうか探す
                    for (int i = 0; i < this.Rows.Count; ++i)
                    {
                        object columnValue = this.Rows[i].Cells[0].Value;
                        if (columnValue == null)
                        {
                            this.FinishEdit(e.RowIndex);
                            return;
                        }

                        int lookingKey = int.Parse(columnValue.ToString());
                        if (key == lookingKey)
                        {
                            // すでに存在するキーフレームと同じ値は受け付けられない
                            this.FinishEdit(e.RowIndex);
                            return;
                        }
                    }

                    // 入力値を再度適用する
                    this.Rows[e.RowIndex].Cells[0].Value = key;

                    // ソート(編集中→編集中で呼ばれると例外を吐くのでそうならないようにしないといけない)
                    this.Sort(this.Columns[0], ListSortDirection.Ascending);

                    // リストに同期
                    this.SyncData();
                }
            }
            else
            {
                // 値のカラム
                // 編集前の文字列を数値化
                float backupValueFloat = 0.0f;
                float.TryParse(this.backupValue, out backupValueFloat);

                float value;
                if (float.TryParse(inputValue, out value) == false)
                {
                    // 実数に変換できない値だったら編集前に戻す
                    this.Rows[e.RowIndex].Cells[e.ColumnIndex].Value = backupValueFloat;
                }
                else
                {
                    var col = this.Columns[e.ColumnIndex] as TableEditorColumn;

                    if (col != null)
                    {
                        // 最大値と最小値の間に収まるようにする
                        if (value > col.MaxCellValue)
                        {
                            value = col.MaxCellValue;
                        }
                        else if (value < col.MinCellValue)
                        {
                            value = col.MinCellValue;
                        }

                        value = (float)Math.Round(value, this.Digit);
                    }

                    this.Rows[e.RowIndex].Cells[e.ColumnIndex].Value = value;
                    this.SyncData();
                }
            }

            this.FinishEdit(e.RowIndex);
        }

        /// <summary>
        /// The finish edit.
        /// </summary>
        /// <param name="rowIndex">
        /// The row index.
        /// </param>
        private void FinishEdit(int rowIndex)
        {
            this.LastCurrentRowIndex = rowIndex;

            if (this.OnEndCellEdit != null)
            {
                this.OnEndCellEdit();
            }

            this.isEditing = false;
        }

        /// <summary>
        /// コンテキストメニュー初期化
        /// </summary>
        private void InitializeContextMenu()
        {
            this.ContextMenuStrip = ContextMenuUtility.CreateCopyPaste(
                () =>
                {
                    if (this.CurrentCell == null)
                    {
                        return;
                    }

                    var value = this.Values[this.LastCurrentRowIndex];

                    Clipboard.SetData(
                        ColorClipboardData.ClipboardFormat,
                        new ColorClipboardData
                        {
                            R = this.IndexDrawMode == 1 ? value.ColorValue.R : 1.0f,
                            G = this.IndexDrawMode == 1 ? value.ColorValue.G : 1.0f,
                            B = this.IndexDrawMode == 1 ? value.ColorValue.B : 1.0f,
                            A = this.IndexDrawMode == 2 ? value.ColorValue.R : 1.0f,
                            EnabledRgb = this.IndexDrawMode == 1,
                            EnabledAlpha = this.IndexDrawMode == 2
                        });
                },
                () =>
                {
                    if (Clipboard.ContainsData(ColorClipboardData.ClipboardFormat))
                    {
                        var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                        switch (this.IndexDrawMode)
                        {
                            case 1:
                                this.Rows[this.CurrentCell.RowIndex].Cells[1].Value = data.R;
                                this.Rows[this.CurrentCell.RowIndex].Cells[2].Value = data.G;
                                this.Rows[this.CurrentCell.RowIndex].Cells[3].Value = data.B;
                                break;
                            case 2:
                                this.Rows[this.CurrentCell.RowIndex].Cells[1].Value = data.A;
                                break;
                            default:
                                return;
                        }

                        // 同期
                        this.SyncData();
                        this.Invalidate();
                        if (this.OnEndCellEdit != null)
                        {
                            this.OnEndCellEdit();
                        }
                    }
                });

            this.ContextMenuStrip.Opening += (s, e) =>
            {
                var mousePos = this.PointToClient(Cursor.Position);
                var testResult = this.HitTest(mousePos.X, mousePos.Y);

                bool cancel = this.IndexDrawMode == 0;
                cancel |= testResult.ColumnIndex < -1;
                cancel |= testResult.ColumnIndex >= this.ColumnCount;
                cancel |= testResult.RowIndex < 0;
                cancel |= testResult.RowIndex >= this.RowCount;

                if (cancel)
                {
                    e.Cancel = true;
                    return;
                }

                this.CurrentCell
                    = this.Rows[testResult.RowIndex].Cells[Math.Max(testResult.ColumnIndex, 0)];
                this.LastCurrentRowIndex = testResult.RowIndex;

                var enabledPaste = Clipboard.ContainsData(ColorClipboardData.ClipboardFormat);

                if (enabledPaste)
                {
                    var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                    enabledPaste = (data.EnabledRgb == (this.IndexDrawMode == 1)) &&
                                    (data.EnabledAlpha == (this.IndexDrawMode == 2));
                }

                this.ContextMenuStrip.Items[ContextMenuUtility.CopyPasteMenuPasteIndex].Enabled = enabledPaste;
            };
        }

        /// <summary>
        /// DataGridからListへデータを同期します。
        /// </summary>
        /// <exception cref="InvalidDataException">
        /// 無効なデータが入力されていた場合は例外をスローします。
        /// コントロール側の制御でそのようなデータは入力できないので、
        /// 例外が発生した時は正しく例外的な事態だと思われます。
        /// </exception>
        private void SyncData()
        {
            // 同期前に値を持っている場合はそれをCloneしておく
            EffectMaker.Foundation.Primitives.Primitivef valueSrc = null;
            if (this.valueTable.Count != 0)
            {
                valueSrc = this.valueTable[0].Value.Clone() as Primitivef;
            }

            this.valueTable.Clear();

            //// ここでDataGridから値を詰め込む
            for (int i = 0; i < this.RowCount; ++i)
            {
                // キーが空欄だったらbreak
                if (Rows[i].Cells[0].Value == null)
                {
                    // 追加途中のrowsについては例外ではなく処理の打ち切りで対応
                    if (i == this.RowCount - 1)
                    {
                        return;
                    }
                    else
                    {
                        // 途中でキーが空欄だったら例外
                        throw new InvalidDataException();
                    }
                }

                int key;
                if (int.TryParse(
                    Rows[i].Cells[0].Value.ToString(),
                    NumberStyles.Integer,
                    null,
                    out key) == false)
                {
                    // 追加途中のrowsについては例外ではなく処理の打ち切りで対応
                    if (i == this.RowCount - 1)
                    {
                        return;
                    }

                    // キーが整数じゃなかったら例外
                    throw new InvalidDataException();
                }

                //// ある行のデータが実数かを判定する
                for (int j = 1; j < this.ColumnCount; ++j)
                {
                    if (Rows[i].Cells[j].Value == null)
                    {
                        return;
                    }

                    string cellValue = Rows[i].Cells[j].Value.ToString();
                    float floatVal = 0.0f;
                    if (float.TryParse(cellValue, out floatVal) == false)
                    {
                        // 追加途中のrowsについては例外ではなく処理の打ち切りで対応
                        if (i == this.RowCount - 1)
                        {
                            return;
                        }

                        // 実数にも変換できないデータがあったら例外を投げる
                        // (そもそもそのようなデータは入力できないようにするのが正しい)
                        throw new InvalidDataException();
                    }
                }

                // クローンした値がある場合はそれを使う
                EffectMaker.Foundation.Primitives.Vector4f value = null;
                if (valueSrc != null)
                {
                    value = valueSrc.Clone() as Vector4f;
                }

                if (value == null)
                {
                    value = new Vector4f();

                    // 次回以降はクローンしたものを使う
                    valueSrc = value.Clone() as Vector4f;
                }

                for (int j = 1; j < this.ColumnCount; ++j)
                {
                    value[j - 1] = float.Parse(Rows[i].Cells[j].Value.ToString());
                }

                this.valueTable.AddKeyFrame(key, value, false);
            }
        }

        /// <summary>
        /// 行番号描画用CellPaintingイベントハンドラ
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void DataGridView_CellPainting(
            object sender,
            DataGridViewCellPaintingEventArgs e)
        {
            // 列ヘッダーかどうか調べる
            if (e.ColumnIndex < 0 && e.RowIndex >= 0)
            {
                // セルを描画する
                e.Paint(e.ClipBounds, DataGridViewPaintParts.All);

                // 行番号を描画する範囲を決定する
                // e.AdvancedBorderStyleやe.CellStyle.Paddingは無視しています
                Rectangle indexRect = e.CellBounds;
                indexRect.Inflate(-2, -2);

                // 行番号を描画する
                TextRenderer.DrawText(
                    e.Graphics,
                    (e.RowIndex + 1).ToString(),
                    e.CellStyle.Font,
                    indexRect,
                    e.CellStyle.ForeColor,
                    TextFormatFlags.Right | TextFormatFlags.VerticalCenter);

                // RGB値による矩形描画
                if (this.IndexDrawMode == 1)
                {
                    float r = float.Parse(this.Rows[e.RowIndex].Cells[1].Value.ToString());
                    float g = float.Parse(this.Rows[e.RowIndex].Cells[2].Value.ToString());
                    float b = float.Parse(this.Rows[e.RowIndex].Cells[3].Value.ToString());
                    Rectangle colorRect = e.CellBounds;
                    colorRect.X += 24;
                    colorRect.Y += 4;
                    colorRect.Width = 12;
                    colorRect.Height = 12;

                    ColorRgba brushColor = this.IsLinearMode
                        ? ColorUtility.NormalizeColor(new ColorRgba(r, g, b, 1.0f))
                        : ColorUtility.SrgbToLinear(ColorUtility.NormalizeColor(new ColorRgba(r, g, b, 1.0f)));

                    using (var brush = new SolidBrush(brushColor.ToWinColor()))
                    using (var pen = new Pen(Color.Black))
                    {
                        e.Graphics.FillRectangle(brush, colorRect);
                        e.Graphics.DrawRectangle(pen, colorRect);
                    }
                }
                else if (this.IndexDrawMode == 2)
                {
                    int a = (int)(255.0f * float.Parse(this.Rows[e.RowIndex].Cells[1].Value.ToString()));
                    Rectangle colorRect = e.CellBounds;
                    colorRect.X += 24;
                    colorRect.Y += 4;
                    colorRect.Width = 12;
                    colorRect.Height = 12;
                    int satA = Math.Min(255, a);
                    using (var brush = new SolidBrush(Color.FromArgb(satA, satA, satA)))
                    using (var pen = new Pen(Color.Black))
                    {
                        e.Graphics.FillRectangle(brush, colorRect);
                        e.Graphics.DrawRectangle(pen, colorRect);
                    }
                }

                // 描画が完了したことを知らせる
                e.Handled = true;
            }
        }
    }
}
