﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UIControls.Specifics.Sliders.ValueTransforms
{
    /// <summary>
    /// Transforms input values using a fixed ratio.
    /// </summary>
    public class FixedRatioValueTransform : IValueTransform<float[]>
    {
        /// <summary>
        /// Stores the boundary value used to determine the zero.
        /// </summary>
        private const float ZeroEpsilon = 1e-6f;

        /// <summary>
        /// Stores the index used to lead the transformation of the other values.
        /// </summary>
        private int leadingIndex;

        /// <summary>
        /// Stores the ratio of each value when initialized.
        /// </summary>
        private float[] ratios, keptRatios;

        /// <summary>
        /// Flag telling whether the leading value is zero or not.
        /// </summary>
        private bool isZero;

        /// <summary>
        /// Initializes the FixedRatioValueTransform instance.
        /// </summary>
        /// <param name="leadingIndex">
        /// The index that lead the transformation
        /// of the other values.
        /// </param>
        /// <param name="argRatios">
        /// The arg Ratios.
        /// </param>
        public FixedRatioValueTransform(int leadingIndex, float[] argRatios)
        {
            if (leadingIndex < 0)
            {
                throw new ArgumentOutOfRangeException("leadingIndex");
            }

            this.leadingIndex = leadingIndex;
            this.keptRatios = argRatios;
        }

        /// <summary>
        /// Initializes the IValueTransform with initial values.
        /// </summary>
        /// <param name="initialValue">The initial values.</param>
        public void Initialize(float[] initialValue)
        {
            if (this.leadingIndex >= initialValue.Length)
            {
                throw new ArgumentOutOfRangeException("leadingIndex");
            }

            float reference = this.keptRatios[this.leadingIndex];

            if (MathUtility.IsZero(reference, ZeroEpsilon))
            {
                this.ratios = null;
                this.isZero = true;
            }
            else
            {
                this.ratios = this.keptRatios.Select(v => v / reference).ToArray();
                this.isZero = false;
            }
        }

        /// <summary>
        /// Transforms the input values using a fixed ratio computed at initialize time.
        /// </summary>
        /// <param name="value">The input values to transform.</param>
        /// <returns>Returns the transformed values.</returns>
        public float[] TransformValue(float[] value)
        {
            if (this.isZero)
            {
                this.Initialize(value);
                return value;
            }

            return this.ratios.Select(ratio => value[this.leadingIndex] * ratio).ToArray();
        }
    }
}
