﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Render.Renderable;

namespace EffectMaker.UIControls.Specifics.CurveEditor.RenderingElements
{
    /// <summary>
    /// グリッド・軸などの座標系を表す描画要素です。
    /// </summary>
    internal class CoordinateSystem : IRenderingElement
    {
        /// <summary>
        /// グリッド用ラインプール
        /// </summary>
        private readonly ManyLineSegments gridSegments = new ManyLineSegments();

        /// <summary>
        /// 区切り用ラインプール
        /// </summary>
        private readonly ManyLineSegments divSegments = new ManyLineSegments();

        /// <summary>
        /// 軸X
        /// </summary>
        private readonly LineSegment axisX;

        /// <summary>
        /// 軸Y
        /// </summary>
        private readonly LineSegment axisY;

        /// <summary>
        /// ラベル配列
        /// </summary>
        private readonly List<TextBlock> labels = new List<TextBlock>();

        /// <summary>
        /// 0固定ピンのイメージ
        /// </summary>
        private readonly ImageRenderable pinImage;

        /// <summary>
        /// グリッドの間隔
        /// </summary>
        private SizeF gridScale = new SizeF(100.0f, 100.0f);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CoordinateSystem()
        {
            this.axisX = new LineSegment
            {
                BorderColor = Color.SlateBlue,
                Vertex1 = new PointF(-1000000.0f, Constants.ViewportOriginY),
                Vertex2 = new PointF(1000000.0f, Constants.ViewportOriginY)
            };
            this.axisY = new LineSegment
            {
                BorderColor = Color.SlateBlue,
                Vertex1 = new PointF(Constants.ViewportOriginX, 1000000.0f),
                Vertex2 = new PointF(Constants.ViewportOriginX, -1000000.0f)
            };
            this.pinImage = new ImageRenderable
            {
                Image = Properties.Resources.Icon_Curve_LockZero_Pin,
                HAlignment = HAlignment.Center,
                VAlignment = VAlignment.Center,
                Padding = new Padding(1, 1, 0, 0)
            };
            this.gridSegments.Vertices = new List<PointF>();
            this.gridSegments.BorderColor = Color.LightGray;
            this.divSegments.Vertices = new List<PointF>();
            this.divSegments.BorderColor = Color.SlateGray;
            this.divSegments.BorderThickness = 1.5f;
        }

        /// <summary>
        /// アップデートします。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="validScale">スケール</param>
        /// <param name="zeroPinVisible">0固定ピンの有無</param>
        /// <param name="postFix">時間ラベルの接尾辞</param>
        /// <param name="normalizeAt">1.0をいくつの値とするか</param>
        /// <param name="labelDigit">値ラベルの小数点以下桁数</param>
        /// <param name="labelPrefix">値ラベルの接頭辞</param>
        public void Update(
            LayeredViewports viewports,
            PointF validScale,
            bool zeroPinVisible,
            string postFix,
            float normalizeAt,
            int labelDigit,
            string labelPrefix)
        {
            this.Hide(viewports);

            PointF tmpScale = CalcGridScale(validScale);
            this.gridScale.Width = tmpScale.X;
            this.gridScale.Height = tmpScale.Y;

            PointF borderPos = viewports.Main.TransformPoint(new PointF(
                viewports.Main.X, viewports.Main.Y + viewports.Main.Height));

            var posLT = viewports.Main.TransformPoint(viewports.Main.Location);
            var posRB = viewports.Main.TransformPoint(
                new PointF(viewports.Main.X + viewports.Main.Width, viewports.Main.Y + viewports.Main.Height));

            //// X軸方向の処理

            var startX = (float)((Math.Ceiling((posLT.X - Constants.ViewportOriginX) / this.gridScale.Width) - 1)
                * this.gridScale.Width) + Constants.ViewportOriginX;
            var endX = (float)((Math.Floor((posRB.X - Constants.ViewportOriginX) / this.gridScale.Width) + 1)
                * this.gridScale.Width) + Constants.ViewportOriginX;

            for (var i = startX; i <= endX; i += this.gridScale.Width)
            {
                var gridX1 = new PointF(i, posLT.Y);

                this.gridSegments.Vertices.Add(gridX1);
                this.gridSegments.Vertices.Add(new PointF(i, posRB.Y));

                PointF labelPos = viewports.Underlay.TransformPoint(
                        viewports.Main.InverseTransformPoint(gridX1));

                PointF tmpPos = viewports.Overlay.TransformPoint(
                        viewports.Underlay.InverseTransformPoint(labelPos));

                if (tmpPos.X < -0.01f || tmpPos.X > viewports.Overlay.Width + 0.01f)
                {
                    continue;
                }

                labelPos.X += 19.0f;
                labelPos.Y = viewports.Overlay.Y + viewports.Overlay.Bounds.Height;

                this.divSegments.Vertices.Add(new PointF(i, borderPos.Y));
                this.divSegments.Vertices.Add(new PointF(i, borderPos.Y - (10.0f / validScale.Y)));

                string labelStr;
                if (this.gridScale.Width < Constants.ViewportFullWidth / 100.0f)
                {
                    labelStr = string.Format(
                        "{0:f1}{1}", Constants.CalcTimeFromPos(gridX1.X, 1), postFix);
                }
                else
                {
                    labelStr = string.Format(
                        "{0:d}{1}", (int)Math.Round(Constants.CalcTimeFromPos(gridX1.X)), postFix);
                }

                this.MakeLabel(labelPos, labelStr);
            }

            //// Y軸方向の処理

            var startY = (float)((Math.Floor((posLT.Y - Constants.ViewportOriginY) / this.gridScale.Height) - 5)
                * this.gridScale.Height) + Constants.ViewportOriginY;
            var endY = (float)((Math.Ceiling((posRB.Y - Constants.ViewportOriginY) / this.gridScale.Height) + 5)
                * this.gridScale.Height) + Constants.ViewportOriginY;

            for (var i = endY; i >= startY; i -= this.gridScale.Height)
            {
                var gridY1 = new PointF(posLT.X, i);
                this.gridSegments.Vertices.Add(gridY1);
                this.gridSegments.Vertices.Add(new PointF(posRB.X, i));

                PointF labelPos =
                    viewports.Underlay.TransformPoint(viewports.Main.InverseTransformPoint(gridY1));

                PointF tmpPos =
                    viewports.Overlay.TransformPoint(viewports.Underlay.InverseTransformPoint(labelPos));

                if (tmpPos.Y < -0.01f || tmpPos.Y > viewports.Overlay.Height + 0.01f)
                {
                    continue;
                }

                labelPos.X = viewports.Main.X + 2.0f;
                labelPos.Y -= 12.0f;

                this.divSegments.Vertices.Add(new PointF(borderPos.X, i));
                this.divSegments.Vertices.Add(new PointF(borderPos.X + (10.0f / validScale.X), i));

                string labelStr;
                if (labelDigit == 0 || this.gridScale.Height > Constants.ViewportFullHeight)
                {
                    labelStr = string.Format(
                        labelPrefix + "{0:d}",
                        (int)Math.Round(Constants.CalcValueFromPos(gridY1.Y, normalizeAt), 0));
                }
                else
                {
                    labelStr = string.Format(
                        labelPrefix + "{0:f2}",
                        Math.Round(Constants.CalcValueFromPos(gridY1.Y, normalizeAt), labelDigit));
                }

                var label = this.MakeLabel(labelPos, labelStr);

                if (labelStr == labelPrefix + "0" || labelStr == labelPrefix + "0.00")
                {
                    var tmpLoc = label.Location;
                    tmpLoc.Y += 14.0f;
                    tmpLoc.X -= 46.0f;
                    this.pinImage.Location = tmpLoc;
                    viewports.Underlay.RemoveRenderable(this.pinImage);
                    if (zeroPinVisible)
                    {
                        label.TextColor = Color.Red;
                        viewports.Underlay.AddRenderable(this.pinImage);
                    }
                }
            }

            this.Show(viewports);
        }

        /// <summary>
        /// 表示します。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Show(LayeredViewports viewports)
        {
            viewports.Underlay.AddRenderableRange(this.labels);
            viewports.Main.AddRenderable(this.gridSegments);
            viewports.Main.AddRenderable(this.divSegments);
            viewports.Main.AddRenderable(this.axisX);
            viewports.Main.AddRenderable(this.axisY);
        }

        /// <summary>
        /// 非表示にします。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Hide(LayeredViewports viewports)
        {
            viewports.Main.RemoveRenderable(this.axisX);
            viewports.Main.RemoveRenderable(this.axisY);

            viewports.Main.RemoveRenderable(this.gridSegments);
            this.gridSegments.Vertices.Clear();
            viewports.Main.RemoveRenderable(this.divSegments);
            this.divSegments.Vertices.Clear();

            viewports.Underlay.RemoveRenderableRange(this.labels);
            foreach (var label in this.labels)
            {
                label.Dispose();
            }

            this.labels.Clear();
        }

        /// <summary>
        /// ラベルの作成
        /// </summary>
        /// <param name="labelPos">ラベルの表示位置</param>
        /// <param name="labelStr">表示文字列</param>
        /// <returns>作成したラベル</returns>
        private TextBlock MakeLabel(PointF labelPos, string labelStr)
        {
            var label = new TextBlock
            {
                Location = labelPos,
                Font = new Font(FontFamily.GenericSansSerif, 9.0f),
                Text = labelStr,
                FillColor = Color.Transparent,
                BorderColor = Color.Transparent,
                TextColor = Color.Purple,
                AutoSize = true,
                HAlignment = HAlignment.Right,
            };
            this.labels.Add(label);

            return label;
        }

        /// <summary>
        /// グリッド間隔の計算
        /// </summary>
        /// <param name="argScale">スケール</param>
        /// <returns>求めたグリッド間隔</returns>
        private static PointF CalcGridScale(PointF argScale)
        {
            var retVal = new PointF();

            var scaleTableX = new List<double>
            {
                0.000000002,
                0.00000001, 0.00000002,
                0.0000001, 0.0000002,
                0.000001, 0.000002,
                0.00001, 0.00002,
                0.0001, 0.0002,
                0.001, 0.002,
                0.01, 0.02,
                0.1, 0.2,
                1.0, 2.0, 5.0, 10.0, 20.0, 50.0, 100.0, 200.0
            };
            double minBorderX = 0.25 * 0.2 * Math.Pow(0.1, 8.0);
            for (int i = 0; i < scaleTableX.Count; ++i)
            {
                if (argScale.X < minBorderX)
                {
                    retVal.X = (float)(Constants.ViewportFullWidth / scaleTableX[i]);
                    break;
                }

                if (i < scaleTableX.Count - 1)
                {
                    minBorderX *= scaleTableX[i + 1] / scaleTableX[i];
                }
                else
                {
                    minBorderX *= 2.0;
                }
            }

            if (minBorderX > 16.0)
            {
                retVal.X = Constants.ViewportFullWidth / 200.0f;
            }

            var scaleTableY = new List<double>()
            {
                0.000000002,
                0.00000001, 0.00000002,
                0.0000001, 0.0000002,
                0.000001, 0.000002,
                0.00001, 0.00002,
                0.0001, 0.0002,
                0.001, 0.002,
                0.01, 0.02,
                0.1, 0.2,
                1.0, 2.0, 4.0, 5.0, 10.0, 20.0, 50.0, 100.0
            };

            double minBorderY = 0.25 * 0.2 * Math.Pow(0.1, 8.0);
            for (int i = 0; i < scaleTableY.Count; ++i)
            {
                if (argScale.Y < minBorderY)
                {
                    retVal.Y = (float)(Constants.ViewportFullHeight / scaleTableY[i]);
                    break;
                }

                if (i < scaleTableY.Count - 1)
                {
                    minBorderY *= scaleTableY[i + 1] / scaleTableY[i];
                }
                else
                {
                    minBorderY *= 2.0;
                }
            }

            if (minBorderY > 16.0)
            {
                retVal.Y = Constants.ViewportFullHeight / 100.0f;
            }

            return retVal;
        }
    }
}
