﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// コントロールが非アクティブな時にも表示できるツールチップです。
    /// </summary>
    internal class ExtendedToolTip
    {
        /// <summary>
        /// ツールチップ
        /// </summary>
        private readonly ToolTip toolTip;

        /// <summary>
        /// 親コントロール
        /// </summary>
        private readonly Control parent;

        /// <summary>
        /// 親の親までハンドラが必要か否か
        /// </summary>
        private readonly bool needGrandParentHandling = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ExtendedToolTip(Control control, bool needGrandParent)
        {
            this.toolTip = new ToolTip
            {
                AutoPopDelay = 1200,
                InitialDelay = 1,
                ReshowDelay = 1
            };
            this.parent = control;
            this.needGrandParentHandling = needGrandParent;
        }

        /// <summary>
        /// コントロールの上にツールチップを乗せます.
        /// </summary>
        /// <param name="ctl">ツールチップを乗せたいコントロール.</param>
        /// <param name="str">乗せるテキスト.</param>
        /// <param name="x">表示位置X</param>
        /// <param name="y">表示位置Y</param>
        /// <param name="d">表示時間</param>
        public void PutToolTip(Control ctl, string str, int x = 20, int y = -25, int d = 2000)
        {
            this.PutToolTipWithLogic(ctl, () => str, x, y, d);
        }

        /// <summary>
        /// コントロールの上にツールチップを乗せます.
        /// </summary>
        /// <param name="ctl">ツールチップを乗せたいコントロール.</param>
        /// <param name="str">乗せるテキストを返す関数.</param>
        /// <param name="x">表示位置X</param>
        /// <param name="y">表示位置Y</param>
        /// <param name="d">表示時間</param>
        public void PutToolTipWithLogic(Control ctl, Func<string> str, int x = 20, int y = -25, int d = 2000)
        {
            ctl.MouseEnter += (s, e) => this.toolTip.Show(str(), ctl, x, y, d);
            ctl.MouseLeave += (s, e) => this.toolTip.Hide(ctl);
            this.PutToolTipForInactiveWithLogic(ctl, str, x, y, d);
        }

        /// <summary>
        /// 非アクティブなコントロールの上にツールチップを乗せます.
        /// </summary>
        /// <param name="ctl">ツールチップを乗せたいコントロール.</param>
        /// <param name="str">乗せるテキストを返す関数.</param>
        /// <param name="x">表示位置X</param>
        /// <param name="y">表示位置Y</param>
        /// <param name="d">表示時間</param>
        private void PutToolTipForInactiveWithLogic(Control ctl, Func<string> str, int x = 20, int y = -25, int d = 2000)
        {
            Action<object, MouseEventArgs> toolTipHanlder = (s, e) =>
            {
                if ((!ctl.Enabled || !this.parent.Enabled) && ctl.Visible)
                {
                    if (this.toolTip.Tag != ctl &&
                        ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                    {
                        this.toolTip.Show(str(), ctl, x, y, d);
                        this.toolTip.Tag = ctl;
                    }
                    else if (this.toolTip.Tag == ctl &&
                        !ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                    {
                        this.toolTip.Hide(ctl);
                        this.toolTip.Tag = null;
                    }
                }
            };
            this.parent.MouseMove += new MouseEventHandler(toolTipHanlder);
            if (this.needGrandParentHandling)
            {
                this.parent.Parent.MouseMove += new MouseEventHandler(toolTipHanlder);
            }
        }
    }
}
