﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブエディタの主にビューポート関連の処理です。
    /// </summary>
    public partial class CurveEditor
    {
        #region ビュー操作 - public

        /// <summary>
        /// 初回表示時の初期化処理です。
        /// エミッタ時間アニメで編集対象を切り替えた際にも使用します。
        /// </summary>
        /// <param name="noResize">ビューポートをリセットしたくない場合はtrue.</param>
        public void Initialize(bool noResize = false)
        {
            // チャンネルの指定がされている場合はそれらを反映する
            if (this.Channels.Count > 0)
            {
                this.manipurator.Clear(this.viewports);

                foreach (var t in this.checkboxList)
                {
                    t.Visibility = Visibility.Hidden;
                }

                this.panel_EditSelection.Visible = false;

                this.manipurator.Initialize(this, this.Channels.Count);
                if (this.Channels.Count > 1)
                {
                    for (int i = 0; i < this.Channels.Count; ++i)
                    {
                        this.panel_EditSelection.Visible = true;
                        this.checkboxList[i].Visibility = Visibility.Visible;
                        this.checkboxList[i].Text = this.Channels[i].Name;
                        this.checkboxList[i].ForeColor = Constants.NodeColors[i];
                    }
                }
            }

            if (!noResize)
            {
                this.ResetViewport();
            }
            else
            {
                this.Rescale();
            }

            if (this.values.Count == 0)
            {
                // 初期キーを追加
                var pos = new PointF(this.CalcPosFromTime(0.0f), this.CalcPosFromValue(this.DefaultValue));
                pos = this.viewports.Main.InverseTransformPoint(pos);
                this.manipurator.AddKey(pos, this.viewports, this.checkboxList, null);

                this.SyncData();
            }
            else
            {
                this.SyncView();
            }

            if (this.IntegerOnly && this.combobox_SnapValue.SelectedIndex < 4)
            {
                this.combobox_SnapValue.SelectedIndex = 4;
            }

            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.SetupTooltips();
            this.SetupHelpButtonForInactive();
            this.UpdateResetButton();
            this.UpdateTarget();
        }

        /// <summary>
        /// カーブのレイアウトに併せてビューポートの位置とスケールを調整します。
        /// </summary>
        /// <param name="timeScale">
        /// タイムラインもキーの最小~最大でクランプするならtrue,フルサイズのままにするならfalse.
        /// </param>
        /// <param name="disableAxis">
        /// 0:両軸とも変更,1:値軸のみ,2:時間軸のみ.
        /// </param>
        public void AdjustViewport(bool timeScale, int disableAxis = 0)
        {
            var maxPos = new PointF(
                this.CalcPosFromTime(0.0f),
                this.CalcPosFromValue(1.0f));
            var minPos = new PointF(
                this.CalcPosFromTime(100.0f),
                this.CalcPosFromValue(0.0f));

            // 全てのキーを含む矩形領域を計算
            // キーがないときはデフォルトの 0～100F、0.0～1.0 の領域にする
            if (timeScale)
            {
                this.manipurator.GetUsingRectangleXY(ref minPos, ref maxPos);
                this.AdjustRectangleTimeScale(ref maxPos, ref minPos);
            }
            else
            {
                this.manipurator.GetUsingRectangleY(ref minPos, ref maxPos);
                this.AdjustRectangleNotTimeScale(ref maxPos, ref minPos);
            }

            // 0固定モードに応じて上下の範囲を計算
            this.AjustRectangleByZeroPinMode(ref maxPos, ref minPos);

            // ビューポートのスケールを設定
            this.AdjustScale(disableAxis, maxPos, minPos);

            // ビューポートの位置を設定
            this.AdjustTranslation(disableAxis, minPos);

            if (this.values.Count > 0)
            {
                this.manipurator.Reconnect(this.viewports);
                this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);

                if (this.dragging)
                {
                    this.MoveNode(this.CalcLockPos(this.viewports.GetClampedMousePosition()));
                }
            }

            this.UpdateTarget();
            this.RequestDraw();
        }

        /// <summary>
        /// 再描画コール
        /// </summary>
        /// <param name="all">全ての要素を更新するか否か(省略可・その場合はビューポートのみを再描画)</param>
        public void RequestDraw(bool all = false)
        {
            if (all)
            {
                this.button_LoopMode.Invalidate();
                this.combobox_InterMode.Invalidate();
                this.text_Frame.Invalidate();
                this.text_Value.Invalidate();
                this.UpdateResetButton();
            }

            this.Invalidate(this.viewports.RenderRect, false);
            this.Update();
        }

        #endregion

        #region ビューアップデート - public

        /// <summary>
        /// リセットボタンの状態更新
        /// </summary>
        public void UpdateResetButton()
        {
            if (!this.Enabled)
            {
                this.button_Reset.Enabled = false;
                this.button_Reset.BackgroundImage = Properties.Resources.Icon_Clear_Lock;
                return;
            }

            bool check = this.values.Count >= 2;

            if (this.values.Count != 0)
            {
                check |= this.values[0].Frame != 0;
                for (int i = 0; i < this.Channels.Count; ++i)
                {
                    check |= this.values[0].Value[i] != this.DefaultValue;
                }
            }

            for (int i = 0; i < this.Channels.Count; ++i)
            {
                check |= this.checkboxList[i].Checked != true;
            }

            check |= this.combobox_SnapFrame.SelectedIndex != 0;
            check |= this.combobox_SnapValue.SelectedIndex != (this.IntegerOnly ? 4 : 2);
            check |= this.otherKeySnapping != true;
            check |= this.ZeroPinMode != this.defaultZeroPin;

            check |= this.loopEnd != 100;
            check |= this.randomize != 0;
            check |= this.loopMode != 0;
            check |= this.interMode != 0;

            check |= Math.Abs(this.ValidScale.X - this.InitialScaleX) > 0.01f;

            var basePoint = new PointF(this.CalcPosFromValue(0.0f), this.CalcPosFromValue(this.NormalizeAt));
            var initialY = this.viewports.Main.Height / (basePoint.X - basePoint.Y);
            check |= Math.Abs(this.ValidScale.Y - initialY) > 0.01f;

            check |= this.viewports.IsInitialTranslation;

            if (check)
            {
                this.button_Reset.Enabled = true;
                this.button_Reset.BackgroundImage = Properties.Resources.Icon_Clear_Off;
            }
            else
            {
                this.button_Reset.Enabled = false;
                this.button_Reset.BackgroundImage = Properties.Resources.Icon_Clear_Lock;
            }
        }

        /// <summary>
        /// カーブ以外のビューポートの状態をリセットします。
        /// </summary>
        public void ResetViewport()
        {
            var basePoint = new PointF(this.CalcPosFromValue(0.0f), this.CalcPosFromValue(this.NormalizeAt));
            this.scale.Y = this.viewports.Main.Height / (basePoint.X - basePoint.Y);

            this.scale.X = this.InitialScaleX;

            this.viewports.Main.Scale = this.ValidScale;
            this.viewports.Main.Update(null);
            this.viewports.Main.Translation = new PointF(0.0f, 0.0f);
            this.LockTranslation(true);
            this.Rescale();
            this.UpdateInvalidArea();

            this.viewports.ResetInitialTranslation();

            this.button_Reset.Enabled = false;
            this.button_Reset.BackgroundImage = Properties.Resources.Icon_Clear_Lock;
        }

        #endregion

        #region ビュー操作 - private

        /// <summary>
        /// 平行移動を調整します。
        /// </summary>
        /// <param name="disableAxis">0:両軸とも変更,1:値軸のみ,2:時間軸のみ.</param>
        /// <param name="minPos">矩形の右下.</param>
        private void AdjustTranslation(int disableAxis, PointF minPos)
        {
            var target = this.viewports.Main.TransformPoint(this.viewports.Main.Location);
            var now = this.viewports.Main.Translation;

            if (disableAxis == 0)
            {
                this.viewports.Main.Translation = new PointF(
                    now.X + target.X - minPos.X,
                    now.Y + target.Y - minPos.Y);
            }
            else if (disableAxis == 1)
            {
                this.viewports.Main.Translation = new PointF(
                    now.X, now.Y + target.Y - minPos.Y);
            }
            else
            {
                this.viewports.Main.Translation = new PointF(
                    now.X + target.X - minPos.X, now.Y);
            }

            this.Rescale();
        }

        /// <summary>
        /// スケールを調整します。
        /// </summary>
        /// <param name="disableAxis">0:両軸とも変更,1:値軸のみ,2:時間軸のみ.</param>
        /// <param name="maxPos">矩形の左上.</param>
        /// <param name="minPos">矩形の右下.</param>
        private void AdjustScale(int disableAxis, PointF maxPos, PointF minPos)
        {
            if (disableAxis == 0)
            {
                this.scale.X = this.viewports.Main.Width / (maxPos.X - minPos.X);
                this.scale.Y = this.viewports.Main.Height / (maxPos.Y - minPos.Y);
            }
            else if (disableAxis == 1)
            {
                this.scale.Y = this.viewports.Main.Height / (maxPos.Y - minPos.Y);
            }
            else
            {
                this.scale.X = this.viewports.Main.Width / (maxPos.X - minPos.X);
            }

            this.viewports.Main.Scale = this.ValidScale;
            this.viewports.Main.Update(null);
        }

        /// <summary>
        /// ZeroPinMode で矩形を調整します。
        /// </summary>
        /// <param name="maxPos">矩形の左上.</param>
        /// <param name="minPos">矩形の右下.</param>
        private void AjustRectangleByZeroPinMode(ref PointF maxPos, ref PointF minPos)
        {
            switch (this.ZeroPinMode)
            {
                case 0:
                    minPos.Y = this.CalcPosFromValue(0.0f);
                    break;
                case 1:
                    var minVal = this.CalcValueFromPos(minPos.Y);
                    var maxVal = this.CalcValueFromPos(maxPos.Y);
                    if (Math.Abs(minVal) > Math.Abs(maxVal))
                    {
                        minPos.Y = this.CalcPosFromValue(Math.Abs(minVal));
                        maxPos.Y = this.CalcPosFromValue(-Math.Abs(minVal));
                    }
                    else
                    {
                        minPos.Y = this.CalcPosFromValue(Math.Abs(maxVal));
                        maxPos.Y = this.CalcPosFromValue(-Math.Abs(maxVal));
                    }

                    break;
                case 2:
                case -1:
                    maxPos.Y = this.CalcPosFromValue(0.0f);
                    break;
            }
        }

        /// <summary>
        /// 矩形を調整します（横軸のカーブフィッティングあり）。
        /// </summary>
        /// <param name="maxPos">矩形の左上.</param>
        /// <param name="minPos">矩形の右下.</param>
        private void AdjustRectangleTimeScale(ref PointF maxPos, ref PointF minPos)
        {
            if (this.EditorMode == EditorModes.ParticleTimeAnim)
            {
                // パーティクル時間ならタイムラインは0～100で固定
                minPos.X = this.CalcPosFromTime(0.0f);
                maxPos.X = this.CalcPosFromTime(this.MaxFrame);
            }
            else if (Math.Abs(maxPos.X - minPos.X) < 0.0001f)
            {
                minPos.X = this.CalcPosFromTime(0.0f);
                maxPos.X = this.CalcPosFromTime(this.MaxFrame);
            }

            // 同値のときはキーの値±0.5f
            if (Math.Abs(maxPos.Y - minPos.Y) < 0.0001f)
            {
                var tmpValue = this.CalcValueFromPos(maxPos.Y);
                minPos.Y = this.CalcPosFromValue(Math.Min(this.MaxLimit, tmpValue + 0.5f));
                maxPos.Y = this.CalcPosFromValue(Math.Max(this.MinLimit, tmpValue - 0.5f));
            }
        }

        /// <summary>
        /// 矩形を調整します（横軸のカーブフィッティングなし）。
        /// </summary>
        /// <param name="maxPos">矩形の左上.</param>
        /// <param name="minPos">矩形の右下.</param>
        private void AdjustRectangleNotTimeScale(ref PointF maxPos, ref PointF minPos)
        {
            // タイムラインは0～100で固定
            minPos.X = this.CalcPosFromTime(0.0f);
            maxPos.X = this.CalcPosFromTime(this.MaxFrame);

            var tmpMax = this.CalcValueFromPos(maxPos.Y);
            var tmpMin = this.CalcValueFromPos(minPos.Y);

            var normalized = this.EditorMode == EditorModes.ParticleTimeAnim ? 1.0f : this.NormalizeAt;

            // 0で同値の場合は-1～1
            if (Math.Abs(tmpMin) < 0.0001f && Math.Abs(tmpMax) < 0.0001f)
            {
                minPos.Y = this.CalcPosFromValue(Math.Min(this.MaxLimit, normalized));
                maxPos.Y = this.CalcPosFromValue(Math.Max(this.MinLimit, -normalized));
            }
            else
            {
                if (minPos.Y >= this.CalcPosFromValue(0.0f))
                {
                    minPos.Y = this.CalcPosFromValue(0.0f) - 0.01f;
                }
                else if (minPos.Y > this.CalcPosFromValue(Math.Min(this.MaxLimit, normalized)))
                {
                    minPos.Y = this.CalcPosFromValue(Math.Min(this.MaxLimit, normalized)) - 0.01f;
                }

                if (maxPos.Y <= this.CalcPosFromValue(0.0f))
                {
                    maxPos.Y = this.CalcPosFromValue(0.0f);
                }
                else if (maxPos.Y < this.CalcPosFromValue(Math.Max(this.MinLimit, -normalized)))
                {
                    maxPos.Y = this.CalcPosFromValue(Math.Max(this.MinLimit, -normalized));
                }
            }
        }

        /// <summary>
        /// センタードラッグによるスクロール、軸単位の拡縮を行います。
        /// </summary>
        /// <param name="argLoc">マウス座標</param>
        private void ScrollViewport(PointF argLoc)
        {
            this.viewports.Scroll(argLoc, ref this.scale, () => this.ValidScale);
            this.LockTranslation();
            this.Rescale();
            if (this.values.Count > 0)
            {
                this.manipurator.Reconnect(this.viewports);
                this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            }

            if (this.dragging)
            {
                this.MoveNode(this.CalcLockPos(argLoc));
            }
            else
            {
                this.UpdateSelection(argLoc, false);
            }
        }

        /// <summary>
        /// ホイールスクロールによるズームを処理します。
        /// </summary>
        /// <param name="e">ホイールイベントの引数</param>
        private void ZoomViewport(MouseEventArgs e)
        {
            PointF originPos = this.viewports.Main.InverseTransformPoint(
                new PointF(this.CalcPosFromTime(0.0f), this.CalcPosFromValue(0.0f)));
            this.viewports.Zoom(e.Delta, originPos, this.ZeroPinMode, ref this.scale, () => this.ValidScale);
            this.LockTranslation();
            this.Rescale();

            if (this.values.Count > 0)
            {
                this.manipurator.Reconnect(this.viewports);
                this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
                this.UpdateTarget();
            }

            if (!this.dragging)
            {
                this.UpdateSelection(this.PointToClient(Cursor.Position), false);
            }
        }

        /// <summary>
        /// 非表示領域が出ないようにスクロールを固定します。
        /// </summary>
        /// <param name="forceUnder">強制的に下辺を0に固定するフラグ</param>
        private void LockTranslation(bool forceUnder = false)
        {
            var target = this.viewports.Main.TransformPoint(this.viewports.Main.Location);
            var targetEnd = this.viewports.Main.TransformPoint(new PointF(
                this.viewports.Main.X + this.viewports.Main.Width,
                this.viewports.Main.Y + this.viewports.Main.Height));

            var now = this.viewports.Main.Translation;
            var minX = now.X + target.X - this.CalcPosFromTime(0.0f);
            var maxX = now.X + targetEnd.X - this.CalcPosFromTime(this.MaxCurrentModeTime);
            if (now.X > minX)
            {
                this.viewports.Main.Translation = new PointF(minX + 0.00001f, now.Y);
            }
            else if (now.X < maxX - 0.0001f)
            {
                this.viewports.Main.Translation = new PointF(maxX - 0.00001f, now.Y);
            }

            now = this.viewports.Main.Translation;
            int zeroIdx = this.ZeroPinMode;

            // 「無し」か「非表示(マイナス域なし)」の場合
            if ((zeroIdx == 3 || zeroIdx == -1) && !forceUnder)
            {
                var minY = now.Y + target.Y - this.CalcPosFromValue(this.MaxLimit);
                var maxY = now.Y + targetEnd.Y - this.CalcPosFromValue(-this.MaxLimit);
                if (now.Y > minY)
                {
                    this.viewports.Main.Translation = new PointF(now.X, minY);
                }
                else if (now.Y < maxY)
                {
                    this.viewports.Main.Translation = new PointF(now.X, maxY);
                }
            }
            else
            {
                float offset = this.viewports.Main.Height * 1.0f;
                if (this.MinLimit < 0.0f && !forceUnder)
                {
                    switch (this.ZeroPinMode)
                    {
                        case 0:
                            offset = 0.001f;
                            break;
                        case 1:
                            offset = this.viewports.Main.Height * 0.5f;
                            break;
                        case 2:
                            offset = this.viewports.Main.Height * 1.0f;
                            break;
                    }
                }

                var targetCenter = this.viewports.Main.TransformPoint(new PointF(
                    this.viewports.Main.Location.X,
                    this.viewports.Main.Location.Y + offset));

                float targetY = now.Y + targetCenter.Y - this.CalcPosFromValue(0.0f);

                if (zeroIdx == 1 ||
                    (zeroIdx == 0 && now.Y > targetY) ||
                    (zeroIdx == 2 && now.Y < targetY) ||
                    forceUnder)
                {
                    this.viewports.Main.Translation = new PointF(now.X, targetY);
                }
            }
        }

        /// <summary>
        /// 時間テキストボックスの内容をカーブに適用します。
        /// </summary>
        private void ApplyTextFrame()
        {
            if (this.targetNode == null)
            {
                return;
            }

            int tmp;

            if (!int.TryParse(this.text_Frame.Text, out tmp))
            {
                this.text_Frame.Text = this.textFrameBackup;
                return;
            }

            if (this.text_Frame.Text == this.textFrameBackup)
            {
                return;
            }

            if (this.values.Any(keyframe => keyframe.Frame == tmp))
            {
                this.text_Frame.Text = this.textFrameBackup;
                return;
            }

            this.CheckDeselection();
            var pos = this.manipurator.GetNodePos(this.targetNode);
            pos.X = this.CalcPosFromTime(tmp);
            this.MoveNode(this.viewports.Main.InverseTransformPoint(pos));

            if (tmp < 0)
            {
                tmp = 0;
            }
            else if (tmp > this.MaxCurrentModeTime)
            {
                tmp = (int)this.MaxCurrentModeTime;
            }

            this.text_Frame.Text = tmp.ToString("D");
            this.textFrameBackup = this.text_Frame.Text;

            // データを更新して履歴を積む
            this.SyncData();
            this.RequestDraw();
        }

        /// <summary>
        /// 値テキストボックスの内容をカーブに適用します。
        /// </summary>
        private void ApplyTextValue()
        {
            if (this.targetNode == null)
            {
                return;
            }

            float tmp;

            if (!float.TryParse(this.text_Value.Text, out tmp))
            {
                this.text_Value.Text = this.textValueBackup;
                return;
            }

            if (this.text_Value.Text == this.textValueBackup)
            {
                return;
            }

            this.CheckDeselection();
            var pos = this.manipurator.GetNodePos(this.targetNode);
            pos.Y = this.CalcPosFromValue(tmp);
            this.noSnapping = true;
            this.MoveNode(this.viewports.Main.InverseTransformPoint(pos));
            this.noSnapping = false;

            if (tmp < this.MinLimit)
            {
                tmp = this.MinLimit;
            }
            else if (tmp > this.MaxLimit)
            {
                tmp = this.MaxLimit;
            }

            this.text_Value.Text = this.IntegerOnly ? ((int)tmp).ToString("D") : tmp.ToString("F4");
            this.textValueBackup = this.text_Value.Text;

            // データを更新して履歴を積む
            this.SyncData();
            this.RequestDraw();
        }

        /// <summary>
        /// ループ終端テキストボックスの内容を適用します。
        /// </summary>
        /// <param name="passNotify">履歴に積みたくない時はtrue、積む時はfalse.</param>
        private void ApplyLoopEnd(bool passNotify = false)
        {
            int tmp;

            if (!int.TryParse(this.text_LoopEnd.Text, out tmp))
            {
                this.text_LoopEnd.Text = this.textLoopEndBackup;
                return;
            }

            if (tmp < 1)
            {
                if (passNotify)
                {
                    this.text_LoopEnd.Text = tmp.ToString();
                    return;
                }

                tmp = 1;
            }
            else if (tmp > 65535)
            {
                if (passNotify)
                {
                    this.text_LoopEnd.Text = tmp.ToString();
                    return;
                }

                tmp = 65535;
            }

            if (passNotify)
            {
                this.text_LoopEnd.Text = tmp.ToString();
                this.LogicalTreeElementExtender.SetValue(
                    ref this.loopEnd,
                    tmp,
                    BindingUpdateType.PropertyChanged,
                    "LoopEnd");
            }
            else
            {
                this.LoopEnd = tmp;
            }

            this.textLoopEndBackup = this.text_LoopEnd.Text;
        }


        #endregion

        #region ビューアップデート - private

        /// <summary>
        /// マウスオーバーした際の選択マーカーを更新する。
        /// </summary>
        private void UpdateSelectedMarker()
        {
            if (this.viewports == null)
            {
                return;
            }

            this.viewports.Overlay.RemoveRenderable(this.selectedMarker);
            this.framePopup.Hide(this.viewports);
            if (this.selectedNodes.Count <= 0)
            {
                return;
            }

            this.viewports.Overlay.AddRenderable(this.selectedMarker);
            this.selectedMarker.Center = this.selectedNodes[0].Location + Constants.NodeCenterOffset;
            this.UpdateSelectedFrame();
        }

        /// <summary>
        /// ノードにカーソルを当てている際のフレーム数ポップアップを更新します。
        /// </summary>
        private void UpdateSelectedFrame()
        {
            this.framePopup.Hide(this.viewports);
            if (this.selectedNodes.Count <= 0 || this.EditorMode == EditorModes.EmitterTimeAnim)
            {
                return;
            }

            string tempTxt;
            if (this.EditorMode == EditorModes.ParticleTimeAnim && this.GetParticleLifeExecutable != null)
            {
                var reciever = new float[1];
                this.GetParticleLifeExecutable.Execute(reciever);
                tempTxt = string.Format(
                    "{0:d}f",
                    (int)Math.Round(reciever[0] * 0.01f * this.CalcTimeFromPos(this.manipurator.GetNodePos(this.selectedNodes[0]).X)));
            }
            else
            {
                tempTxt = string.Format(
                    "{0:d}{1}",
                    (int)Math.Round(this.CalcTimeFromPos(this.manipurator.GetNodePos(this.selectedNodes[0]).X)),
                    this.EditorMode == EditorModes.ParticleTimeAnim ? "%" : "f");
            }

            this.framePopup.Update(this.viewports, tempTxt, this.selectedNodes[0].Location);
            this.framePopup.Show(this.viewports);
        }

        /// <summary>
        /// ズームに伴う表示要素のサイズ変更
        /// </summary>
        private void Rescale()
        {
            this.manipurator.Rescale(this.viewports);

            if (this.targetNode != null)
            {
                this.UpdateTarget();
            }

            this.UpdateGrid();

            this.UpdateSelectedMarker();
        }

        /// <summary>
        /// クライアントのサイズ変更に伴うビューポートのサイズ調整
        /// </summary>
        private void ResizeViewport()
        {
            if (this.viewports == null)
            {
                return;
            }

            this.viewports.Resize(this);
            this.UpdateGrid();
        }

        /// <summary>
        /// 無効エリアの更新
        /// </summary>
        private void UpdateInvalidArea()
        {
            if (this.viewports == null || this.invalidArea == null)
            {
                return;
            }

            this.invalidArea.Update(
                this.viewports.Main,
                this.Enabled,
                this.CalcPosFromTime(this.MaxCurrentModeTime),
                this.CalcPosFromValue(this.MaxLimit),
                this.CalcPosFromValue(this.MinLimit));
        }

        /// <summary>
        /// マウスオーバーによるノード選択状態の更新
        /// </summary>
        /// <param name="argLoc">マウス座標</param>
        /// <param name="targetUpdate">ターゲットマーカーの更新を行うか否か</param>
        /// <returns>選択キーがあればtrue,そうでなければfalse.</returns>
        private bool UpdateSelection(PointF argLoc, bool targetUpdate)
        {
            this.selectedNodes.Clear();
            this.selectedNodes.AddRange(this.manipurator.PickNodes(this.viewports, argLoc, this.checkboxList));

            // 選択されたキーがあったら
            if (this.selectedNodes.Count != 0)
            {
                this.UpdateSelectedMarker();

                if (targetUpdate)
                {
                    this.targetNode = this.selectedNodes[0];
                }

                this.UpdateTarget();

                return true;
            }

            // なかったら
            this.viewports.Overlay.RemoveRenderable(this.selectedMarker);
            this.framePopup.Hide(this.viewports);

            return false;
        }

        /// <summary>
        /// 編集ターゲットを更新します。
        /// </summary>
        private void UpdateTarget()
        {
            if (this.viewports == null)
            {
                return;
            }

            if (this.targetNode != null)
            {
                var tmpPos = this.targetNode.Location + Constants.NodeCenterOffset;
                tmpPos.X = (int)(tmpPos.X + 0.5f);
                tmpPos.Y = (int)(tmpPos.Y + 0.5f);
                if (!this.viewports.IsInnerMouseSpace(this.viewports.Overlay.InverseTransformPoint(tmpPos)))
                {
                    this.viewports.Overlay.RemoveRenderable(this.targetMarker);
                    this.UpdateTextBox();
                    return;
                }

                this.viewports.Overlay.RemoveRenderable(this.targetMarker);
                this.viewports.Overlay.AddRenderable(this.targetMarker);
                this.targetMarker.Location = new Point((int)tmpPos.X, (int)tmpPos.Y);
            }
            else
            {
                this.viewports.Overlay.RemoveRenderable(this.targetMarker);
            }

            this.UpdateTextBox();
        }

        /// <summary>
        /// 現在の編集ターゲットの位置でもう一度ノードの選択をやり直します。
        /// 軸選択チェックボックスの状態が変化した時などの対応用です。
        /// </summary>
        private void CheckDeselection()
        {
            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            if (this.targetNode != null)
            {
                var point = this.viewports.Overlay.InverseTransformPoint(this.targetNode.Location + Constants.NodeCenterOffset);
                this.UpdateSelection(point, false);
                this.viewports.Overlay.RemoveRenderable(this.selectedMarker);
                this.framePopup.Hide(this.viewports);
                if (this.selectedNodes.Count == 0)
                {
                    this.targetNode = null;
                    this.UpdateTarget();
                }
            }
        }

        /// <summary>
        /// グリッドと軸ラベルの更新
        /// </summary>
        private void UpdateGrid()
        {
            if (this.viewports == null || this.coordinateSystem == null)
            {
                return;
            }

            this.coordinateSystem.Update(
                this.viewports,
                this.ValidScale,
                this.ZeroPinMode != 3 && this.ZeroPinMode != -1,
                this.EditorMode == EditorModes.ParticleTimeAnim ? "%" : string.Empty,
                this.NormalizeAt,
                this.LabelDigit,
                this.LabelPrefix);

            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.UpdateInvalidArea();
        }

        /// <summary>
        /// テキストボックスの表示内容を更新します。
        /// </summary>
        private void UpdateTextBox()
        {
            if (this.targetNode == null)
            {
                this.text_Frame.Text = string.Empty;
                this.text_Frame.Enabled = false;
                this.text_Value.Text = string.Empty;
                this.text_Value.Enabled = false;
                this.viewports.Overlay.RemoveRenderable(this.targetMarker);
                return;
            }

            var pos = this.manipurator.GetNodePos(this.targetNode);

            this.text_Frame.Enabled = true;
            this.text_Frame.Text = ((int)this.CalcTimeFromPos(pos.X)).ToString("D");
            this.text_Value.Enabled = true;
            this.text_Value.Text = this.IntegerOnly ?
                ((int)Math.Round(this.CalcValueFromPos(pos.Y))).ToString("D") :
                this.CalcValueFromPos(pos.Y).ToString("F4");
        }

        #endregion

        #region ユーティリティ - private

        /// <summary>
        /// スクロールをずらさずにカーブエディタにフォーカスを当てる
        /// </summary>
        private void FocusWithoutScroll()
        {
            UIPanel panel = null;
            Control test = this.Parent;
            while (test != null)
            {
                if (test is UIPanel)
                {
                    panel = test as UIPanel;
                    if (panel.AutoScroll)
                    {
                        break;
                    }

                    panel = null;
                }

                test = test.Parent;
            }

            if (panel != null)
            {
                var prev = panel.FocusScrollEnable;
                panel.FocusScrollEnable = false;
                this.text_FocusDummy.Select();
                panel.FocusScrollEnable = prev;
            }
            else if (this.EditorMode == EditorModes.EmitterTimeAnim)
            {
                this.text_FocusDummy.Select();
            }
        }

        #endregion
    }
}
