﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.TabPages;
using ColorClipboardData = EffectMaker.Foundation.ClipboardDataTypes.ColorClipboardData;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// カラーピッカー
    /// </summary>
    public partial class UIColorPicker : UIUserControl
    {
        /// <summary>
        /// NtbTime_ValueChanged処理を有効にするか？
        /// </summary>
        private bool enableNtbTimeValueChanged = true;

        /// <summary>
        /// 値変更処理を有効にするか？
        /// </summary>
        private bool valueEditEnabled = true;

        /// <summary>
        /// 色
        /// </summary>
        private ColorRgba value = new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

        /// <summary>
        /// RGB編集を有効にするか？
        /// </summary>
        private bool rgbEditEnabled;

        /// <summary>
        /// アルファ編集を有効にするか？
        /// </summary>
        private bool alphaEditEnabled;

        /// <summary>
        /// IsLinearMode プロパティのバッキングフィールドです。
        /// </summary>
        private bool isLinearMode = false;

        /// <summary>
        /// ガンマ補正を有効にするかどうか。
        /// </summary>
        private bool gammaEnabled = false;

        /// <summary>
        /// タブオーダーのロック
        /// </summary>
        private bool lockingTabOrder = false;

        /// <summary>
        /// The blocking focus.
        /// </summary>
        private bool blockingFocus = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UIColorPicker()
        {
            this.InitializeComponent();

            // 時間ラベルのテキスト
            this.lblTime.Text = Properties.Resources.ColorPickerLabelTime;

            // 初期の編集モードを "sRGB" にする
            this.cbxColorMode.SelectedIndex = 1;

            this.ntbTime.Deformat = src =>
            {
                src = src.Trim();

                // 文字列最後のパーセント記号を削る
                if (src.Substring(src.Length - 1, 1) == "%")
                {
                    src = src.Substring(0, src.Length - 1);
                }

                return src;
            };

            this.InvalidateInnerValue();
            this.InitializeContextMenu();

            this.uiTextBox1.GotFocus += (s, e) =>
            {
                if (this.blockingFocus)
                {
                    return;
                }

                bool isShiftDown = (Control.ModifierKeys & Keys.Shift) == Keys.Shift;

                ////if (isShiftDown && this.lockingTabOrder)
                ////{
                ////    this.blockingFocus = true;
                ////    this.uiTextBox2.Focus();
                ////    this.blockingFocus = false;
                ////}

                this.FocusNext(!isShiftDown);
            };

            this.uiTextBox2.GotFocus += (s, e) =>
            {
                if (this.blockingFocus)
                {
                    return;
                }

                bool isShiftDown = (Control.ModifierKeys & Keys.Shift) == Keys.Shift;

                ////if (!isShiftDown && this.lockingTabOrder)
                ////{
                ////    this.blockingFocus = true;
                ////    this.uiTextBox1.Focus();
                ////    this.blockingFocus = false;
                ////}

                this.FocusNext(!isShiftDown);
            };

            this.ntbTime.ToolTipDuration = -1;
            this.ntbTime.ToolTipPoint = new Point(this.ntbTime.Size.Width + 1, this.ntbTime.Size.Height + 1);
        }

        /// <summary>
        /// Event triggered when the color value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged;

        /// <summary>
        /// 色
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                return (ColorRgba)this.value.Clone();
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                if (this.valueEditEnabled == false)
                {
                    return;
                }

                this.value.Set(value);

                this.InvalidateInnerValue();
             }
        }

        /// <summary>
        /// 時間
        /// </summary>
        public int Time
        {
            get
            {
                return (int)this.ntbTime.Value;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                this.ntbTime.Value = (float)value;
            }
        }

        /// <summary>
        /// データコンテキスト切り替え中は強制的に無効化
        /// </summary>
        public new bool Enabled
        {
            get
            {
                return base.Enabled;
            }

            set
            {
                base.Enabled = !this.IsUpdatingDataContext && value;
            }
        }

        /// <summary>
        /// 編集中色
        /// </summary>
        public ColorRgba EditingValue
        {
            get
            {
                return this.Value.Clone() as ColorRgba;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                using (new AnonymousDisposable(() => this.valueEditEnabled = true))
                {
                    this.valueEditEnabled = false;

                    this.InvalidateInnerValue();
                }
            }
        }

        /// <summary>
        /// 編集中時間
        /// </summary>
        public int EditingTime
        {
            get
            {
                return this.Time;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                using (new AnonymousDisposable(() => this.enableNtbTimeValueChanged = true))
                {
                    this.enableNtbTimeValueChanged = false;

                    this.ntbTime.Value = (float)value;
                }
            }
        }

        /// <summary>
        /// 編集中時間のツールチップテキスト
        /// </summary>
        public string EditingTimeToolTipText
        {
            get
            {
                return this.ntbTime.ToolTipText;
            }

            set
            {
                this.ntbTime.ToolTipText = value;
            }
        }

        /// <summary>
        /// 時間を有効にするか？
        /// </summary>
        public bool TimeEditEnabled
        {
            get
            {
                return this.ntbTime.Enabled;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                this.ntbTime.Enabled = value;
            }
        }

        /// <summary>
        /// 時間編集UIの表示・非表示を設定
        /// </summary>
        public bool TimeEditVisibale
        {
            get
            {
                return this.ntbTime.Visible;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                this.lblTime.Visibility = value ? Visibility.Visible : Visibility.Hidden;
                this.ntbTime.Visible = value;
            }
        }

        /// <summary>
        /// RGB編集を有効にするか？
        /// </summary>
        public bool RgbEditEnabled
        {
            get
            {
                return this.rgbEditEnabled;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                this.rgbEditEnabled = value;
                this.UpdateColorControlEnabled();
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "RgbEditEnabled");
            }
        }

        /// <summary>
        /// アルファ編集を有効にするか？
        /// </summary>
        public bool AlphaEditEnabled
        {
            get
            {
                return this.alphaEditEnabled;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                this.alphaEditEnabled = value;
                this.UpdateColorControlEnabled();
                this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "AlphaEditEnabled");
            }
        }

        /// <summary>
        /// 最大時間
        /// </summary>
        public int MaxTime
        {
            get
            {
                return (int)this.ntbTime.Maximum;
            }

            set
            {
                if (this.IsUpdatingDataContext)
                {
                    return;
                }

                this.ntbTime.Maximum = (float)value;
            }
        }

        /// <summary>
        /// リニア編集モードかどうか取得または設定します。
        /// </summary>
        public bool IsLinearMode
        {
            get
            {
                return this.isLinearMode;
            }

            set
            {
                if (value != this.isLinearMode)
                {
                    this.isLinearMode = value;

                    // リニア編集モードに合わせてカラーモードを変更する
                    if (this.isLinearMode == false && this.cbxColorMode.SelectedIndex == 0)
                    {
                        this.cbxColorMode.SelectedIndex = 1;
                    }
                    else if (this.isLinearMode == true && this.cbxColorMode.SelectedIndex == 1)
                    {
                        this.cbxColorMode.SelectedIndex = 0;
                    }

                    this.InvalidateInnerValue();
                }
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether is locking tab order.
        /// </summary>
        public bool IsLockingTabOrder
        {
            get
            {
                return this.lockingTabOrder;
            }

            set
            {
                this.lockingTabOrder = value;

                if (this.lockingTabOrder)
                {
                    this.Focus();
                }
            }
        }

        /// <summary>
        /// Gets or sets the IExecutable to run when the Value property changes.
        /// </summary>
        public IExecutable ValueChangedExecutable { get; set; }

        /// <summary>
        /// Gets or sets the custom parameter of the IExecutable
        /// to run when the Value property changes.
        /// </summary>
        public object ValueChangedExecutableParameter { get; set; }

        /// <summary>
        /// 内部カラーの Invalidate を行います。
        /// </summary>
        private void InvalidateInnerValue()
        {
            ColorRgba innerValue = this.Value;

            // 編集モードに合わせて値を変換する
            if (this.IsLinearMode == false && this.gammaEnabled == true)
            {
                innerValue = ColorUtility.SrgbToLinear(innerValue);
            }
            else if (this.IsLinearMode == true && this.gammaEnabled == false)
            {
                innerValue = ColorUtility.LinearToSrgb(innerValue);
            }

            this.hcpPanel.ColorRGBA = innerValue;
            this.spcRgb.Value = innerValue;
            this.spaAlpha.Value = innerValue;

            ColorRgba backColor = this.IsLinearMode ? ColorUtility.SrgbToLinear(this.Value) : this.Value;
            byte alpha = (byte)ColorUtility.ClampByte(backColor.A * 255.0f);

            this.pnlRgb.BackColor = ColorUtility.NormalizeColor(backColor).ToWinColor();
            this.pnlAlpha.BackColor = Color.FromArgb(alpha, alpha, alpha);
        }

        /// <summary>
        /// カラーパネル値変更イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void HcpPanel_ValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var oldValue = new ColorRgba(this.Value);

            ColorRgba value = this.hcpPanel.ColorRGBA;

            // 編集モードに合わせて値を変換する
            if (this.IsLinearMode == false && this.gammaEnabled == true)
            {
                value = ColorUtility.LinearToSrgb(value);
            }
            else if (this.IsLinearMode == true && this.gammaEnabled == false)
            {
                value = ColorUtility.SrgbToLinear(value);
            }

            this.Value = value;

            this.InvokeValueChangedEvent(e);

            this.TriggerValueChangedExecutable(
                new ValueChangedExEventArgs(
                    oldValue,
                    this.Value.Clone(),
                    e.Changing,
                    this.ValueChangedExecutableParameter));

            // マウスアップをロストすることがあるため無効化
            ////Application.DoEvents();
        }

        /// <summary>
        /// RGB値変更イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void SpcRgb_ValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var oldValue = new ColorRgba(this.Value);

            ColorRgba value = new ColorRgba(this.spcRgb.Value.R, this.spcRgb.Value.G, this.spcRgb.Value.B, this.spaAlpha.Value.A);

            // 編集モードに合わせて値を変換する
            if (this.IsLinearMode == false && this.gammaEnabled == true)
            {
                value = ColorUtility.LinearToSrgb(value);
            }
            else if (this.IsLinearMode == true && this.gammaEnabled == false)
            {
                value = ColorUtility.SrgbToLinear(value);
            }

            this.Value = value;

            this.InvokeValueChangedEvent(e);

            this.TriggerValueChangedExecutable(
                new ValueChangedExEventArgs(
                    oldValue,
                    this.Value.Clone(),
                    e.Changing,
                    this.ValueChangedExecutableParameter));

            // マウスアップをロストすることがあるため無効化
            ////Application.DoEvents();
        }

        /// <summary>
        /// アルファ値変更イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void SpaAlpha_ValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            this.SpcRgb_ValueChanged(sender, e);
        }

        /// <summary>
        /// 値変更後イベントの発行
        /// </summary>
        /// <param name="e">イベント引数</param>
        private void InvokeValueChangedEvent(SequentialValueChangedEventArgs e)
        {
            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, e);
            }
        }

        /// <summary>
        /// 色モード変更イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void CbxColorMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            System.Diagnostics.Debug.Assert(sender is ComboBox, "sender is ComboBox");

            ColorModeType colorMode = ColorModeType.RGB;
            this.gammaEnabled = false;

            switch ((sender as ComboBox).SelectedIndex)
            {
                case 0:
                    colorMode = ColorModeType.RGB;
                    this.gammaEnabled = true;
                    break;
                case 1:
                    colorMode = ColorModeType.sRGB;
                    this.gammaEnabled = false;
                    break;
                case 2:
                    colorMode = ColorModeType.HSV;
                    this.gammaEnabled = this.IsLinearMode;
                    break;
            }

            this.spcRgb.ColorMode = colorMode;
            this.hcpPanel.GammaEnabled = this.gammaEnabled;

            this.InvalidateInnerValue();
        }

        /// <summary>
        /// Trigger the value changed executable event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        private void TriggerValueChangedExecutable(ValueChangedExEventArgs e)
        {
            IExecutable executable = this.ValueChangedExecutable;
            if (executable != null && executable.CanExecute(e))
            {
                if (e.CustomParameter is string)
                {
                    e = new ValueChangedExEventArgs(
                        e.OldValue,
                        e.NewValue,
                        e.IsChanging,
                        e.CustomParameter,
                        (string)e.CustomParameter != "ColorPickerValue");
                }

                executable.Execute(e);
            }

            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "EditingValue");
        }

        /// <summary>
        /// 時間入力コントロールEnabled変更イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void NtbTime_EnabledChanged(object sender, EventArgs e)
        {
            this.lblTime.Enabled = this.ntbTime.Enabled;
        }

        /// <summary>
        /// 時間入力コントロール値変更イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void NtbTime_ValueChanged(object sender, EventArgs e)
        {
            if (this.enableNtbTimeValueChanged == false)
            {
                return;
            }

            this.LogicalTreeElementExtender.NotifyPropertyChanged(BindingUpdateType.PropertyChanged, "Time");
        }

        /// <summary>
        /// 時間入力クリックイベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void NtbTime_Click(object sender, EventArgs e)
        {
            if (OptionStore.RootOptions.Interface.ClickToSelectAllInTextBox)
            {
                this.ntbTime.SelectAll();
            }
        }

        /// <summary>
        /// 色編集コントロールの有効状態を作る
        /// </summary>
        private void UpdateColorControlEnabled()
        {
            this.spcRgb.BeginSuppressEvent();
            this.spaAlpha.BeginSuppressEvent();
            this.blockingFocus = true;

            {
                // RGB編集コントロール
                this.hcpPanel.Enabled     = this.RgbEditEnabled;
                this.pnlRgb.Enabled       = this.RgbEditEnabled;
                this.cbxColorMode.Enabled = this.RgbEditEnabled;
                this.spcRgb.Enabled       = this.RgbEditEnabled;

                // アルファ編集コントロール
                this.pnlAlpha.Enabled   = this.AlphaEditEnabled;
                this.spaAlpha.Enabled   = this.AlphaEditEnabled;
            }

            this.blockingFocus = false;
            this.spcRgb.EndSuppressEvent();
            this.spaAlpha.EndSuppressEvent();
        }

        /// <summary>
        /// The focus next.
        /// </summary>
        /// <param name="isForward">
        /// The is forward.
        /// </param>
        private void FocusNext(bool isForward)
        {
            // 末端のキーならページ全体のタブオーダーへ移る
            Control test = this.Parent;
            while (test != null)
            {
                var tabPage = test as PropertyTabPageBase;
                if (tabPage != null)
                {
                    tabPage.SelectNextChildControl(isForward);
                    break;
                }

                test = test.Parent;
            }
        }

        /// <summary>
        /// コンテキストメニュー初期化
        /// </summary>
        private void InitializeContextMenu()
        {
            var contextMenuStrip = ContextMenuUtility.CreateCopyPaste(
                () =>
                {
                    Clipboard.SetData(
                        ColorClipboardData.ClipboardFormat,
                        new ColorClipboardData
                        {
                            R = this.Value.R,
                            G = this.Value.G,
                            B = this.Value.B,
                            A = this.Value.A,
                            EnabledRgb = this.RgbEditEnabled,
                            EnabledAlpha = this.AlphaEditEnabled
                        });
                },
                () =>
                {
                    if (Clipboard.ContainsData(ColorClipboardData.ClipboardFormat))
                    {
                        var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                        var oldValue = new ColorRgba(this.Value);

                        this.Value = new ColorRgba(
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.R : 1.0f,
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.G : 1.0f,
                            (this.RgbEditEnabled && data.EnabledRgb) ? data.B : 1.0f,
                            (this.AlphaEditEnabled && data.EnabledAlpha) ? data.A : 1.0f);

                        this.InvokeValueChangedEvent(new SequentialValueChangedEventArgs(false));

                        this.TriggerValueChangedExecutable(
                            new ValueChangedExEventArgs(
                                oldValue,
                                this.Value.Clone(),
                                false,
                                this.ValueChangedExecutableParameter));
                    }
                });

            contextMenuStrip.Opening += (s, e) =>
            {
                var enabledPaste = Clipboard.ContainsData(ColorClipboardData.ClipboardFormat);

                if (enabledPaste)
                {
                    var data = Clipboard.GetData(ColorClipboardData.ClipboardFormat) as ColorClipboardData;

                    enabledPaste = (data.EnabledRgb == this.RgbEditEnabled) &&
                                   (data.EnabledAlpha == this.AlphaEditEnabled);
                }

                contextMenuStrip.Items[ContextMenuUtility.CopyPasteMenuPasteIndex].Enabled = enabledPaste;
            };

            this.pnlRgb.ContextMenuStrip = contextMenuStrip;
            this.pnlAlpha.ContextMenuStrip = contextMenuStrip;
        }
    }
}
