﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Specifics.Sliders;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Control class for the color slider panel.
    /// </summary>
    [DefaultEvent("ValueChanged")]
    public partial class SliderPanelColor : UserControl
    {
        /// <summary>
        /// カラープレビューの矩形幅。
        /// </summary>
        private const int ColorPreviewRectSize = 36;

        /// <summary>
        /// コンテキストメニュー。
        /// </summary>
        private UIContextMenuStrip contextMenu = new UIContextMenuStrip();

        /// <summary>
        /// イベント抑制深さ
        /// </summary>
        private int suppressEventDepth;

#if false // TODO:コード復帰
        /// <summary>
        /// cmiCopy.
        /// </summary>
        private UIToolStripMenuItem cmiCopy = null;

        /// <summary>
        /// cmiPaste.
        /// </summary>
        private UIToolStripMenuItem cmiPaste = null;
#endif

        /// <summary>
        /// カラーモード.
        /// </summary>
        private ColorModeType colorMode = ColorModeType.RGB;

        /// <summary>
        /// ColorSliderControllerRGB.
        /// </summary>
        private ColorSliderControllerRGB controllerRGB = new ColorSliderControllerRGB();

        /// <summary>
        /// ColorSliderControllerHSV.
        /// </summary>
        private ColorSliderControllerHSV controllerHSV = new ColorSliderControllerHSV();

        /// <summary>
        /// isMouseOverSwatch.
        /// </summary>
        private bool isMouseOverSwatch = false;

        /// <summary>
        /// 最後に発生したValueChangedイベントの情報。
        /// </summary>
        private SequentialValueChangedEventArgs lastValueChangedEventArgs = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public SliderPanelColor()
        {
            this.DoubleBuffered = true;

            this.InitializeComponent();

            this.CreateContextMenu();
#if false // TODO:コード復帰
            TheApp.OnGammaCorrectionModeChanged += this.OnGammaCorrectionModeChanged;
#endif
            this.controllerRGB.ValueChanged += this.OnControllerValueChanged;
            this.controllerHSV.ValueChanged += this.OnControllerValueChanged;
            this.textElement1.ValueChanged += this.OnDialSliderValueChanged;
            this.textElement2.ValueChanged += this.OnDialSliderValueChanged;
            this.textElement3.ValueChanged += this.OnDialSliderValueChanged;

            this.controllerRGB.SetColorElementSliders(
                                 this.sliderElement1,
                                 this.sliderElement2,
                                 this.sliderElement3);
        }

        /// <summary>
        /// Event triggered when the hue value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the color edit mode. (allows RGB or HSV only)
        /// </summary>
        public ColorModeType ColorMode
        {
            get
            {
                return this.colorMode;
            }

            set
            {
                if (value == this.colorMode)
                {
                    return;
                }

                ColorRgba oldValue = this.Value;

                if (this.colorMode == ColorModeType.RGB || this.colorMode == ColorModeType.sRGB)
                {
                    this.controllerRGB.ClearColorElementSliders();
                }
                else if (this.colorMode == ColorModeType.HSV)
                {
                    this.controllerHSV.ClearColorElementSliders();
                }

                this.colorMode = value;

                if (this.colorMode == ColorModeType.RGB || this.colorMode == ColorModeType.sRGB)
                {
                    this.lblElement1.Text = "R";
                    this.lblElement2.Text = "G";
                    this.lblElement3.Text = "B";

                    this.controllerRGB.SetColorElementSliders(
                         this.sliderElement1,
                         this.sliderElement2,
                         this.sliderElement3);

                    this.controllerRGB.GammaEnabled = (this.colorMode == ColorModeType.RGB);

                    this.Value = oldValue;
                }
                else if (this.colorMode == ColorModeType.HSV)
                {
                    this.lblElement1.Text = "H";
                    this.lblElement2.Text = "S";
                    this.lblElement3.Text = "V";

                    this.controllerHSV.SetColorElementSliders(
                                         this.sliderElement1,
                                         this.sliderElement2,
                                         this.sliderElement3);

                    this.Value = oldValue;
                }

                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the color value.
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                if (this.ColorMode == ColorModeType.RGB || this.ColorMode == ColorModeType.sRGB)
                {
                    return this.controllerRGB.Value;
                }
                else if (this.ColorMode == ColorModeType.HSV)
                {
                    return this.controllerHSV.Value;
                }
                else
                {
                    return new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);
                }
            }

            set
            {
#if false
                if (this.Enabled == false)
                {
                    return;
                }
#endif

                this.BeginSuppressEvent();

                if (this.ColorMode == ColorModeType.RGB || this.ColorMode == ColorModeType.sRGB)
                {
                    this.controllerRGB.Value = value;
                }
                else if (this.ColorMode == ColorModeType.HSV)
                {
                    this.controllerHSV.Value = value;
                }

                this.textElement1.Value = this.sliderElement1.Value;
                this.textElement2.Value = this.sliderElement2.Value;
                this.textElement3.Value = this.sliderElement3.Value;

                this.Invalidate();

                this.EndSuppressEvent();
            }
        }

        /// <summary>
        /// イベントを抑制するかどうか。
        /// </summary>
        private bool IsSuppressEvent
        {
            get { return this.suppressEventDepth > 0; }
        }

        /// <summary>
        /// イベント抑制開始
        /// </summary>
        public void BeginSuppressEvent()
        {
            ++this.suppressEventDepth;
        }

        /// <summary>
        /// イベント抑制終了
        /// </summary>
        public void EndSuppressEvent()
        {
            --this.suppressEventDepth;
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            var rect = new Rectangle(
                0,
                (this.DisplayRectangle.Height - ColorPreviewRectSize) / 2,
                ColorPreviewRectSize,
                ColorPreviewRectSize);

            // Check if the mouse cursor is over the color swatch.
            this.isMouseOverSwatch = rect.Contains(e.Location);

            base.OnMouseMove(e);
        }

        /// <summary>
        /// スライダーテキストが変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnDialSliderValueChanged(object sender, EventArgs e)
        {
            if (this.IsSuppressEvent)
            {
                return;
            }

            this.BeginSuppressEvent();

            this.sliderElement1.Value = this.textElement1.Value;
            this.sliderElement2.Value = this.textElement2.Value;
            this.sliderElement3.Value = this.textElement3.Value;

            this.Invalidate();

            if (this.ValueChanged != null)
            {
                this.lastValueChangedEventArgs = new SequentialValueChangedEventArgs(false);
                this.ValueChanged(this, this.lastValueChangedEventArgs);
            }

            this.EndSuppressEvent();
        }

        /// <summary>
        /// Handle ValueChanged event from the controller.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnControllerValueChanged(
                     object sender,
                     SequentialValueChangedEventArgs e)
        {
            this.lastValueChangedEventArgs = e;

            if (this.IsSuppressEvent)
            {
                return;
            }

            this.BeginSuppressEvent();

            if (sender is FloatUpDown)
            {
                this.sliderElement1.Value = this.textElement1.Value;
                this.sliderElement2.Value = this.textElement2.Value;
                this.sliderElement3.Value = this.textElement3.Value;
            }
            else
            {
                this.textElement1.Value = this.sliderElement1.Value;
                this.textElement2.Value = this.sliderElement2.Value;
                this.textElement3.Value = this.sliderElement3.Value;
            }

            this.Invalidate();

            if (this.ValueChanged != null)
            {
                // OnControllerValueChangedメソッドは処理中に同じメソッドがコールされることがある(スライダーを高速クリックしたときなど)。
                // そのときe.Changing=trueのイベントを優先して送りたいので、最後に発生したイベント情報を送るようにする。
                this.ValueChanged(this, e);
            }

            this.EndSuppressEvent();
        }

        /// <summary>
        /// Handle OnGammaCorrectionModeChanged event from the application.
        /// </summary>
        private void OnGammaCorrectionModeChanged()
        {
            this.Invalidate();
        }

        /// <summary>
        /// Handle Opening event from the color swatch context menu.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnContextMenuOpening(object sender, CancelEventArgs e)
        {
#if false // TODO:コード復帰
            e.Cancel = !(this.m_bMouseOverSwatch);

            bool isPasteEnabled = true;

            var data =
                Clipboard.GetData(UAColorValueEdit.CLIPBOARD_DATA_FORMAT) as UAColorValueEdit.ColorEditClipboardData;
            if (data == null)
            {
                isPasteEnabled = false;
            }
            else if (data.HasColorR == false ||
                     data.HasColorG == false ||
                     data.HasColorB == false)
            {
                isPasteEnabled = false;
            }

            this.m_cmiPaste.Enabled = isPasteEnabled;
#endif
        }

        /// <summary>
        /// Helper method for creating the context menu for the color swatch.
        /// </summary>
        private void CreateContextMenu()
        {
#if false // TODO:コード復帰
            // "Copy" menu item.
            this.m_cmiCopy = new UIToolStripMenuItem()
            {
                Text = res.Strings.COPY,
                Image = Properties.Resources.TB_CopyHS
            };

            // Copy handler.
            this.m_cmiCopy.Click += (s, e) =>
            {
                ColorRgba color = this.Value;

                var data = new UAColorValueEdit.ColorEditClipboardData();

                data.HasColorR = true;
                data.HasColorG = true;
                data.HasColorB = true;
                data.HasColorA = false;
                data.ColorR = color.R;
                data.ColorG = color.G;
                data.ColorB = color.B;
                data.ColorA = 1.0f;

                Clipboard.SetData(UAColorValueEdit.CLIPBOARD_DATA_FORMAT, data);
            };

            this.m_contextMenu.Items.Add(this.m_cmiCopy);

            // "Paste" menu item.
            this.m_cmiPaste = new UIToolStripMenuItem()
            {
                Text = res.Strings.PASTE,
                Image = Properties.Resources.TB_PasteHS
            };

            // Paste handler.
            this.m_cmiPaste.Click += (s, e) =>
            {
                var data =
                    Clipboard.GetData(UAColorValueEdit.CLIPBOARD_DATA_FORMAT) as UAColorValueEdit.ColorEditClipboardData;
                if (data == null)
                {
                    return;
                }

                if (data.HasColorR == false ||
                    data.HasColorG == false ||
                    data.HasColorB == false)
                {
                    return;
                }

                this.Value = new ColorRgba(data.ColorR, data.ColorG, data.ColorB, 1.0f);

                if (this.ValueChanged != null)
                {
                    this.ValueChanged(this, new SequentialValueChangedEventArgs(false));
                }
            };

            this.m_contextMenu.Items.Add(this.m_cmiPaste);

            this.m_contextMenu.Opening += this.OnContextMenuOpening;

            this.ContextMenuStrip = this.m_contextMenu;
#endif
        }
    }
}
