﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Specifics.ColorPicker;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Class for RGB color element slider controller.
    /// </summary>
    public class ColorSliderControllerRGB
    {
        /// <summary>
        /// Ｒスライダ。
        /// </summary>
        private ColorElementSlider sliderR = null;

        /// <summary>
        /// Ｇスライダ。
        /// </summary>
        private ColorElementSlider sliderG = null;

        /// <summary>
        /// Ｂスライダ。
        /// </summary>
        private ColorElementSlider sliderB = null;

        /// <summary>
        /// GammaEnabled プロパティのバッキングフィールドです。
        /// </summary>
        private bool gammaEnabled = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ColorSliderControllerRGB()
        {
        }

        /// <summary>
        /// Event triggered when the color value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// Get or set the color value.
        /// </summary>
        public ColorRgba Value
        {
            get
            {
                if (this.sliderR == null || this.sliderG == null || this.sliderB == null)
                {
                    return new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);
                }

                return new ColorRgba(
                           this.sliderR.Value,
                           this.sliderG.Value,
                           this.sliderB.Value,
                           1.0f);
            }

            set
            {
                if (this.sliderR == null || this.sliderG == null || this.sliderB == null)
                {
                    return;
                }

                this.sliderR.Value = value.R;
                this.sliderG.Value = value.G;
                this.sliderB.Value = value.B;

                this.UpdateSliderRImage();
                this.UpdateSliderGImage();
                this.UpdateSliderBImage();
            }
        }

        /// <summary>
        /// ガンマ補正を有効にするかどうか取得または設定します。
        /// </summary>
        public bool GammaEnabled
        {
            get
            {
                return this.gammaEnabled;
            }

            set
            {
                if (value != this.gammaEnabled)
                {
                    this.gammaEnabled = value;

                    if (this.sliderR != null && this.sliderG != null && this.sliderB != null)
                    {
                        this.sliderR.GammaEnabled = this.gammaEnabled;
                        this.sliderG.GammaEnabled = this.gammaEnabled;
                        this.sliderB.GammaEnabled = this.gammaEnabled;
                    }
                }
            }
        }

        /// <summary>
        /// Set color element sliders.
        /// </summary>
        /// <param name="sliderR">Red color slider.</param>
        /// <param name="sliderG">Green color slider.</param>
        /// <param name="sliderB">Blue color slider.</param>
        public void SetColorElementSliders(
                    ColorElementSlider sliderR,
                    ColorElementSlider sliderG,
                    ColorElementSlider sliderB)
        {
            this.sliderR = sliderR;
            this.sliderG = sliderG;
            this.sliderB = sliderB;

            this.sliderR.GammaEnabled = this.GammaEnabled;
            this.sliderG.GammaEnabled = this.GammaEnabled;
            this.sliderB.GammaEnabled = this.GammaEnabled;

            this.sliderR.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;
            this.sliderG.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;
            this.sliderB.RequestUpdateBitmap += this.OnSliderRequestUpdateBitmap;

            this.sliderR.ValueChanged += this.OnSliderValueChanged;
            this.sliderG.ValueChanged += this.OnSliderValueChanged;
            this.sliderB.ValueChanged += this.OnSliderValueChanged;

            this.UpdateSliderRImage();
            this.UpdateSliderGImage();
            this.UpdateSliderBImage();
        }

        /// <summary>
        /// Clear the controlling color element sliders.
        /// </summary>
        public void ClearColorElementSliders()
        {
            if (this.sliderR != null)
            {
                this.sliderR.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderR.ValueChanged        -= this.OnSliderValueChanged;

                this.sliderR = null;
            }

            if (this.sliderG != null)
            {
                this.sliderG.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderG.ValueChanged        -= this.OnSliderValueChanged;

                this.sliderG = null;
            }

            if (this.sliderB != null)
            {
                this.sliderB.RequestUpdateBitmap -= this.OnSliderRequestUpdateBitmap;
                this.sliderB.ValueChanged        -= this.OnSliderValueChanged;

                this.sliderB = null;
            }
        }

        /// <summary>
        /// Update the red color slider background image.
        /// </summary>
        private void UpdateSliderRImage()
        {
            if (this.sliderR == null || this.sliderG == null || this.sliderB == null)
            {
                return;
            }

            this.UpdateSliderImage(
                 this.sliderR,
                 new ColorRgba(0.0f, this.sliderG.Value, this.sliderB.Value, 1.0f),
                 new ColorRgba(1.0f, this.sliderG.Value, this.sliderB.Value, 1.0f));
        }

        /// <summary>
        /// Update the green color slider background image.
        /// </summary>
        private void UpdateSliderGImage()
        {
            if (this.sliderR == null || this.sliderG == null || this.sliderB == null)
            {
                return;
            }

            this.UpdateSliderImage(
                 this.sliderG,
                 new ColorRgba(this.sliderR.Value, 0.0f, this.sliderB.Value, 1.0f),
                 new ColorRgba(this.sliderR.Value, 1.0f, this.sliderB.Value, 1.0f));
        }

        /// <summary>
        /// Update the blue color slider background image.
        /// </summary>
        private void UpdateSliderBImage()
        {
            if (this.sliderR == null || this.sliderG == null || this.sliderB == null)
            {
                return;
            }

            this.UpdateSliderImage(
                 this.sliderB,
                 new ColorRgba(this.sliderR.Value, this.sliderG.Value, 0.0f, 1.0f),
                 new ColorRgba(this.sliderR.Value, this.sliderG.Value, 1.0f, 1.0f));
        }

        /// <summary>
        /// Update a single color element slider background image.
        /// </summary>
        /// <param name="target">The slider to update.</param>
        /// <param name="color0">The start color of the gradation.</param>
        /// <param name="color1">The end color of the gradation.</param>
        private void UpdateSliderImage(
                     ColorElementSlider target,
                     ColorRgba color0,
                     ColorRgba color1)
        {
            if (target == null)
            {
                return;
            }

            Bitmap img = target.ColorBarBitmap;
            if (img == null)
            {
                return;
            }

            if (img.Width <= 0)
            {
                return;
            }

            using (var g = Graphics.FromImage(img))
            using (var brush = new LinearGradientBrush(
                                                   g.VisibleClipBounds,
                                                   ColorUtility.ToWinColor(color0),
                                                   ColorUtility.ToWinColor(color1),
                                                   0.0f,
                                                   false))
            {
                g.FillRectangle(brush, g.VisibleClipBounds);
            }

            target.Invalidate();
        }

        /// <summary>
        /// Handle RequestUpdateBitmap event for the sliders.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSliderRequestUpdateBitmap(
                     object sender,
                     EventArgs e)
        {
            if (sender == null)
            {
                return;
            }
            else if (sender == this.sliderR)
            {
                this.UpdateSliderRImage();
            }
            else if (sender == this.sliderG)
            {
                this.UpdateSliderGImage();
            }
            else if (sender == this.sliderB)
            {
                this.UpdateSliderBImage();
            }
        }

        /// <summary>
        /// Handle ValueChanged event for the sliders.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSliderValueChanged(
                     object sender,
                     SequentialValueChangedEventArgs e)
        {
            if (sender == null)
            {
                return;
            }

            if (sender == this.sliderR)
            {
                this.UpdateSliderGImage();
                this.UpdateSliderBImage();
            }
            else if (sender == this.sliderG)
            {
                this.UpdateSliderRImage();
                this.UpdateSliderBImage();
            }
            else if (sender == this.sliderB)
            {
                this.UpdateSliderRImage();
                this.UpdateSliderGImage();
            }
            else
            {
                return;
            }

            // Fire the event.
            if (this.ValueChanged != null)
            {
                this.ValueChanged(this, e);
            }
        }
    }
}
