﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// Behavior that allow the TreeNode on which it is attach to
    /// reorder its children using mouse drag and drop.
    /// </summary>
    public class TreeNodeReorderingBehavior : Behavior<UITreeView>
    {
        /// <summary>
        /// Stores the TreeView instance of the TreeNode the behavior is attached to.
        /// </summary>
        private UITreeView treeView;

        /// <summary>
        /// Stores the TreeNode instance where the mouse left button has been pressed.
        /// </summary>
        private UITreeNode nodeAtMouseDown;

        /// <summary>
        /// 現在カーソルが当たっているドロップ先候補のノードです。
        /// </summary>
        private UITreeNode nodeAtDropTarget;

        /// <summary>
        /// Called when attached to a TreeNode.
        /// </summary>
        protected override void OnAttached()
        {
            this.treeView = this.AssociatedObject;

            if (this.treeView == null)
            {
                return;
            }

            this.treeView.AllowDrop = true;

            this.treeView.BeforeDoDragDrop += this.OnTreeViewBeforeDoDragDrop;
            this.treeView.DragEnter += this.OnTreeViewDragEnter;
            this.treeView.DragOver += this.OnTreeViewDragOver;
            this.treeView.AfterRenderNode += this.OnTreeViewAfterRenderNode;
        }

        /// <summary>
        /// Called before being detached to the TreeNode.
        /// </summary>
        protected override void OnDetaching()
        {
            this.treeView.QueryContinueDrag -= this.OnTreeViewQueryContinueDrag;

            this.treeView.AfterRenderNode -= this.OnTreeViewAfterRenderNode;
            this.treeView.DragOver -= this.OnTreeViewDragOver;
            this.treeView.DragEnter -= this.OnTreeViewDragEnter;
            this.treeView.BeforeDoDragDrop -= this.OnTreeViewBeforeDoDragDrop;
        }

        /// <summary>
        /// Called before DoDragDrop is called on the UITreeView.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnTreeViewBeforeDoDragDrop(object sender, UITreeView.BeforeDragEventArgs e)
        {
            this.nodeAtMouseDown = (UITreeNode)this.treeView.DraggingNode;
            this.nodeAtDropTarget = null;
            this.treeView.Invalidate();

            e.AllowedEffects = DragDropEffects.Move;

            this.treeView.QueryContinueDrag += this.OnTreeViewQueryContinueDrag;
        }

        /// <summary>
        /// Called when the mouse start to drag a node.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Custom event argument.</param>
        private void OnTreeViewDragEnter(object sender, DragEventArgs e)
        {
            e.Effect = DragDropEffects.Move;
        }

        /// <summary>
        /// ドラッグ中のマウスカーソルを更新する
        /// </summary>
        /// <param name="sender">ツリービュー</param>
        /// <param name="e">イベント引数</param>
        private void OnTreeViewDragOver(object sender, DragEventArgs e)
        {
            // ファイルのドラッグアンドドロップに対応する
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy | DragDropEffects.Move;
                return;
            }

            var location = this.treeView.PointToClient(Control.MousePosition);
            var currentNode = this.GetHalfSlideHitNode(location);

            e.Effect = this.GetAdjustedDropTarget(currentNode) == null ? DragDropEffects.None : DragDropEffects.Move;
        }

        /// <summary>
        /// ドロップ先候補となる位置を示すボーダーラインを描画します。
        /// </summary>
        /// <param name="sender">ツリービュー</param>
        /// <param name="e">イベント引数</param>
        private void OnTreeViewAfterRenderNode(object sender, DrawTreeNodeEventArgs e)
        {
            if (this.GetAdjustedDropTarget(this.nodeAtDropTarget) == null)
            {
                // 有効な候補が選択されていないなら描かない
                return;
            }

            // ノードの上下関係を判定する
            var nodes = this.nodeAtDropTarget.Parent.Children;
            int targetIndex = nodes.FindIndex(this.nodeAtDropTarget);
            int dragIndex = nodes.FindIndex(this.nodeAtMouseDown);

            int lineY = 0;
            TreeNode drawTarget = null;
            if (dragIndex > targetIndex)
            {
                // ドラッグして上側に引っ張ってる時は上側に線を描く
                lineY = e.Bounds.Y - 2;
                drawTarget = this.nodeAtDropTarget;
            }
            else if (dragIndex < targetIndex)
            {
                // ドラッグして下側に引っ張ってる時は、展開状態に応じてちょいと工夫がいる
                if (this.nodeAtDropTarget.IsExpanded && this.nodeAtDropTarget.Nodes.Count > 0)
                {
                    // 展開されていたらそのノードの末尾のノードを取得してその下側に線を描く
                    lineY = e.Bounds.Y + e.Bounds.Height - 2;
                    drawTarget = this.nodeAtDropTarget.LastNode;
                }
                else
                {
                    // 閉じてる場合は普通に下側に描く
                    lineY = e.Bounds.Y + e.Bounds.Height - 2;
                    drawTarget = this.nodeAtDropTarget;
                }
            }

            if (e.Node != drawTarget)
            {
                // ドロップ先候補以外のノードには何もしない
                return;
            }

            try
            {
                e.Graphics.DrawRectangle(Pens.Black, e.Bounds.X, lineY, e.Bounds.Width, 1);
            }
            catch
            {
                // 領域外侵犯の判定がセンシティブなので、とりあえず握りつぶし
            }
        }

        /// <summary>
        /// Called continuously when a node is being dragged.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Custom event argument.</param>
        private void OnTreeViewQueryContinueDrag(object sender, QueryContinueDragEventArgs e)
        {
            if (e.EscapePressed || (Control.MouseButtons & MouseButtons.Right) == MouseButtons.Right)
            {
                // ESCキーと右クリックによるドラッグ中断
                this.nodeAtMouseDown = null;
                this.nodeAtDropTarget = null;
                this.treeView.Invalidate();
                this.treeView.QueryContinueDrag -= this.OnTreeViewQueryContinueDrag;
                return;
            }

            if ((Control.MouseButtons & MouseButtons.Left) == MouseButtons.Left)
            {
                // ドラッグ継続処理
                var location = this.treeView.PointToClient(Control.MousePosition);
                this.OnTreeViewDraggingMouseMove(location);
            }
            else if ((Control.MouseButtons & MouseButtons.Left) != MouseButtons.Left)
            {
                // ドラッグ終了処理
                this.OnTreeViewDraggingMouseUp();
            }
        }

        /// <summary>
        /// Called the mouse moves over the TreeView while a node is dragged.
        /// </summary>
        /// <param name="location">Mouse position, in TreeView space.</param>
        private void OnTreeViewDraggingMouseMove(Point location)
        {
            TreeNode currentNode = this.GetHalfSlideHitNode(location);
            if (currentNode == this.nodeAtMouseDown)
            {
                // ドラッグ開始ノードだったら初期状態に復帰する
                this.nodeAtDropTarget = null;
                this.treeView.Invalidate();
                return;
            }

            TreeNode currentConsideringGroup = this.GetAdjustedDropTarget(currentNode);
            if (currentConsideringGroup == null)
            {
                // ドロップ対象ノード以外だとしてもステータスを変化させない
                return;
            }

            this.nodeAtDropTarget = currentConsideringGroup as UITreeNode;
            this.treeView.Invalidate();
        }

        /// <summary>
        /// Called when the left mouse button is released.
        /// </summary>
        private void OnTreeViewDraggingMouseUp()
        {
            this.treeView.QueryContinueDrag -= this.OnTreeViewQueryContinueDrag;

            this.ExecuteNodeSwap();

            this.nodeAtMouseDown = null;
            this.nodeAtDropTarget = null;
            this.treeView.Invalidate();
        }

        /// <summary>
        /// Execute the SwapNodesExecutable with appropriate parameter.
        /// </summary>
        private void ExecuteNodeSwap()
        {
            if (this.nodeAtMouseDown == null || this.nodeAtDropTarget == null)
            {
                return;
            }

            var parentNode = this.nodeAtMouseDown.Parent as UITreeNode;
            if (parentNode == null)
            {
                return;
            }

            var swap = parentNode.SwapNodesExecutable;
            if (swap != null)
            {
                var info = Tuple.Create(this.nodeAtMouseDown.Index, this.nodeAtDropTarget.Index);
                if (swap.CanExecute(info))
                {
                    swap.Execute(info);
                }
            }
        }

        /// <summary>
        /// ノードの高さの半分ぶんずらした位置のノードを取得します。
        /// </summary>
        /// <param name="location">マウス座標</param>
        /// <returns>ヒットしたノード</returns>
        private TreeNode GetHalfSlideHitNode(Point location)
        {
            var currentNode = this.treeView.GetNodeAt(location);
            if (currentNode == null ||
                currentNode == this.nodeAtMouseDown)
            {
                return currentNode;
            }

            // 上下どちらに引っ張ってるかは全ノードを平たくしたリスト内におけるインデックスで判定する
            var allNodes = this.treeView.GetAllNodes().ToList();
            if (allNodes.IndexOf((UITreeNode)currentNode) < allNodes.IndexOf(this.nodeAtMouseDown))
            {
                currentNode = this.treeView.GetNodeAt(location.X, location.Y + currentNode.Bounds.Height / 2);
            }
            else
            {
                currentNode = this.treeView.GetNodeAt(location.X, location.Y - currentNode.Bounds.Height / 2);
            }

            return currentNode;
        }

        /// <summary>
        /// ドロップ先として有効なノードか検証し、その結果をノードとして返します。
        /// 候補ノードの親まで辿り、同種の兄弟ノードが見つかったらそれを候補として返します。
        /// </summary>
        /// <param name="currentNode">マウスでポイントしているノード</param>
        /// <returns>
        /// ・ポイントしているノードが受け入れ可能ならそのまま返す。
        /// ・親に受け入れ可能なノードがいるならそれを返す。
        /// ・どうサーチしても受け入れ可能なノードが見つからなければnullを返す。
        /// </returns>
        private TreeNode GetAdjustedDropTarget(TreeNode currentNode)
        {
            if (currentNode == null || this.nodeAtMouseDown == null)
            {
                return null;
            }

            var current = currentNode;
            while (current != null)
            {
                if (current.Parent == this.nodeAtMouseDown.Parent &&
                    current.GetType() == this.nodeAtMouseDown.GetType())
                {
                    return current;
                }

                current = current.Parent;
            }

            return null;
        }
    }
}
