﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.ClipboardDataTypes;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// カラーコントロール系コンテキストメニュー振る舞い
    /// </summary>
    public class ColorControlContextMenuBehavior : Behavior<UIPanel>, INotifyPropertyChanged
    {
        /// <summary>
        /// タグ
        /// </summary>
        private string tag = string.Empty;

        /// <summary>
        /// ターゲット
        /// </summary>
        private ColorTargetType target = (ColorTargetType)(-1);

        /// <summary>
        /// Raised when the value of a property changes.
        /// </summary>
        public new event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// タグ
        /// </summary>
        public string Tag
        {
            get
            {
                return this.tag;
            }

            set
            {
                this.tag = value;

                var handler = this.PropertyChanged;
                if (handler != null)
                {
                    handler(this, new PropertyChangedEventArgs("Tag"));
                }
            }
        }

        /// <summary>
        /// ターゲット
        /// </summary>
        public ColorTargetType Target
        {
            get
            {
                return this.target;
            }

            set
            {
                this.target = value;

                var handler = this.PropertyChanged;
                if (handler != null)
                {
                    handler(this, new PropertyChangedEventArgs("Target"));
                }
            }
        }

        /// <summary>
        /// アタッチするときの処理を行います。
        /// </summary>
        protected override void OnAttached()
        {
            this.AssociatedObject.ContextMenuStrip = this.CreateContextMenu();
        }

        /// <summary>
        /// デタッチするときの処理を行います。
        /// </summary>
        protected override void OnDetaching()
        {
            // 何もしない
        }

        /// <summary>
        /// コンテキストメニュー生成
        /// </summary>
        /// <returns>コンテキストメニュー</returns>
        private ContextMenuStrip CreateContextMenu()
        {
            var menu = new UIContextMenuStrip();
            {
                menu.SetLogicalParent(this.AssociatedObject);

                // グループ内容をコピー
                var itemCopyGroup = new UIToolStripMenuItem
                {
                    Text = Properties.Resources.MenuCopyGroup,
                         Image = Properties.Resources.Icon_Copy
                };
                itemCopyGroup.AddBinding("Executable", "CopyGroupExecutable").UpdateElement();
                menu.Items.Add(itemCopyGroup);

                // グループ内容を貼り付け
                var itemPasteGroup = new UIToolStripMenuItem
                {
                    Text = Properties.Resources.MenuPasteGroup,
                         Image = Properties.Resources.Icon_Paste
                };
                itemPasteGroup.AddBinding("Executable", "PasteGroupExecutable").UpdateElement();
                menu.Items.Add(itemPasteGroup);

                // グループ内容をデフォルトに戻す
                var itemDefaultGroup = new UIToolStripMenuItem
                {
                    Text = Properties.Resources.MenuResetGroupDefaultValues,
                };
                itemDefaultGroup.AddBinding("Executable", "ResetDefaultValueExecutable").UpdateElement();
                menu.Items.Add(itemDefaultGroup);

                // セパレータ
                menu.Items.Add(new ToolStripSeparator());

                // コピー
                var itemCopyColor = new UIToolStripMenuItem
                {
                    Text = string.Format(Properties.Resources.MenuCopy, this.Tag),
                         Image = Properties.Resources.Icon_CopyColor
                };
                itemCopyColor.ExecutionParameter = this.Target;
                itemCopyColor.AddBinding("Executable", "CopyColorExecutable").UpdateElement();
                menu.Items.Add(itemCopyColor);

                // 貼り付け
                var itemPasteColor = new UIToolStripMenuItem
                {
                    Text = string.Format(Properties.Resources.MenuPaste, this.Tag),
                         Image = Properties.Resources.Icon_PasteColor
                };
                itemPasteColor.ExecutionParameter = this.Target;
                itemPasteColor.AddBinding("Executable", "PasteColorExecutable").UpdateElement();
                menu.Items.Add(itemPasteColor);

                menu.Opening += (s, e) =>
                {
                    // クリップボードデータが無い
                    if (Clipboard.ContainsData(ColorUnitClipboardData.ClipboardFormat) == false)
                    {
                        itemPasteColor.Enabled = false;
                    }
                    else
                    {
                        var copyData = Clipboard.GetData(ColorUnitClipboardData.ClipboardFormat) as ColorUnitClipboardData;

                        itemPasteColor.Enabled = ColorUtility.MakeColorElementType(copyData.Target) ==
                                                 ColorUtility.MakeColorElementType(this.Target);
                    }
                };
            }

            return menu;
        }
    }
}
