﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// UDD2.0ドキュメントをパースするためのユーティリティを提供します。
    /// </summary>
    public static class XmlParseUtil
    {
        /// <summary>
        /// 名前を指定してノードの属性を読み取ります。
        /// </summary>
        /// <typeparam name="T">読み取る値の型</typeparam>
        /// <param name="node">ノード</param>
        /// <param name="attributeName">属性名</param>
        /// <param name="defaultValue">初期値(型の明示)</param>
        /// <returns>読み取った値(読み取れなかった場合は初期値を返す)</returns>
        public static T ReadAttribute<T>(XmlNode node, string attributeName, T defaultValue)
        {
            if (node.Attributes == null) return defaultValue;
            var att = node.Attributes.OfType<XmlAttribute>().FirstOrDefault(a => a.Name == attributeName);
            if (att == null) return defaultValue;

            if (typeof(T) == typeof(string))
            {
                return (T)(object)LocalizeUtility.Translate(att.Value);
            }

            if (typeof(T).IsArray)
            {
                var conv = TypeDescriptor.GetConverter(typeof(T).GetElementType());
                var splitted = att.Value.Split(' ');

                var result = (Array)Activator.CreateInstance(typeof(T), splitted.Length);
                for (int i = 0; i < splitted.Length; ++i)
                {
                    result.SetValue(conv.ConvertFromString(splitted[i]), i);
                }

                return (T)(object)result;
            }
            else
            {
                var conv = TypeDescriptor.GetConverter(typeof(T));
                if (conv.CanConvertFrom(typeof(string)))
                {
                    return (T)conv.ConvertFromString(att.Value);
                }
            }

            return defaultValue;
        }

        /// <summary>
        /// 名前を指定してノードの属性に記述された列挙体の値を読み取ります。
        /// </summary>
        /// <typeparam name="T">読み取る値の型</typeparam>
        /// <param name="node">ノード</param>
        /// <param name="attributeName">属性名</param>
        /// <param name="defaultValue">初期値(型の明示)</param>
        /// <returns>読み取った値(読み取れなかった場合は初期値を返す)</returns>
        public static T ReadEnumAttribute<T>(XmlNode node, string attributeName, T defaultValue) where T : struct
        {
            if (node.Attributes == null) return defaultValue;
            var att = node.Attributes.OfType<XmlAttribute>().FirstOrDefault(a => a.Name == attributeName);
            if (att == null) return defaultValue;

            T result;
            if (Enum.TryParse(att.Value, true, out result))
            {
                return result;
            }

            return defaultValue;
        }

        /// <summary>
        /// StringBuilderの末尾に二重改行が付くのを抑制します。
        /// </summary>
        /// <param name="builder">ビルダーインスタンス</param>
        internal static void ChopDoubleNewLine(StringBuilder builder)
        {
            if (builder.Length <= Environment.NewLine.Length * 2)
            {
                return;
            }

            if (builder.ToString().EndsWith(Environment.NewLine + Environment.NewLine))
            {
                builder.Remove(builder.Length - Environment.NewLine.Length, Environment.NewLine.Length);
            }
        }
    }
}
