﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Serialization;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// The group definition serializer for .udd/.usd files.
    /// </summary>
    [Serializable]
    public class GroupDefinition : UserDefinitionBase
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public GroupDefinition() :
            base()
        {
            this.GroupType = string.Empty;
            this.Template = null;
            this.Controls = new List<UserDefinitionBase>();
        }

        /// <summary>
        /// Get or set the type of the group.
        /// </summary>
        [XmlAttribute("Type")]
        public string GroupType { get; set; }

        /// <summary>
        /// Get or set the template for the child controls.
        /// </summary>
        [XmlElement("Template", typeof(TemplateDefinition))]
        public TemplateDefinition Template { get; set; }

        /// <summary>
        /// Get or set the controls in the group.
        /// </summary>
        [XmlElement("CheckBox", typeof(CheckBoxDefinition))]
        [XmlElement("RadioButton", typeof(RadioButtonDefinition))]
        [XmlElement("IntSlider", typeof(IntSliderDefinition))]
        [XmlElement("FloatSlider", typeof(FloatSliderDefinition))]
        [XmlElement("Slider", typeof(UsdFloatSliderDefinition))]
        [XmlElement("Group", typeof(GroupDefinition))]
        public List<UserDefinitionBase> Controls { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            if (base.ReadXml(context) == false)
            {
                return false;
            }

            this.GroupType = this.ReadAttribute(context, "Type", this.GroupType);
            this.Template = this.ReadElement(context, "Template", this.Template);

            var nameTypeMap = new Dictionary<string, Type>()
            {
                { "CheckBox", typeof(CheckBoxDefinition) },
                { "RadioButton", typeof(RadioButtonDefinition) },
                { "IntSlider", typeof(IntSliderDefinition) },
                { "FloatSlider", typeof(FloatSliderDefinition) },
                { "Slider", typeof(UsdFloatSliderDefinition) },
                { "Group", typeof(GroupDefinition) },
            };

            this.Controls = this.ReadElementsByTagNames<UserDefinitionBase>(
                context,
                nameTypeMap).ToList();

            return true;
        }

        /// <summary>
        /// This method should be called after deserialized from Xml
        /// so that the definition can prepare its data before it can
        /// be used.
        /// </summary>
        public override void PostLoad()
        {
            base.PostLoad();

            if (this.Template != null)
            {
                this.Template.PostLoad();
                foreach (UserDefinitionBase control in this.Controls)
                {
                    control.ApplyTemplate(this.Template);
                }
            }

            foreach (UserDefinitionBase control in this.Controls)
            {
                control.PostLoad();
            }
        }
    }
}
