﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Xml.Serialization;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// The floating point slider definition serializer for .udd files.
    /// </summary>
    [Serializable]
    public class FloatSliderDefinition : UserDefinitionBase
    {
        /// <summary>The default value for FixRange property.</summary>
        private const bool DefaultFixRangeValue = false;

        /// <summary>The default value for FixMinRange property.</summary>
        private const bool DefaultFixMinRangeValue = true;

        /// <summary>The default value for InitMinRange property.</summary>
        private const float DefaultInitMinRangeValue = 0.0f;

        /// <summary>The default value for InitMaxRange property.</summary>
        private const float DefaultInitMaxRangeValue = 1.0f;

        /// <summary>The default value for MinMaxRange property.</summary>
        private const float DefaultMinMaxRangeValue = 0.0001f;

        /// <summary>The default value for MaxMaxRange property.</summary>
        private const float DefaultMaxMaxRangeValue = 1000000.0f;

        /// <summary>The default value for DefaultValue property.</summary>
        private const float DefaultDefaultValueValue = 0.0f;

        /// <summary>The default value for DeltaLevel property.</summary>
        private const int DefaultDeltaLevelValue = 3;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static FloatSliderDefinition()
        {
            Type myType = typeof(FloatSliderDefinition);

            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("FixRange"), DefaultFixRangeValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("FixMinRange"), DefaultFixMinRangeValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("InitMinRange"), DefaultInitMinRangeValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("InitMaxRange"), DefaultInitMaxRangeValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("MinMaxRange"), DefaultMinMaxRangeValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("MaxMaxRange"), DefaultMaxMaxRangeValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("DefaultValue"), DefaultDefaultValueValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("DeltaLevel"), DefaultDeltaLevelValue);
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public FloatSliderDefinition() :
            base()
        {
        }

        /// <summary>
        /// Get or set the flag indicating whether the range is fixed.
        /// </summary>
        [XmlAttribute("FixRange")]
        public bool FixRange
        {
            get { return this.GetSmartPropertyValue(() => this.FixRange); }
            set { this.SetSmartPropertyValue(() => this.FixRange, value); }
        }

        /// <summary>
        /// Get or set the flag indicating whether the minimum range is fixed.
        /// </summary>
        [XmlAttribute("FixMinRange")]
        public bool FixMinRange
        {
            get { return this.GetSmartPropertyValue(() => this.FixMinRange); }
            set { this.SetSmartPropertyValue(() => this.FixMinRange, value); }
        }

        /// <summary>
        /// Get or set the initial minimum range.
        /// </summary>
        [XmlAttribute("InitMinRange")]
        public float InitMinRange
        {
            get { return this.GetSmartPropertyValue(() => this.InitMinRange); }
            set { this.SetSmartPropertyValue(() => this.InitMinRange, value); }
        }

        /// <summary>
        /// Get or set the initial maximum range.
        /// </summary>
        [XmlAttribute("InitMaxRange")]
        public float InitMaxRange
        {
            get { return this.GetSmartPropertyValue(() => this.InitMaxRange); }
            set { this.SetSmartPropertyValue(() => this.InitMaxRange, value); }
        }

        /// <summary>
        /// Get or set the minimum value of the maximum range.
        /// </summary>
        [XmlAttribute("MinMaxRange")]
        public float MinMaxRange
        {
            get { return this.GetSmartPropertyValue(() => this.MinMaxRange); }
            set { this.SetSmartPropertyValue(() => this.MinMaxRange, value); }
        }

        /// <summary>
        /// Get or set the maximum value of the maximum range.
        /// </summary>
        [XmlAttribute("MaxMaxRange")]
        public float MaxMaxRange
        {
            get { return this.GetSmartPropertyValue(() => this.MaxMaxRange); }
            set { this.SetSmartPropertyValue(() => this.MaxMaxRange, value); }
        }

        /// <summary>
        /// Get or set the minimum value.
        /// </summary>
        [XmlIgnore]
        public float MinValue { get; private set; }

        /// <summary>
        /// Get or set the maximum value.
        /// </summary>
        [XmlIgnore]
        public float MaxValue { get; private set; }

        /// <summary>
        /// Get or set the default value.
        /// </summary>
        [XmlAttribute("Default")]
        public float DefaultValue
        {
            get { return this.GetSmartPropertyValue(() => this.DefaultValue); }
            set { this.SetSmartPropertyValue(() => this.DefaultValue, value); }
        }

        /// <summary>
        /// Get or set the default value.
        /// </summary>
        [XmlAttribute("DeltaLevel")]
        public int DeltaLevel
        {
            get { return this.GetSmartPropertyValue(() => this.DeltaLevel); }
            set { this.SetSmartPropertyValue(() => this.DeltaLevel, value); }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            if (base.ReadXml(context) == false)
            {
                return false;
            }

            if (this.HasAttribute(context, "FixRange") == true)
            {
                this.FixRange = this.ReadAttribute(context, "FixRange", DefaultFixRangeValue);
            }

            if (this.HasAttribute(context, "FixMinRange") == true)
            {
                this.FixMinRange = this.ReadAttribute(context, "FixMinRange", DefaultFixMinRangeValue);
            }

            if (this.HasAttribute(context, "InitMinRange") == true)
            {
                this.InitMinRange = this.ReadAttribute(context, "InitMinRange", DefaultInitMinRangeValue);
            }

            if (this.HasAttribute(context, "InitMaxRange") == true)
            {
                this.InitMaxRange = this.ReadAttribute(context, "InitMaxRange", DefaultInitMaxRangeValue);
            }

            if (this.HasAttribute(context, "MinMaxRange") == true)
            {
                this.MinMaxRange = this.ReadAttribute(context, "MinMaxRange", DefaultMinMaxRangeValue);
            }

            if (this.HasAttribute(context, "MaxMaxRange") == true)
            {
                this.MaxMaxRange = this.ReadAttribute(context, "MaxMaxRange", DefaultMaxMaxRangeValue);
            }

            if (this.HasAttribute(context, "Default") == true)
            {
                this.DefaultValue = this.ReadAttribute(context, "Default", DefaultDefaultValueValue);
            }

            if (this.HasAttribute(context, "DeltaLevel") == true)
            {
                this.DeltaLevel = this.ReadAttribute(context, "DeltaLevel", DefaultDeltaLevelValue);
            }

            return true;
        }

        /// <summary>
        /// Apply template to the definition.
        /// </summary>
        /// <param name="template">The template.</param>
        public override void ApplyTemplate(TemplateDefinition template)
        {
            base.ApplyTemplate(template);

            template.ApplyTemplateValue(this, "FixRange");
            template.ApplyTemplateValue(this, "FixMinRange");
            template.ApplyTemplateValue(this, "InitMinRange");
            template.ApplyTemplateValue(this, "InitMaxRange");
            template.ApplyTemplateValue(this, "MinMaxRange");
            template.ApplyTemplateValue(this, "MaxMaxRange");
            template.ApplyTemplateValue(this, "DefaultValue");
            template.ApplyTemplateValue(this, "DeltaLevel");
        }

        /// <summary>
        /// This method should be called after deserialized from Xml
        /// so that the definition can prepare its data before it can
        /// be used.
        /// </summary>
        public override void PostLoad()
        {
            if (this.FixRange == true)
            {
                this.MinValue = this.InitMinRange;
                this.MaxValue = this.InitMaxRange;
            }
            else
            {
                if (this.FixMinRange == true)
                {
                    this.MinValue = this.InitMinRange;
                }
                else
                {
                    this.MinValue = -1.0f * this.MaxMaxRange;
                }

                this.MaxValue = this.MaxMaxRange;
            }
        }
    }

    /// <summary>
    /// The floating point slider definition serializer for .usd files.
    /// (the floating point slider tags are called 'Slider' instead of 'FloatSlider')
    /// </summary>
    [Serializable]
    public class UsdFloatSliderDefinition : FloatSliderDefinition
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public UsdFloatSliderDefinition() :
            base()
        {
        }
    }
}
