﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.DataModelMaker.Core.Template;
using ObsoleteUserDataConverter.Properties;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// Writes custom shader UI definition Xml file.
    /// </summary>
    public class CustomShaderUIDefinitionWriter
    {
        /// <summary>Constant template tag name "Label"</summary>
        private const string TemplateTagNameLabel = "Label";

        /// <summary>Constant template tag name "LabelEn"</summary>
        private const string TemplateTagNameLabelEn = "LabelEn";

        /// <summary>Constant template tag name "VertexShaderPaths"</summary>
        private const string TemplateTagNameVertexShaderPaths = "VertexShaderPaths";

        /// <summary>Constant template tag name "FragmentShaderPaths"</summary>
        private const string TemplateTagNameFragmentShaderPaths = "FragmentShaderPaths";

        /// <summary>Constant template tag name "DataModelName"</summary>
        private const string TemplateTagNameDataModelName = "DataModelName";

        /// <summary>Constant template tag name "ShaderPath"</summary>
        private const string TemplateTagNameShaderPath = "ShaderPath";

        /// <summary>The template helper that parses and stores the template.</summary>
        private static readonly TemplateHelper MainTemplateHelper = new TemplateHelper();

        /// <summary>The template helper that parses and stores the template for shader paths.</summary>
        private static readonly TemplateHelper ShaderPathTemplateHelper = new TemplateHelper();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static CustomShaderUIDefinitionWriter()
        {
            MainTemplateHelper.ParseSimpleTemplate(
                Resources.CustomShaderUIDef);

            ShaderPathTemplateHelper.ParseSimpleTemplate(
                Resources.CustomShaderUIDef_ShaderPath);
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public CustomShaderUIDefinitionWriter()
        {
        }

        /// <summary>
        /// Write custom shader UI definition Xml file that is generated
        /// from the given user definition.
        /// </summary>
        /// <param name="def">The custom shader definition to generate from.</param>
        /// <param name="dataModelName">The data model name.</param>
        /// <param name="filePath">The Xml file path to write to.</param>
        /// <param name="language">The UI language.</param>
        /// <returns>True on success.</returns>
        public bool Write(
            CustomShaderDefinition def,
            string dataModelName,
            string filePath,
            string language = null)
        {
            MainTemplateHelper.ResetTemplateTags();

            MainTemplateHelper.SetTemplateTagValue(TemplateTagNameLabel, def.GetLabelForLanguage(language));
            MainTemplateHelper.SetTemplateTagValue(TemplateTagNameLabelEn, def.LabelEN);
            MainTemplateHelper.SetTemplateTagValue(TemplateTagNameDataModelName, dataModelName);

            string shaderPaths = string.Empty;
            foreach (string path in def.VertexShaderPaths)
            {
                ShaderPathTemplateHelper.ResetTemplateTags();

                ShaderPathTemplateHelper.SetTemplateTagValue(
                    TemplateTagNameShaderPath,
                    path);

                shaderPaths += ShaderPathTemplateHelper.Compose();
            }

            MainTemplateHelper.SetTemplateTagValue(TemplateTagNameVertexShaderPaths, shaderPaths);

            shaderPaths = string.Empty;
            foreach (string path in def.FragmentShaderPaths)
            {
                ShaderPathTemplateHelper.ResetTemplateTags();

                ShaderPathTemplateHelper.SetTemplateTagValue(
                    TemplateTagNameShaderPath,
                    path);

                shaderPaths += ShaderPathTemplateHelper.Compose();
            }

            MainTemplateHelper.SetTemplateTagValue(TemplateTagNameFragmentShaderPaths, shaderPaths);

            string xml = MainTemplateHelper.Compose();

            File.WriteAllText(filePath, xml);

            return true;
        }
    }
}
