﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.ComponentModel;
    using System.Globalization;
    using System.Xml;
    using System.Xml.Schema;
    using System.Xml.Serialization;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// RGBA 成分を保持するカラーです。
    /// </summary>
    [Serializable]
    public sealed class RgbaColor : ObservableObject,
                                    IRgbaColor,
                                    IEquatable<RgbaColor>
    {
        /// <summary>
        /// アルファ値が利用されないことを示すフラグです。
        /// </summary>
        [NonSerialized]
        public const bool AlphaNotUsed = false;

        private readonly bool isAlphaEnabled;
        private float r = 0.0f;
        private float g = 0.0f;
        private float b = 0.0f;
        private float a = 1.0f;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public RgbaColor()
        {
            this.isAlphaEnabled = true;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="r">r 成分です。</param>
        /// <param name="g">g 成分です。</param>
        /// <param name="b">b 成分です。</param>
        /// <param name="a">a 成分です。</param>
        public RgbaColor(float r, float g, float b, float a)
            : this(r, g, b, a, true)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="r">r 成分です。</param>
        /// <param name="g">g 成分です。</param>
        /// <param name="b">b 成分です。</param>
        /// <param name="a">a 成分です。</param>
        /// <param name="isAlphaEnabled">アルファが有効かどうかです。</param>
        public RgbaColor(float r, float g, float b, float a, bool isAlphaEnabled)
        {
            this.isAlphaEnabled = isAlphaEnabled;
            if (!this.isAlphaEnabled)
            {
                a = 0.0f;
            }

            this.Set(r, g, b, a);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定する RgbaColor です。</param>
        public RgbaColor(IRgbaColor source)
        {
            this.isAlphaEnabled = source.IsAlphaEnabled;
            this.Set(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定する IColor です。</param>
        public RgbaColor(IColor source)
        {
            this.isAlphaEnabled = true;
            this.Set(source);
        }

        /// <summary>
        /// R 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float R
        {
            get
            {
                return this.r;
            }

            set
            {
                if (this.r == value)
                {
                    return;
                }

                this.r = value;
                OnPropertyChanged("R");
            }
        }

        /// <summary>
        /// G 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float G
        {
            get
            {
                return this.g;
            }

            set
            {
                if (this.g == value)
                {
                    return;
                }

                this.g = value;
                OnPropertyChanged("G");
            }
        }

        /// <summary>
        /// B 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float B
        {
            get
            {
                return this.b;
            }

            set
            {
                if (this.b == value)
                {
                    return;
                }

                this.b = value;
                OnPropertyChanged("B");
            }
        }

        /// <summary>
        /// A 成分を取得または設定します。
        /// </summary>
        [NotifyParentProperty(true)]
        public float A
        {
            get
            {
                if (!this.IsAlphaEnabled)
                {
                    return 0.0f;
                }

                return this.a;
            }

            set
            {
                if (!this.IsAlphaEnabled)
                {
                    return;
                }

                if (this.a == value)
                {
                    return;
                }

                this.a = value;
                OnPropertyChanged("A");
            }
        }

        /// <summary>
        /// H 成分を取得します。
        /// </summary>
        public float H
        {
            get
            {
                return this.ToHsvaColor().H;
            }
        }

        /// <summary>
        /// S 成分を取得します。
        /// </summary>
        public float S
        {
            get
            {
                return this.ToHsvaColor().S;
            }
        }

        /// <summary>
        /// V 成分を取得します。
        /// </summary>
        public float V
        {
            get
            {
                return this.ToHsvaColor().V;
            }
        }

        /// <summary>
        /// アルファが有効か取得します。
        /// </summary>
        public bool IsAlphaEnabled
        {
            get { return this.isAlphaEnabled; }
        }

        /// <summary>
        /// 2 つのベクトルを加算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsを加算した結果のベクトルです。</returns>
        public static RgbaColor operator +(RgbaColor lhs, RgbaColor rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new RgbaColor(lhs.R + rhs.R, lhs.G + rhs.G, lhs.B + rhs.B, lhs.A + rhs.A, lhs.IsAlphaEnabled | rhs.IsAlphaEnabled);
        }

        /// <summary>
        /// 2 つのベクトルを減算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsを減算した結果のベクトルです。</returns>
        public static RgbaColor operator -(RgbaColor lhs, RgbaColor rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new RgbaColor(lhs.R - rhs.R, lhs.G - rhs.G, lhs.B - rhs.B, lhs.A - rhs.A, lhs.IsAlphaEnabled | rhs.IsAlphaEnabled);
        }

        /// <summary>
        /// カラーにスカラーを乗算します。
        /// </summary>
        /// <param name="vec">演算の左のカラーです。</param>
        /// <param name="scalar">演算の右のカラーです。</param>
        /// <returns>カラーとscalarを乗算した結果のカラーです。</returns>
        public static RgbaColor operator *(RgbaColor vec, float scalar)
        {
            Ensure.Argument.NotNull(vec);

            return new RgbaColor(vec.R * scalar, vec.G * scalar, vec.B * scalar, vec.A * scalar, vec.IsAlphaEnabled);
        }

        /// <summary>
        /// カラーにスカラーを乗算します。
        /// </summary>
        /// <param name="vec">演算の左のカラーです。</param>
        /// <param name="scalar">演算の右のカラーです。</param>
        /// <returns>カラーとscalarを乗算した結果のカラーです。</returns>
        public static RgbaColor operator /(RgbaColor vec, float scalar)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Operation.DividerNotZero(scalar);

            float invScale = 1.0f / scalar;
            return new RgbaColor(
                vec.R * invScale, vec.G * invScale, vec.B * invScale, vec.A * invScale, vec.IsAlphaEnabled);
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="r">r 成分です。</param>
        /// <param name="g">g 成分です。</param>
        /// <param name="b">b 成分です。</param>
        /// <param name="a">a 成分です。</param>
        public void Set(float r, float g, float b, float a)
        {
            this.R = r;
            this.G = g;
            this.B = b;
            this.A = a;
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="source">設定する値です。</param>
        public void Set(IRgbaColor source)
        {
            Ensure.Argument.NotNull(source);

            this.R = source.R;
            this.G = source.G;
            this.B = source.B;
            this.A = source.A;
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="source">設定する値です。</param>
        public void Set(IColor source)
        {
            Ensure.Argument.NotNull(source);

            IRgbaColor rgbaColor = source as IRgbaColor;
            if (rgbaColor == null)
            {
                HsvaColor hsvaColor = source as HsvaColor;
                Ensure.Operation.ObjectNotNull(hsvaColor);
                rgbaColor = hsvaColor.ToRgbaColor();
            }

            this.Set(rgbaColor);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as RgbaColor);
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new RgbaColor(this);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(RgbaColor other)
        {
            if (other == null)
            {
                return false;
            }

            return (this.R == other.R)
                && (this.G == other.G)
                && (this.B == other.B)
                && (this.A == other.A);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as RgbaColor);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return this.R.GetHashCode() ^
                   this.G.GetHashCode() ^
                   this.B.GetHashCode() ^
                   this.A.GetHashCode();
        }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format("[{0} {1} {2} {3}]", this.R, this.G, this.B, this.A);
        }

        /// <summary>
        /// HsvaColor を取得します。
        /// </summary>
        /// <returns>HsvaColor です。</returns>
        public IHsvaColor ToHsvaColor()
        {
            float r = Math.Max( 0.0f, Math.Min( 1.0f, this.R ) );
            float g = Math.Max( 0.0f, Math.Min( 1.0f, this.G ) );
            float b = Math.Max( 0.0f, Math.Min( 1.0f, this.B ) );

            float max = Math.Max( Math.Max( r, g ), b );
            float min = Math.Min( Math.Min( r, g ), b );

            if (max == 0.0f)
            {
                return new HsvaColor(0.0f, 0.0f, 0.0f, this.A);
            }

            if (max == min)
            {
                return new HsvaColor(0.0f, 0.0f, max, this.A);
            }

            float h = 0.0f;

            if (max == r)
            {
                h = ((g - b) / (max - min)) + 0.0f;
                h /= 6.0f;
                h = h - (float)Math.Floor(h);
            }
            else if (max == g)
            {
                h = ((b - r) / (max - min)) + 2.0f;
                h /= 6.0f;
                h = h - (float)Math.Floor(h);
            }
            else
            {
                h = ((r - g) / (max - min)) + 4.0f;
                h /= 6.0f;
                h = h - (float)Math.Floor(h);
            }

            float s = (max - min) / max;
            float v = max;

            return new HsvaColor(h, s, v, this.A);
        }

        /// <summary>
        /// HSVAでRGBA色を設定します。
        /// </summary>
        /// <param name="color">The HSVA color.</param>
        public void FromHsvaColor( IHsvaColor color )
        {
            this.FromHsvaColor( color.H, color.S, color.V, color.A );
        }

        /// <summary>
        /// HSVAでRGBA色を設定します。
        /// </summary>
        /// <param name="h">Hue</param>
        /// <param name="s">Saturation</param>
        /// <param name="v">Value</param>
        /// <param name="a">Alpha</param>
        public void FromHsvaColor( float h,
                                   float s,
                                   float v,
                                   float a )
        {
            if (s == 0.0f)
            {
                this.Set( v, v, v, a );
                return;
            }

            FloatUtility.Clamp(h, 0.0f, 1.0f);

            if (h == 1.0f)
            {
                h = 0.0f;
            }

            h = h * 6.0f;
            int hi = ((int)h) % 6;

            float f = h - hi;
            float p = v * (1 - s);
            float q = v * (1 - (f * s));
            float t = v * (1 - ((1 - f) * s));

            switch (hi)
            {
                case 0:
                    this.Set( v, t, p, a );
                    return;
                case 1:
                    this.Set( q, v, p, a );
                    return;
                case 2:
                    this.Set( p, v, t, a );
                    return;
                case 3:
                    this.Set( p, q, v, a );
                    return;
                case 4:
                    this.Set( t, p, v, a );
                    return;
                case 5:
                    this.Set( v, p, q, a );
                    return;
                default:
                    this.Set( 0.0f, 0.0f, 0.0f, a );
                    return;
            }
        }
    }
}
