﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Xml;

    /// <summary>
    /// XML コンテンツのリゾルバです。
    /// </summary>
    public class XmlContentResolver : IXmlContentResolver
    {
        /// <summary>
        /// バージョンアトリビュート名です。
        /// </summary>
        private const string VersionAttributeName = "Version";

        /// <summary>
        /// アップデータです。
        /// </summary>
        private readonly IDictionary<string, IXmlContentUpdater> contentUpdaters =
            new Dictionary<string, IXmlContentUpdater>();

        /// <summary>
        /// 対象とするコンテンツの名前です。
        /// </summary>
        private string targetContentName = string.Empty;

        /// <summary>
        /// 有効なバージョン条件をチェックする処理です。
        /// </summary>
        private Predicate<string> validVersionCheckHandle;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="targetContentName">対象とするコンテンツの名前です。</param>
        public XmlContentResolver(string targetContentName)
        {
            Ensure.Argument.StringNotEmpty(targetContentName);

            this.TargetContentName = targetContentName;
        }

        /// <summary>
        /// 有効なバージョン条件をチェックする処理を取得または設定します。
        /// </summary>
        public Predicate<string> ValidVersionCheckHandler
        {
            get
            {
                return this.validVersionCheckHandle;
            }

            set
            {
                this.validVersionCheckHandle = value;
            }
        }

        /// <summary>
        /// 対象とするコンテンツの名前を取得または設定します。
        /// </summary>
        public string TargetContentName
        {
            get
            {
                return this.targetContentName;
            }

            set
            {
                Ensure.Argument.StringNotEmpty(value);
                this.targetContentName = value;
            }
        }

        /// <summary>
        /// Updaterを登録します。一度だけ呼びだします。2度目以降の呼び出しでは
        /// InvalidOperationExceptionを送出します。
        /// </summary>
        /// <param name="contentUpdaterSet">Updaterのセットです。</param>
        public void InitializeUpdaters(IXmlContentUpdater[] contentUpdaterSet)
        {
            Ensure.Argument.NotNull(contentUpdaterSet);

            // 要素がゼロではないので、関数が2回以上実行されている可能性があります。
            Ensure.Operation.True(this.contentUpdaters.Count == 0);

            foreach (IXmlContentUpdater updater in contentUpdaterSet)
            {
                this.contentUpdaters.Add(updater.Version, updater);
            }
        }

        /// <summary>
        /// コンテンツの名前からコンテンツを解決します。
        /// <para>
        /// validVersionが満たされない場合には、BadArgumentExceptionを送出します。
        /// </para>
        /// </summary>
        /// <param name="name">コンテンツの名前です。</param>
        /// <param name="xmlSubDocument">コンテンツXML文章です。</param>
        /// <returns>解決したコンテンツです。</returns>
        public IXmlContent Resolve(string name, string xmlSubDocument)
        {
            Ensure.Argument.StringNotEmpty(name);
            Ensure.Argument.StringNotEmpty(xmlSubDocument);

            // バージョン更新をします。
            xmlSubDocument = this.Update(name, xmlSubDocument);

            // コンテンツを作成して返します。
            return this.CreateXmlContent(xmlSubDocument);
        }

        /// <summary>
        /// IXmlContentインスタンスを生成します。
        /// <para>
        /// IXmlContentのサブクラスを実体化する場合は本メソッドをオーバーライドします。
        /// </para>
        /// </summary>
        /// <param name="xmlSubDocument">コンテンツXML文章です。</param>
        /// <returns>コンテンツの実体です。</returns>
        protected virtual IXmlContent CreateXmlContent(string xmlSubDocument)
        {
            Ensure.Argument.StringNotEmpty(xmlSubDocument);

            // コンテンツを作成して返します。
            IXmlContent content = new XmlContent();
            content.Xml = xmlSubDocument;

            return content;
        }

        /// <summary>
        /// できる限りのバージョンアップを実行します。
        /// バージョン更新後もバージョンが更新されていない場合は
        /// InvalidOperationException例外を送出します。
        /// </summary>
        /// <param name="name">コンテンツの名前です。</param>
        /// <param name="xmlSubDocument">コンテンツです。</param>
        /// <returns>バージョンアップされたコンテンツです。</returns>
        private string Update(string name, string xmlSubDocument)
        {
            Ensure.Argument.StringNotEmpty(name);
            Ensure.Argument.StringNotEmpty(xmlSubDocument);

            // できる限りのバージョンアップを実行します。
            string version = string.Empty;
            string lastVersion = null;
            while (true)
            {
                using (XmlTextReader reader = new XmlTextReader(new StringReader(xmlSubDocument)))
                {
                    reader.Read();

                    // バージョンに対応したアップデータを取得します。
                    version = reader.GetAttribute(VersionAttributeName);

                    // バージョン更新後もバージョンが更新されていない場合は例外を送出します。
                    Ensure.Operation.True((lastVersion == null) || (version != lastVersion));

                    IXmlContentUpdater updater = this.FindUpdaterByVersionString(version);

                    if (updater == null)
                    {
                        // アップデータが取得できなれば中断します。
                        break;
                    }
                    else
                    {
                        // バージョン更新に伴う変更を行います。
                        xmlSubDocument = updater.Update(xmlSubDocument);
                    }

                    lastVersion = version;
                }
            }

            // 最終的なバージョン文字列が求めるものではない場合は例外を投げます。
            if (this.ValidVersionCheckHandler != null)
            {
                if (!this.ValidVersionCheckHandler(version))
                {
                    string message = string.Format(
                        "Input content[{0}] version is invalid.([{1}])", name, version);
                    Ensure.Operation.True(false, message);
                }
            }

            return xmlSubDocument;
        }

        /// <summary>
        /// 指定バージョンのアップデートに対応するIContentUpdaterを探します。
        /// </summary>
        /// <param name="version">指定バージョンです。</param>
        /// <returns>IContentUpdaterです。</returns>
        private IXmlContentUpdater FindUpdaterByVersionString(string version)
        {
            if (string.IsNullOrEmpty(version))
            {
                return null;
            }

            IXmlContentUpdater result;
            this.contentUpdaters.TryGetValue(version, out result);
            return result;
        }
    }
}
