﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Windows.Forms;
using NintendoWare.ToolDevelopmentKit.Logs;
using NWCore.Protocols;

namespace NWCore.Viewer
{
    #region Packet handler delegation

    /// <summary>
    /// Packet handler delegation.
    /// </summary>
    /// <param name="header">The received packet header.</param>
    /// <param name="packetBody">The received packet body.</param>
    /// <param name="extraPacketData">Extra packet data.</param>
    public delegate void PacketHandler(Header header, Protocol4FReader reader);

    #endregion

    /// <summary>
    /// MCSマネージャ。
    /// </summary>
    public sealed class MCSManager
    {
        #region Packet handler info

        /// <summary>
        /// Class for packet handler info.
        /// </summary>
        private class PacketHandlerInfo
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="msgType">The message type the handler handles.</param>
            /// <param name="handler">The handler function.</param>
            public PacketHandlerInfo( MessageType msgType,
                                      PacketHandler handler)
            {
                m_msgType       = msgType;
                m_packetHandler = handler;
                m_invoker       = null;
            }


            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="msgType">The message type the handler handles.</param>
            /// <param name="handler">The handler function.</param>
            public PacketHandlerInfo( MessageType msgType,
                                      PacketHandler handler,
                                      Control invoker )
            {
                m_msgType       = msgType;
                m_packetHandler = handler;
                m_invoker       = invoker;
            }


            /// <summary>
            /// Get the message type this handler is listening to.
            /// </summary>
            public MessageType MessageType
            {
                get { return m_msgType; }
            }


            /// <summary>
            /// Get the message handler function.
            /// </summary>
            public PacketHandler Handler
            {
                get { return m_packetHandler; }
            }


            /// <summary>
            /// Get the invoker for executing the handler.
            /// </summary>
            public Control Invoker
            {
                get { return m_invoker; }
            }


            private MessageType   m_msgType;
            private PacketHandler m_packetHandler;
            private Control       m_invoker;
        }

        #endregion

        #region Message processor

        /// <summary>
        /// Class for message processor. The existence of this class
        /// is due to the fact that the MCSManager is designed as a
        /// static class, thus you cannot pass its reference to the
        /// CafeConnecter, and the Connector is in a source namespace,
        /// shouldn't know about MCSManager, so this is a little hacky,
        /// but work well.
        /// </summary>
        private class MessageProcessor : IMessageProcessor
        {
            /// <summary>
            /// Dispatch received packet to the registered listeners.
            /// </summary>
            public void ProcessMessage(Header header, Protocol4FReader reader)
            {
                MCSManager.ProcessMessage(header, reader);
            }

            /// <summary>
            /// Check if should filter the message.
            /// </summary>
            /// <param name="msgType">The type of the message.</param>
            /// <returns>True to keep the message, false discard it.</returns>
            public bool FilterMessage(MessageType msgType)
            {
                return MCSManager.HasListener(msgType);
            }
        }

        #endregion

        // メッセージマネージャ
        private static MessageManager s_MessageManager = new MessageManager();
        // 送信中用のダイアログ
//        private static MCSTransmissionDialog s_WaitTransmissionDialog = null;

        //---------------------------------------------------------------------
        // イベント
        //---------------------------------------------------------------------
        /// <summary>
        /// イベント：接続状態変更イベントハンドラ。
        /// </summary>
        public static event ConnectionChangedEventHandler ChangedConnection
        {
            add { s_MessageManager.ConnectionChanged += value; }
            remove { s_MessageManager.ConnectionChanged -= value; }
        }

        /// <summary>
        /// イベント：転送待ちダイアログ表示用イベント
        /// </summary>
        public static event ConnectionChangedEventHandler ShowTransmissionDialog;

        //---------------------------------------------------------------------
        // publicメソッド
        //---------------------------------------------------------------------
        #region 初期化

        /// <summary>
        /// 初期化処理。
        /// </summary>
        public static void Initialize(　string[] winViewerNames　)
        {
            LoadSettings();

            // 初期化
            s_MessageManager.Initialize(winViewerNames);

            // Set message processor
            s_MessageManager.SetMessageProcessor(m_processor);
        }

        /// <summary>
        /// 終了処理
        /// </summary>
        public static void Dispose()
        {
            // 開いているチャンネルがあればクローズします。
            if (IsConnected == true)
            {
                Close();
            }

            s_MessageManager.Dispose();
        }

        /// <summary>
        /// 設定の読み込み。
        /// </summary>
        public static void LoadSettings()
        {
        }

        /// <summary>
        /// ルーティン処理（毎フレーム呼び出し必要があります）
        /// もし呼び出さないときは、ビューア終了で自動切断されない
        /// </summary>
        public static void WndProc(ref Message m)
        {
            if (IsConnected)
            {
                s_MessageManager.FrameProcess(ref m);
            }
        }


        #endregion

        //---------------------------------------------------------------------
        #region 接続関係

        /// <summary>
        /// Set the active connection context.
        /// </summary>
        /// <param name="context">The connection context.</param>
        public static void SetActiveConnectionContext( ConnectionContext context )
        {
            s_MessageManager.SetActiveConnectionContext( context );
        }


        /// <summary>
        /// Set auto connect on or off, and the connection context to connect to.
        /// </summary>
        /// <param name="bEnable">True to enable auto connect.</param>
        /// <param name="context">The connection context.</param>
        public static void AutoConnect( bool bEnable,
                                        ConnectionContext context )
        {
            s_MessageManager.AutoConnect( bEnable, context );
        }


        /// <summary>
        /// 接続
        /// </summary>
        public static bool Connect( ConnectionContext context )
        {
            bool result = true;

            // 接続されていれば再接続
            if (IsConnected)
            {
                Close();
            }

            s_MessageManager.Connect( context );

            WaitTransmission();
            if (s_MessageManager.IsConnected == false)
            {
                // 失敗したら閉じる
                s_MessageManager.Close();
                result = false;
            }

            return result;
        }

        /// <summary>
        /// 通信で利用する文字コード
        /// </summary>
        public static Encoding EncodingType
        {
            get { return s_MessageManager.EncodingType; }
            set { s_MessageManager.EncodingType = value; }
        }

        /// <summary>
        /// ログ出力用
        /// (ログ出力を行うときは設定してください。)
        /// </summary>
        public static ILogger Logger
        {
            get { return s_MessageManager.Logger; }
            set { s_MessageManager.Logger = value; }
        }

        /// <summary>
        /// 自動接続を許可
        /// </summary>
        public static bool IsAutoConnectEnabled
        {
            get { return s_MessageManager.IsAutoConnectEnabled; }
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public static bool IsConnected
        {
            get
            {
                return s_MessageManager.IsConnected;
            }
        }

        /// <summary>
        /// ターゲットバイナリーの拡張子
        /// "_Win.ptcl" / "_Cafe.ptcl" など
        /// </summary>
        public static string BinaryFileExt
        {
            get { return s_MessageManager.BinaryFileExt; }
        }

        /// <summary>
        /// Get the active connection context.
        /// The connection context contains the information of the current connection.
        /// </summary>
        /// <returns>The active connection context, null if it's not currently connected.</returns>
        public static ConnectionContext GetActiveConnectionContext()
        {
            return s_MessageManager.GetActiveConnectionContext();
        }

        /// <summary>
        /// ビック・エンディアンか？
        /// </summary>
        public static bool IsBigEndian
        {
            get { return s_MessageManager.IsBigEndian; }
        }

        /// <summary>
        /// Get the flag indicating whether the connected target uses native formats.
        /// </summary>
        public static bool UseNativeFormat
        {
            get { return s_MessageManager.UseNativeFormat; }
        }

        /// <summary>
        /// ライターを生成
        /// </summary>
        public static Protocol4FWriter CreateWriter(Stream stream)
        {
            return s_MessageManager.CreateWriter(stream);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 閉じる。
        /// </summary>
        public static void Close()
        {
            // メインスレッド以外から呼ぶ場合はInvokeClose()を使う
            if (!IsConnected)
            {
                return;
            }

            // MCSサーバが切断を確認するためのウェイト
            s_MessageManager.Close();
        }

        #endregion

        //---------------------------------------------------------------------
        #region メッセージ

        /// <summary>
        /// メッセージ送信。
        /// </summary>
        public static void SendMessage(NWCore.Viewer.BaseMessage message)
        {
            if (!s_MessageManager.IsConnected)
            {
                return;
            }
            if (!_enable)
            {
                return;
            }

            s_MessageManager.PushMessage(message);
        }

        /// <summary>
        /// メッセージ数の取得。
        /// </summary>
        public static int GetMessageCount()
        {
            if (!s_MessageManager.IsConnected)
            {
                return 0;
            }
            if (!_enable)
            {
                return 0;
            }

            return s_MessageManager.MessageCount;
        }

        /// <summary>
        /// メッセージ送信待ち。
        /// </summary>
        public static bool WaitTransmission()
        {
            if (!s_MessageManager.IsConnected)
            {
                return true;
            }
            if (!_enable)
            {
                return true;
            }

            bool retureResult = false;
#if false
            if (s_WaitTransmissionDialog == null)
            {
                s_WaitTransmissionDialog = new MCSTransmissionDialog();
                DialogResult result = s_WaitTransmissionDialog.WaitForTransmission();
                retureResult = (result == DialogResult.OK);
                s_WaitTransmissionDialog = null;
            }
#else
            if (ShowTransmissionDialog != null)
            {
                ShowTransmissionDialog(null, new MCSConnectionChangedEventArgs(IsConnected));
            }
#endif
            return retureResult;
        }

        /// <summary>
        /// Register handler for the specified message type.
        /// </summary>
        /// <param name="msgType">The message type to listener.</param>
        /// <param name="handler">Handler function for the message.</param>
        /// <param name="invoker">The control to invoke the handler method.</param>
        public static void RegisterMsgListener( MessageType msgType,
                                                PacketHandler handler,
                                                Control invoker = null )
        {
            if (handler == null)
                return;

            m_packetHandlers.Add( new PacketHandlerInfo( msgType,
                                                         handler,
                                                         invoker ) );
        }

        /// <summary>
        /// Remove the message listener.
        /// </summary>
        /// <param name="msgType">Message type to listen.</param>
        /// <param name="handler">
        /// The handler function called when the specified type of
        /// message is received.
        /// </param>
        public static void RemoveMsgListener( MessageType msgType,
                                              PacketHandler handler)
        {
            int i;
            for (i = 0; i < m_packetHandlers.Count; ++i)
            {
                PacketHandlerInfo info = m_packetHandlers[i];
                if (info != null &&
                     info.MessageType == msgType &&
                     info.Handler == handler)
                {
                    m_packetHandlers.RemoveAt(i);
                    return;
                }
            }
        }

        /// <summary>
        /// Check if there is any packet listener for this type of message.
        /// </summary>
        /// <param name="msgType">Type of the message.</param>
        /// <returns>True if there is listeners for this type of message.</returns>
        public static bool HasListener(MessageType msgType)
        {
            int i;
            for ( i=0;i<m_packetHandlers.Count;++i )
            {
                PacketHandlerInfo info = m_packetHandlers[i];
                if ( info!=null &&
                     info.MessageType==msgType )
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Dispatch received packet to the registered listeners.
        /// </summary>
        /// <param name="header">Packet header</param>
        /// <param name="reader">Packet reader</param>
        public static void ProcessMessage(Header header, Protocol4FReader reader)
        {
            int i;
            for (i = 0; i < m_packetHandlers.Count; ++i)
            {
                PacketHandlerInfo info = m_packetHandlers[i];
                if ( info!=null &&
                     info.MessageType==header.Type &&
                     info.Handler!=null )
                {
                    if ( info.Invoker!=null && info.Invoker.InvokeRequired==true )
                    {
                        info.Invoker.Invoke( info.Handler, header, reader );
                    }
                    else
                    {
                        info.Handler( header, reader );
                    }
                }
            }
        }

        #endregion

        //---------------------------------------------------------------------
        // プロパティ
        //---------------------------------------------------------------------
        /// <summary>
        /// 有効無効。
        /// </summary>
        public static bool IsEnable
        {
            get { return _enable; }
            set
            {
                if ((!_enable) && value)
                {
                    //TODO:未対応 -> 選択状態をクリアするパケットを送信する
#if false
                    // 無効から有効になったら選択状態をクリアする
                    EditSelectionMessage.Clear();
#endif
                }
                _enable = value;
                s_MessageManager.IsEnabled = value;
            }
        }

        //---------------------------------------------------------------------
        // privateメソッド
        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private MCSManager() { }

        //---------------------------------------------------------------------
        // メンバ
        //---------------------------------------------------------------------
        #region メンバ

        // 有効無効
        private static bool _enable = true;

        private static List<PacketHandlerInfo> m_packetHandlers =
            new List<PacketHandlerInfo>();

        private static MessageProcessor m_processor = new MessageProcessor();

        #endregion

    }

    //-------------------------------------------------------------------------
    // MCS接続変更イベント
    //-------------------------------------------------------------------------
    #region MCSConnectionChangedEvent

    /// <summary>
    /// MCS接続変更イベントハンドラ。
    /// </summary>
    public delegate void ConnectionChangedEventHandler( object sender,
                                                        MCSConnectionChangedEventArgs e);

    /// <summary>
    /// MCS接続変更イベント引数クラス。
    /// </summary>
    public sealed class MCSConnectionChangedEventArgs : EventArgs
    {
        // 接続しているか
        private readonly bool _connected;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MCSConnectionChangedEventArgs(bool connected)
        {
            _connected = connected;
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public bool IsConnected
        {
            get { return _connected; }
        }
    }

    #endregion

    //-------------------------------------------------------------------------
    // MCS無効ブロック
    //-------------------------------------------------------------------------
    #region MCSDisableBlock

    /// <summary>
    /// MCS無効ブロック。
    /// </summary>
    public sealed class MCSDisableBlock : IDisposable
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MCSDisableBlock()
        {
            _enable = MCSManager.IsEnable;
            MCSManager.IsEnable = false;
        }

        /// <summary>
        /// 破棄。
        /// </summary>
        public void Dispose()
        {
            MCSManager.IsEnable = _enable;
        }

        private bool _enable;
    }
    #endregion

    #region MCSFreezeBlock
    /// <summary>
    /// 描画更新の制御ブロック。
    /// </summary>
    public sealed class MCSFreezeBlock : IDisposable
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MCSFreezeBlock()
        {
            //TODO:未対応　→ 描画更新の制御ブロックするパケットを送信する
            //            FreezeMessage.Send(true);
        }

        /// <summary>
        /// 破棄。
        /// </summary>
        public void Dispose()
        {
            //TODO:未対応　→ 描画更新の制御ブロック解除するパケットを送信する
            //            FreezeMessage.Send(false);
        }
    }

    #endregion

    //-------------------------------------------------------------------------
    // Use this block instance to send messages disregarding whether the MCS
    // manager is enabled or not.
    //-------------------------------------------------------------------------
    #region MCSEnableBlock

    /// <summary>
    /// Use this block instance to send messages disregarding whether the MCS
    /// manager is enabled or not.
    /// </summary>
    public sealed class MCSEnableBlock : IDisposable
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MCSEnableBlock()
        {
            m_bEnable = MCSManager.IsEnable;
            MCSManager.IsEnable = true;
        }

        /// <summary>
        /// 破棄。
        /// </summary>
        public void Dispose()
        {
            MCSManager.IsEnable = m_bEnable;
        }

        private bool m_bEnable;
    }
    #endregion

    //-------------------------------------------------------------------------
}
