﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// PathUtility.
    /// </summary>
    public static class PathUtility
    {
        /// <summary>
        /// Get the directory name of the given path without the ending "\" and "/".
        /// The given path must be an absolute path and must not be empty.
        /// - For a file path, System.IO.Path.GetDirectoryName() returns correct path.
        /// - For a dir path with ending "/" or "\", System.IO.Path.GetDirectoryName()
        ///   returns correct path.
        /// - For a dir path WITHOUT ending "/" or "\", System.IO.Path.GetDirectoryName()
        ///   returns the parent directory path.
        /// So we need a method which always returns the current directory without "/" and "\".
        /// </summary>
        /// <param name="path">The file or directory path.</param>
        /// <returns>The directory path without ending "\" and "/".</returns>
        public static string GetDirectoryName(string path)
        {
            // Validate the given path first ( must not be empty, and must be absolute path ).
            if (string.IsNullOrEmpty(path) == true)
            {
                return string.Empty;
            }

            if (Path.IsPathRooted(path) == false)
            {
                return string.Empty;
            }

            string dirPath;

            // Is the given path a directory?
            // We want the directory path without the ending "\" and "/"
            if (string.IsNullOrEmpty(Path.GetExtension(path)) == true)
            {
                // It is a directory, remove the ending "\" and "/"
                dirPath = path.TrimEnd('\\', '/');
            }
            else
            {
                // It is a file path, just get the directory name.
                dirPath = Path.GetDirectoryName(path);
            }

            return dirPath;
        }

        /// <summary>
        /// True if the path is both set and absolute
        /// </summary>
        /// <param name="path">Path</param>
        /// <returns>True if both set and absolute</returns>
        public static bool IsPathSetAndRooted(string path)
        {
            try
            {
                if (string.IsNullOrEmpty(path) == false &&
                    Path.IsPathRooted(path) == true)
                {
                    return true;
                }
            }
            catch
            {
            }

            return false;
        }

        /// <summary>
        /// List all the valid asset file paths for the given emitter set file path.
        /// The first returned path would be the directory that the emitter set is
        /// at, no matter if it exists or not.
        /// </summary>
        /// <param name="assetFolderName">The folder name of the assets.</param>
        /// <param name="esetFilePath">The emitter set file path ( or the directory ).</param>
        /// <param name="checkDirExistence">True to filter out the paths that do not exist.</param>
        /// <returns>A list of valid and existing asset directory paths.</returns>
        public static List<string> ListValidAssetPaths(
            string assetFolderName,
            string esetFilePath,
            bool checkDirExistence)
        {
            List<string> pathList = new List<string>();

            string dirPath = GetDirectoryName(esetFilePath);

            // Validate the eset file path first.
            if (IsPathSetAndRooted(dirPath) == false)
            {
                return pathList;
            }

            // Add the directory of the eset file to our list.
            pathList.Add(dirPath);

            // If the folder exists, add it to our list.
            string texPath = Path.Combine(dirPath, assetFolderName);
            if (checkDirExistence == false ||
                 Directory.Exists(texPath) == true)
            {
                pathList.Add(texPath);
            }

            // Traverse to the root directory
            while (Path.IsPathRooted(dirPath) == true)
            {
                // Get to the parent directory.
                dirPath = Path.GetDirectoryName(dirPath);
                if (string.IsNullOrEmpty(dirPath) == true)
                {
                    break;
                }

                // If the folder exists, add it to our list.
                texPath = Path.Combine(dirPath, assetFolderName);
                if (checkDirExistence == false ||
                     Directory.Exists(texPath) == true)
                {
                    pathList.Add(texPath);
                }
            }

            return pathList;
        }

        /// <summary>
        /// エクスプローラーでフォルダを開きます。
        /// </summary>
        /// <param name="path">フォルダパス</param>
        /// <param name="selectFile">ファイル選択のON/OFF</param>
        public static void OpenPathByExplorer(
            string path,
            bool selectFile = true)
        {
            if (path.Length > 0)
            {
                string commandline;

                if (selectFile == true)
                {
                    commandline = "/e, /select,\"" + path + "\"";
                }
                else
                {
                    commandline = "/e,\"" + path + "\"";
                }

                System.Diagnostics.Process.Start("EXPLORER.EXE", commandline);
            }
        }

        /// <summary>
        /// 相対パスを取得します。
        /// </summary>
        /// <param name="rootPath">元にするルートパス</param>
        /// <param name="fullPath">相対パスを求めるパス</param>
        /// <returns>相対パスを返します。</returns>
        public static string GetRelativePath(string rootPath, string fullPath)
        {
            try
            {
                if (string.IsNullOrEmpty(rootPath))
                {
                    return fullPath;
                }
                else
                {
                    Uri root = new Uri(rootPath, UriKind.RelativeOrAbsolute);
                    Uri path = new Uri(fullPath, UriKind.RelativeOrAbsolute);
                    path = root.MakeRelativeUri(path);
                    return path.ToString();
                }
            }
            catch
            {
                return fullPath;
            }
        }

        /// <summary>
        /// パスに埋め込まれた環境変数をロードします。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <returns>環境変数をロードしたパスを返します。</returns>
        public static string LoadEnvironments(string path)
        {
            if (string.IsNullOrEmpty(path))
            {
                return path;
            }

            MatchCollection matches;

            // $(Env)に該当する文字列を検索, 置換
            matches = Regex.Matches(path, @"\$\(\w+\)", RegexOptions.IgnoreCase);

            foreach (Match item in matches)
            {
                string envName = item.Value.Substring(2, item.Value.Length - 3);
                string envValue = Environment.GetEnvironmentVariable(envName);

                // 未設定の環境変数は置換しない
                if (envValue == null)
                {
                    continue;
                }

                path = path.Replace(item.Value, envValue);
            }

            // %Env%に該当する文字列を検索, 置換
            matches = Regex.Matches(path, @"\%\w+\%", RegexOptions.IgnoreCase);

            foreach (Match item in matches)
            {
                string envName = item.Value.Substring(1, item.Value.Length - 2);
                string envValue = Environment.GetEnvironmentVariable(envName);

                // 未設定の環境変数は置換しない
                if (envValue == null)
                {
                    continue;
                }

                path = path.Replace(item.Value, envValue);
            }

            return path;
        }

        /// <summary>
        /// Helper method for make the given path consistent as an absolute file path.
        /// </summary>
        /// <param name="path">The file path.</param>
        /// <param name="workingFolder">The working folder.</param>
        /// <returns>The absolute path.</returns>
        public static string ToAbsolutePath(
            string path,
            string workingFolder = null)
        {
            if (string.IsNullOrEmpty(path) == true)
            {
                return string.Empty;
            }

            // 環境変数をロード
            path = LoadEnvironments(path);

            if (Path.IsPathRooted(path) == false)
            {
                if (string.IsNullOrEmpty(workingFolder) == true)
                {
                    return string.Empty;
                }
                else
                {
                    return Path.Combine(workingFolder, path);
                }
            }

            return string.Copy(path);
        }
    }
}
