﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

using EffectMaker.Foundation.Log;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// Utility class for serialization and deserialization.
    /// </summary>
    public abstract class SerializationHelper
    {
        /// <summary>
        /// Event triggers when the serialization helper needs to collect extra types for
        /// deserialization.
        /// </summary>
        public static event EventHandler<RequestExtraTypesEventArgs> RequestExtraTypes;

        /// <summary>
        /// Get extra types for deserialization.
        /// </summary>
        /// <returns>The array of extra types.</returns>
        public static Type[] GetExtraTypes()
        {
            var args = new RequestExtraTypesEventArgs();

            if (RequestExtraTypes != null)
            {
                RequestExtraTypes(null, args);
            }

            return args.ExtraTypes;
        }

        /// <summary>
        /// Deserializes an instance from a given data stream.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="stream">The stream that contains the serialized instance data.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T Load<T>(Stream stream, bool includeExtraTypes = false)
        {
            try
            {
                XmlSerializer serializer = null;
                if (includeExtraTypes == true)
                {
                    serializer = new XmlSerializer(typeof(T), GetExtraTypes());
                }
                else
                {
                    serializer = new XmlSerializer(typeof(T));
                }

                return (T)serializer.Deserialize(stream);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from stream.", typeof(T).Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from a file.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="filePath">The file path.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T Load<T>(string filePath, bool includeExtraTypes = false)
        {
            byte[] buffer = null;

            try
            {
                // First load all contents from the file, to speed up loading.
                using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    buffer = new byte[fs.Length];
                    fs.Read(buffer, 0, (int)fs.Length);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading file from {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }

            try
            {
                XmlSerializer serializer = null;
                if (includeExtraTypes == true)
                {
                    serializer = new XmlSerializer(typeof(T), GetExtraTypes());
                }
                else
                {
                    serializer = new XmlSerializer(typeof(T));
                }

                using (var ms = new MemoryStream(buffer))
                {
                    return (T)serializer.Deserialize(ms);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed deserializing the file contents from {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Serializes an instance to a given stream.
        /// </summary>
        /// <typeparam name="T">The type of object to serialize.</typeparam>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="stream">The stream to serialize the instance to.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save<T>(T instance, Stream stream, bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            XmlSerializer serializer = null;
            if (includeExtraTypes == true)
            {
                serializer = new XmlSerializer(typeof(T), GetExtraTypes());
            }
            else
            {
                serializer = new XmlSerializer(typeof(T));
            }

            if (stream is FileStream)
            {
                using (var ms = new MemoryStream())
                {
                    serializer.Serialize(ms, instance, ns);

                    stream.Write(ms.GetBuffer(), 0, (int)ms.Length);
                }
            }
            else
            {
                serializer.Serialize(stream, instance, ns);
            }
        }

        /// <summary>
        /// Serializes an instance to a file.
        /// </summary>
        /// <typeparam name="T">The type of object to serialize.</typeparam>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="filePath">The file path.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save<T>(T instance, string filePath, bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            XmlSerializer serializer = null;
            if (includeExtraTypes == true)
            {
                serializer = new XmlSerializer(typeof(T), GetExtraTypes());
            }
            else
            {
                serializer = new XmlSerializer(typeof(T));
            }

            int size = 0;
            byte[] buffer = null;
            using (var ms = new MemoryStream())
            {
                serializer.Serialize(ms, instance, ns);

                buffer = ms.GetBuffer();
                size = (int)ms.Length;
            }

            using (var fs = new FileStream(filePath, FileMode.Create, FileAccess.Write))
            {
                fs.Write(buffer, 0, size);
            }
        }

        /// <summary>
        /// Clones the instance by serializing, then deserializing it.
        /// </summary>
        /// <typeparam name="T">The type of obejct to clone.</typeparam>
        /// <param name="instance">The instance to clone.</param>
        /// <returns>Returns a cloned instance of the input one.
        /// Throws an exception on failure.</returns>
        public static T Clone<T>(T instance)
        {
            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T), string.Empty);
                serializer.Serialize(stream, instance, ns);
                stream.Position = 0;
                return (T)serializer.Deserialize(stream);
            }
        }
    }
}
