﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.DataModel.Manager
{
    /// <summary>
    /// ユーザー定義ページ情報.
    /// </summary>
    [Serializable]
    public class UserDataModelInfo
    {
        /// <summary>
        /// コンストラクタ.
        /// </summary>
        public UserDataModelInfo()
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// 追加するタブ名.
        /// </summary>
        [XmlAttribute("Target")]
        public string TargetTabName { get; set; }

        /// <summary>
        /// .csファイル名.
        /// </summary>
        [XmlAttribute("DataModel")]
        public string DataModelFileName { get; set; }

        /// <summary>
        /// .xamlファイル名.
        /// </summary>
        [XmlAttribute("UI")]
        public string XamlFileName { get; set; }

        /// <summary>
        /// コンバート情報ファイル名.
        /// </summary>
        [XmlAttribute("ConversionInfo")]
        public string ConversionInfoFileName { get; set; }
    }

    /// <summary>
    /// ユーザーデータ情報.
    /// </summary>
    [Serializable]
    public class UserDataInfo
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public UserDataInfo()
        {
            this.UserDataModelList = new List<UserDataModelInfo>();
        }

        /// <summary>
        /// ユーザーデータモデル情報のリストです.
        /// </summary>
        [XmlElement("UserDataModelInfo")]
        public List<UserDataModelInfo> UserDataModelList { get; set; }

        /// <summary>
        /// ソースコードのファイルパスです.
        /// </summary>
        [XmlIgnore]
        public IEnumerable<string> SourceCodeFilePaths
        {
            get
            {
                foreach (var file in this.UserDataModelList)
                {
                    yield return file.DataModelFileName;
                }

                foreach (var file in this.UserDataModelList)
                {
                    if (!string.IsNullOrEmpty(file.ConversionInfoFileName))
                    {
                        yield return file.ConversionInfoFileName;
                    }
                }
            }
        }
    }

    /// <summary>
    /// ユーザーページ情報の生成
    /// </summary>
    public class UserPageBuilder
    {
        /// <summary>
        /// セットアップ
        /// </summary>
        /// <param name="filePath">保存するファイルパス.</param>
        public void Setup(string filePath)
        {
            UserDataInfo info = new UserDataInfo();

            // 自動移動.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "Preview",
                DataModelFileName = "PreviewAutoMoveData.cs",
                XamlFileName = "PreviewAutoMoveData.xaml",
                ConversionInfoFileName = string.Empty,
            });

            // カラー.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "Preview",
                DataModelFileName = "PreviewColorData.cs",
                XamlFileName = "PreviewColorData.xaml",
                ConversionInfoFileName = string.Empty,
            });

            // パーティクル放出.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "Preview",
                DataModelFileName = "PreviewParticleEmissionData.cs",
                XamlFileName = "PreviewParticleEmissionData.xaml",
                ConversionInfoFileName = string.Empty,
            });

            // パーティクルスケール.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "Preview",
                DataModelFileName = "PreviewParticleScaleData.cs",
                XamlFileName = "PreviewParticleScaleData.xaml",
                ConversionInfoFileName = string.Empty,
            });

            // パーティクル制御.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "Preview",
                DataModelFileName = "PreviewParticleControlData.cs",
                XamlFileName = "PreviewParticleControlData.xaml",
                ConversionInfoFileName = string.Empty,
            });

            // 連結式ストライプ.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "StripeComplex",
                DataModelFileName = "StripeComplexUserData.cs",
                XamlFileName = "StripeComplexData.xaml",
                ConversionInfoFileName = "StripeComplexBinaryConversionInfo.cs",
            });

            // 履歴式ストライプ.
            info.UserDataModelList.Add(new UserDataModelInfo()
            {
                TargetTabName = "StripeHistory",
                DataModelFileName = "StripeHistoryUserData.cs",
                XamlFileName = "StripeHistoryData.xaml",
                ConversionInfoFileName = "StripeHistoryBinaryConversionInfo.cs",
            });

            // ファイルを開く
            try
            {
                SerializationHelper.Save<UserDataInfo>(info, filePath);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "UserPageBuilder.Setup : Failed saving user data info to file {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
            }
        }
    }
}
