﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.DataModel.AnimationTable
{
    /// <summary>
    /// Class to store animation table data.
    /// </summary>
    public class AnimationTableData : IXmlSerializable,
                                      ISettable,
                                      ICloneable,
                                      IEquatable<object>,
                                      IEnumerable<KeyFrameData>
    {
        /// <summary>
        /// GetHashCode()で使用するハッシュテーブル.
        /// </summary>
        private static readonly int[] HashTable = { 0x77073096, 0x6e0e612c, 0x690951ba, 0x076dc419 };

        /// <summary>The key frame data list.</summary>
        private readonly List<KeyFrameData> keyFrames = new List<KeyFrameData>();

        /// <summary>The flag indicating whether to ignore NotifyKeyFrameSelected calls.</summary>
        private bool ignoreNotifyKeyFrameSelected = false;

        /// <summary>The flag indicating whether multi-selection is enabled.</summary>
        private bool isMultiSelect = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public AnimationTableData()
        {
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source animation table data to copy from.</param>
        public AnimationTableData(AnimationTableData src)
        {
            this.Set(src);
        }

        /// <summary>
        /// コンストラクタ
        /// 初期値キーを追加する
        /// </summary>
        /// <param name="frame">フレーム</param>
        /// <param name="x">値x</param>
        /// <param name="y">値y</param>
        /// <param name="z">値z</param>
        /// <param name="w">値w</param>
        public AnimationTableData(int frame, float x, float y, float z, float w)
        {
            this.AddKeyFrame(frame, new Vector4f(x, y, z, w), false);
        }

        /// <summary>
        /// コンストラクタ
        /// 初期値キーを追加する
        /// </summary>
        /// <param name="keyFrames">キーフレーム</param>
        public AnimationTableData(IEnumerable<FrameValuePair> keyFrames)
        {
            System.Diagnostics.Debug.Assert(keyFrames != null, "keyFrames != null");

            foreach (var fv in keyFrames)
            {
                this.AddKeyFrame(fv.Frame, fv.Value, false);
            }
        }

        /// <summary>
        /// Get the count of the key frames.
        /// </summary>
        public int Count
        {
            get { return this.keyFrames.Count; }
        }

        /// <summary>
        /// Get or set the flag indicating whether multi-selection is enabled.
        /// </summary>
        public bool MultiSelect
        {
            get
            {
                return this.isMultiSelect;
            }

            set
            {
                if (this.isMultiSelect != value)
                {
                    this.isMultiSelect = value;
                    if (value == false)
                    {
                        // Single selection is set, deselect all key frame except
                        // for the first selection.
                        this.ignoreNotifyKeyFrameSelected = true;

                        bool firstSelectionFound = false;
                        foreach (KeyFrameData keyFrame in this.keyFrames)
                        {
                            if (keyFrame.IsSelected == true)
                            {
                                if (firstSelectionFound == false)
                                {
                                    firstSelectionFound = true;
                                }
                                else
                                {
                                    keyFrame.IsSelected = false;
                                }
                            }
                        }

                        this.ignoreNotifyKeyFrameSelected = false;
                    }
                }
            }
        }

        /// <summary>
        /// Get the first selected key frame in the table.
        /// </summary>
        public KeyFrameData SelectedKeyFrame
        {
            get
            {
                return this.keyFrames.FirstOrDefault(key => key.IsSelected);
            }
        }

        /// <summary>
        /// Get the first selected key frame index in the table.
        /// </summary>
        public int SelectedKeyFrameIndex
        {
            get
            {
                for (var i = 0; i != this.keyFrames.Count; ++i)
                {
                    if (this.keyFrames[i].IsSelected)
                    {
                        return i;
                    }
                }

                return -1;
            }
        }

        /// <summary>
        /// Enumerate all the selected key frames.
        /// </summary>
        public IEnumerable<KeyFrameData> SelectedKeyFrames
        {
            get
            {
                return this.keyFrames.Where(key => key.IsSelected);
            }
        }

        /// <summary>
        /// Get or set the key frame data at the specified index.
        /// </summary>
        /// <param name="index">Index to the key frame data.</param>
        /// <returns>The key frame data.</returns>
        public KeyFrameData this[int index]
        {
            get
            {
                if (index < 0 || index >= this.keyFrames.Count)
                {
                    throw new IndexOutOfRangeException();
                }

                return this.keyFrames[index];
            }

            set
            {
                if (index < 0 || index >= this.keyFrames.Count)
                {
                    throw new IndexOutOfRangeException();
                }

                this.keyFrames[index].Set(value);
            }
        }

        /// <summary>
        /// The clear.
        /// </summary>
        public void Clear()
        {
            this.keyFrames.Clear();
        }

        /// <summary>
        /// The get list.
        /// </summary>
        /// <returns>
        /// The <see cref="List"/>.
        /// </returns>
        public List<KeyFrameData> GetList()
        {
            return this.keyFrames;
        }

        /// <summary>
        /// Get enumerator for the key frames.
        /// </summary>
        /// <returns>The key frame enumerator.</returns>
        public IEnumerator<KeyFrameData> GetEnumerator()
        {
            return this.keyFrames.GetEnumerator();
        }

        /// <summary>
        /// Get non-generic enumerator.
        /// </summary>
        /// <returns>The enumerator.</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }

        /// <summary>
        /// Add a new key frame.
        /// </summary>
        /// <param name="frame">The frame number.</param>
        /// <param name="value">The value of the key frame.</param>
        /// <param name="isSelected">isSelected</param>
        /// <returns>True on success. False if the key frame already exists.</returns>
        public bool AddKeyFrame(int frame, Vector4f value, bool isSelected)
        {
            return this.AddKeyFrame(frame, value.X, value.Y, value.Z, value.W, isSelected);
        }

        /// <summary>
        /// Add a new key frame.
        /// </summary>
        /// <param name="frame">The frame number.</param>
        /// <param name="color">The color for the key frame.</param>
        /// <param name="isSelected">isSelected</param>
        /// <returns>True on success. False if the key frame already exists.</returns>
        public bool AddKeyFrame(int frame, ColorRgba color, bool isSelected)
        {
            return this.AddKeyFrame(frame, color.R, color.G, color.B, color.A, isSelected);
        }

        /// <summary>
        /// Add a new key frame.
        /// </summary>
        /// <param name="frame">The frame number.</param>
        /// <param name="x">The value X of the key frame.</param>
        /// <param name="y">The value Y of the key frame.</param>
        /// <param name="z">The value Z of the key frame.</param>
        /// <param name="w">The value W of the key frame.</param>
        /// <param name="isSelected">isSelected</param>
        /// <returns>True on success. False if the key frame already exists.</returns>
        public bool AddKeyFrame(int frame, float x, float y, float z, float w, bool isSelected)
        {
            // Check if any existing key frame has the specified frame number.
            // When the key frame doesn't exist, IndexOfKeyFrame() returns the
            // complement of the index to insert.
            int index = this.IndexOfKeyFrame(frame);
            if (index >= 0)
            {
                return false;
            }

            // Insert the key frame to the correct location.
            // The key frame inserts to the ordered location, so no need to sort.
            this.keyFrames.Insert(~index, new KeyFrameData(frame, x, y, z, w) { Owner = this, IsSelected = isSelected });

            return true;
        }

        /// <summary>
        /// Remove the key frame with the specified frame number.
        /// </summary>
        /// <param name="frame">The frame number.</param>
        public void RemoveKeyFrame(int frame)
        {
            int index = this.IndexOfKeyFrame(frame);
            if (index >= 0)
            {
                this.keyFrames[index].Owner = null;
                this.keyFrames.RemoveAt(index);
            }
        }

        /// <summary>
        /// Remove the key frame at the specified index.
        /// </summary>
        /// <param name="index">The index to the key frame.</param>
        public void RemoveKeyFrameAt(int index)
        {
            if (index >= 0 && index < this.keyFrames.Count)
            {
                this.keyFrames[index].Owner = null;
                this.keyFrames.RemoveAt(index);
            }
        }

        /// <summary>
        /// Find the index of the key frame which has the specified key frame number.
        /// </summary>
        /// <param name="frame">The frame number.</param>
        /// <returns>
        /// The zero-based index of item in the list, if key frame is found,
        /// otherwise, a negative number that is the bitwise complement of the
        /// index of the next element that is larger than item or, if there is
        /// no larger element, the bitwise complement of Count.
        /// </returns>
        public int IndexOfKeyFrame(int frame)
        {
            int count = this.keyFrames.Count;
            if (count <= 0)
            {
                return ~0;
            }

            int lo = 0;
            int hi = count - 1;
            int mid;
            while (lo <= hi)
            {
                mid = (lo + hi) / 2;
                KeyFrameData data = this.keyFrames[mid];
                if (frame == data.Frame)
                {
                    return mid;
                }

                if (frame < data.Frame)
                {
                    hi = mid - 1;
                }
                else
                {
                    lo = mid + 1;
                }
            }

            // Return bitwise complement of the first element greater than value.
            // Since hi is less than lo now, ~lo is the correct item.
            return ~lo;
        }

        /// <summary>
        /// Sort the key frames according to the frame number.
        /// </summary>
        public void SortKeyFrames()
        {
            if (this.keyFrames.Count < 2)
            {
                // No need to sort, return.
                return;
            }

            this.SortKeyFramesInternal(0, this.keyFrames.Count - 1);
        }

        /// <summary>
        /// Compare with the given object and check if they are equal.
        /// </summary>
        /// <param name="other">The object to compare with.</param>
        /// <returns>True if the instances are equal.</returns>
        public override bool Equals(object other)
        {
            var otherTable = other as AnimationTableData;
            if (otherTable == null)
            {
                return false;
            }

            if (otherTable.Count != this.Count)
            {
                return false;
            }

            for (int i = 0; i < this.Count; ++i)
            {
                if (this[i].Equals(otherTable[i]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Get hash code of this key frame table.
        /// </summary>
        /// <returns>The hash code.</returns>
        public override int GetHashCode()
        {
            int num = 42;
            unchecked
            {
                for (int i = 0; i < this.Count; ++i)
                {
                    int hash = this.keyFrames[i].GetHashCode();
                    num = (num * 37) + hash;
                    num ^= hash >> 16;
                    num ^= HashTable[(hash >> 30) & 3];
                }
            }

            return num;
        }

        /// <summary>
        /// Set data.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var srcAnimTableData = src as AnimationTableData;
            if (srcAnimTableData == null)
            {
                return false;
            }

            this.isMultiSelect = srcAnimTableData.isMultiSelect;

            // First remove all the key frames that doesn't exist in the source animation table.
            for (int i = this.keyFrames.Count - 1; i >= 0; --i)
            {
                KeyFrameData myKeyData = this.keyFrames[i];
                if (srcAnimTableData.IndexOfKeyFrame(myKeyData.Frame) < 0)
                {
                    this.RemoveKeyFrameAt(i);
                }
            }

            // Add the non-existing key frames to the list
            // and copy data to the existing key frames.
            foreach (KeyFrameData keyData in srcAnimTableData.keyFrames)
            {
                int index = this.IndexOfKeyFrame(keyData.Frame);
                if (index < 0)
                {
                    // The key frame is added to the correct location,
                    // so no need to sort afterward.

                    //// this.AddKeyFrame(keyData.Frame, keyData.Value);
                    this.AddKeyFrame(keyData.Frame, keyData.Value, keyData.IsSelected);
                }
                else
                {
                    // The key frame exists, just copy the data to it.
                    this.keyFrames[index].Set(keyData);
                }
            }

            return true;
        }

        /// <summary>
        /// Clone this animation table data.
        /// </summary>
        /// <returns>The create animation table data.</returns>
        public object Clone()
        {
            return new AnimationTableData(this);
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            // Read multi-selection flag.
            bool tmpMultiSelect;
            string strMultiSelect = reader.GetAttribute("MultiSelect");
            if (string.IsNullOrEmpty(strMultiSelect) == true ||
                bool.TryParse(strMultiSelect, out tmpMultiSelect) == false)
            {
                tmpMultiSelect = false;
            }

            this.isMultiSelect = tmpMultiSelect;

            // early break if there is no child to process.
            if (reader.IsEmptyElement)
            {
                return;
            }

            this.ignoreNotifyKeyFrameSelected = true;

            // Read key frames.
            bool firstSelectionFound = false;
            string myLocalName = reader.LocalName;
            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element &&
                    reader.Name == "KeyFrame")
                {
                    // Create a key frame to read from Xml.
                    var data = new KeyFrameData()
                    {
                        Owner = this
                    };

                    // Read the key frame and add it to our key frame list.
                    data.ReadXml(reader);
                    this.keyFrames.Add(data);

                    // If single selection is set, deselect all key frames
                    // except the first selection.
                    if (tmpMultiSelect == false &&
                        data.IsSelected == true)
                    {
                        if (firstSelectionFound == false)
                        {
                            firstSelectionFound = true;
                        }
                        else
                        {
                            data.IsSelected = false;
                        }
                    }
                }
                else if (reader.NodeType == XmlNodeType.EndElement &&
                    reader.LocalName == myLocalName)
                {
                    // This is the end tag of the animation table element.
                    // Read once more, so the next element can be read correctly.
                    reader.Read();
                    break;
                }
            }

            this.ignoreNotifyKeyFrameSelected = false;

            this.SortKeyFrames();
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("MultiSelect", this.isMultiSelect.ToString());

            foreach (KeyFrameData data in this.keyFrames)
            {
                writer.WriteStartElement("KeyFrame");
                data.WriteXml(writer);
                writer.WriteEndElement();
            }
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Notify key frame being selected.
        /// </summary>
        /// <param name="selectedKeyFrame">The selected key frame.</param>
        internal void NotifyKeyFrameSelected(KeyFrameData selectedKeyFrame)
        {
            if (this.ignoreNotifyKeyFrameSelected == true)
            {
                return;
            }

            this.ignoreNotifyKeyFrameSelected = true;

            if (this.MultiSelect == false)
            {
                foreach (KeyFrameData keyFrame in this.keyFrames)
                {
                    if (object.ReferenceEquals(keyFrame, selectedKeyFrame) == false)
                    {
                        keyFrame.IsSelected = false;
                    }
                }
            }

            this.ignoreNotifyKeyFrameSelected = false;
        }

        /// <summary>
        /// Internal method for sorting the key frames recursively
        /// according to the frame number.
        /// </summary>
        /// <param name="left">The index to the left element to sort.</param>
        /// <param name="right">The index to the right element to sort.</param>
        private void SortKeyFramesInternal(int left, int right)
        {
            int i = left, j = right;
            KeyFrameData pivot = this.keyFrames[(left + right) / 2];

            while (i <= j)
            {
                while (this.keyFrames[i].Frame < pivot.Frame)
                {
                    ++i;
                }

                while (this.keyFrames[j].Frame > pivot.Frame)
                {
                    --j;
                }

                if (i <= j)
                {
                    // Swap
                    KeyFrameData tmp = this.keyFrames[i];
                    this.keyFrames[i] = this.keyFrames[j];
                    this.keyFrames[j] = tmp;

                    ++i;
                    --j;
                }
            }

            // Recursive calls.
            if (left < j)
            {
                this.SortKeyFramesInternal(left, j);
            }

            if (i < right)
            {
                this.SortKeyFramesInternal(i, right);
            }
        }

        /// <summary>
        /// フレーム・値
        /// </summary>
        public class FrameValuePair
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="frame">フレーム</param>
            /// <param name="x">値x</param>
            /// <param name="y">値y</param>
            /// <param name="z">値z</param>
            /// <param name="w">値w</param>
            public FrameValuePair(int frame, float x, float y, float z, float w)
            {
                this.Frame = frame;
                this.Value = new Vector4f(x, y, z, w);
            }

            /// <summary>
            /// フレーム
            /// </summary>
            public int Frame { get; private set; }

            /// <summary>
            /// 値
            /// </summary>
            public Vector4f Value { get; private set; }
        }
    }
}
