﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_7.Build_0.Revision_0
{
    /// <summary>
    /// Class for storing user shader parameters.
    /// </summary>
    [Serializable]
    public class UserShaderParameter : BaseDataModel<UserShaderParameter>
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserShaderParameter()
        {
            this.Index    = 0;
            this.Bytes    = new uint[4];
            this.BitFlags = new bool[32];
            this.Params   = new float[32];
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">Source to copy from.</param>
        public UserShaderParameter( UserShaderParameter src ) :
            this()
        {
            this.Set( src );
        }


        /// <summary>
        /// Initialize.
        /// </summary>
        public override void Initialize()
        {
            int i;
            for ( i=0;i<this.Bytes.Length;++i )
                this.Bytes[i] = 0;

            for ( i=0;i<this.BitFlags.Length;++i )
                this.BitFlags[i] = false;

            for ( i=0;i<this.Params.Length;++i )
                this.Params[i] = 0.0f;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set parameters from the given source.
        /// </summary>
        /// <param name="src">Source to set from.</param>
        public override void Set( UserShaderParameter src )
        {
            Ensure.Argument.NotNull( src );

            this.Index = src.Index;

            int i;
            for ( i=0;i<this.Bytes.Length;++i )
                this.Bytes[i] = src.Bytes[i];

            for ( i=0;i<this.BitFlags.Length;++i )
                this.BitFlags[i] = src.BitFlags[i];

            for ( i=0;i<this.Params.Length;++i )
                this.Params[i] = src.Params[i];
        }


        /// <summary>
        /// Make a clone of this object.
        /// </summary>
        /// <returns>The created clone object.</returns>
        public override UserShaderParameter Clone()
        {
            return new UserShaderParameter( this );
        }

        #endregion

        #region Properties

        /// <summary>
        /// The index of the user shader parameter.
        /// </summary>
        public int Index { get; set; }


        /// <summary>
        /// ユーザーシェーダスイッチ
        /// </summary>
        [System.Xml.Serialization.XmlElement("Bytes")]
        public uint ComposedBytes
        {
            get
            {
                uint iValue = 0x00000000;
                uint iMask  = 0x000000FF;
                foreach ( uint iByte in this.Bytes )
                {
                    iValue |= iByte & iMask;
                    iMask   = iMask << 8;
                }

                return iValue;
            }

            set
            {
                uint iMask  = 0x000000FF;
                for ( int i=0;i<this.Bytes.Length;++i )
                {
                    this.Bytes[i] = value & iMask;
                    iMask         = iMask << 8;
                }
            }
        }


        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        [System.Xml.Serialization.XmlElement("BitFlags")]
        public uint ComposedBitFlags
        {
            get
            {
                uint iValue = 0x00000000;
                uint iMask  = 0x00000001;
                foreach ( bool bFlag in this.BitFlags )
                {
                    if ( bFlag==true )
                        iValue |= iMask;

                    iMask = iMask << 1;
                }

                return iValue;
            }

            set
            {
                uint iMask = 0x00000001;
                for ( int i=0;i<this.BitFlags.Length;++i )
                {
                    this.BitFlags[i] = ((value & iMask)==0x00000000) ? false : true;
                    iMask            = iMask << 1;
                }
            }
        }


        /// <summary>
        /// ユーザーシェーダスイッチ
        /// </summary>
        [System.Xml.Serialization.XmlIgnore]
        public uint[] Bytes { get; set; }

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        [System.Xml.Serialization.XmlIgnore]
        public bool[] BitFlags { get; set; }

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        [System.Xml.Serialization.XmlArrayItem("Param")]
        public float[] Params { get; set; }

        #endregion
    }


    /// <summary>
    /// Class for storing default user shader parameters.
    /// </summary>
    public class DefaultUserShaderParameter
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public DefaultUserShaderParameter()
        {
            this.Bytes    = new uint[4];
            this.BitFlags = new bool[32];
            this.Params   = new float[32];

            int i;
            for ( i=0;i<this.Bytes.Length;++i )
                this.Bytes[i] = 0x00000000;

            for ( i=0;i<this.BitFlags.Length;++i )
                this.BitFlags[i] = false;

            for ( i=0;i<this.Params.Length;++i )
                this.Params[i] = 0.0f;
        }

        #endregion

        #region Properties

        /// <summary>
        /// ユーザーシェーダスイッチ
        /// </summary>
        public uint[] Bytes { get; set; }

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        public bool[] BitFlags { get; set; }

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public float[] Params { get; set; }

        #endregion
    }
}
